#ifndef SIMULATIONSTATE_H

#define SIMULATIONSTATE_H

#include <errut/errorbase.h>
#include <serut/serializationinterface.h>
#include <string>
#include <vector>
#include <stdio.h>
#include <stdint.h>

#define SIMSTATE_GRIDPROP_N		0
#define SIMSTATE_GRIDPROP_P		1
#define SIMSTATE_GRIDPROP_NI		2
#define SIMSTATE_GRIDPROP_BG		3
#define SIMSTATE_GRIDPROP_V		4
#define SIMSTATE_GRIDPROP_G		5
#define SIMSTATE_GRIDPROP_RF		6
#define SIMSTATE_GRIDPROP_DN		7
#define SIMSTATE_GRIDPROP_DP		8
#define SIMSTATE_GRIDPROP_NMOB		9
#define SIMSTATE_GRIDPROP_PMOB		10
#define SIMSTATE_GRIDPROP_EPSREL	11
#define SIMSTATE_GRIDPROP_VNEXTRA	12
#define SIMSTATE_GRIDPROP_VPEXTRA	13
#define SIMSTATE_GRIDPROP_JNX		14
#define SIMSTATE_GRIDPROP_JNY		15
#define SIMSTATE_GRIDPROP_JPX		16
#define SIMSTATE_GRIDPROP_JPY		17
#define SIMSTATE_GRIDPROP_R		18
#define SIMSTATE_GRIDPROP_DISSPROB	19

#define SIMSTATE_GRIDPROP_NMOB_GAMMA	20
#define SIMSTATE_GRIDPROP_PMOB_GAMMA	21

// reserve some space
#define SIMSTATE_GRIDPROP_MAX		64

#define SIMSTATE_PROP_TEMP		0
#define SIMSTATE_PROP_VDIFF		1
#define SIMSTATE_PROP_PAIRDIST		2
#define SIMSTATE_PROP_KF		3
// reserve some space
#define SIMSTATE_PROP_MAX		64

class SimulationState : public errut::ErrorBase
{
public:
	enum RecombinationModel { Simple, Braun };

	SimulationState();
	~SimulationState();

	bool init(int numX, double realPixelWidth); // 1D
	bool init(int numX, int numY, double realPixelWidth, double realPixelHeight); // 2D
	void clear();
	bool isInitialized() const									{ return m_init; }

	bool isModified() const;
	void setModified(bool f);

	int getDimensions() const;
	int getNumberOfXPixels() const;
	int getNumberOfYPixels() const;
	double getPixelWidth() const;
	double getPixelHeight() const;

	bool save(const std::string &fileName) const;
	bool load(const std::string &fileName);
	bool write(serut::SerializationInterface &s) const;
	bool read(serut::SerializationInterface &s);
	bool import(const SimulationState &state);
	bool import(const SimulationState &state, bool *pGridPropFlags, bool *pDoublePropFlags);
	bool import(const SimulationState &state, bool *pGridPropFlags, bool *pDoublePropFlags, 
		    const std::vector<bool> &excludeMap);

	bool setGridProperty(int propID, int x, double value);
	bool setGridProperty(int propID, int x, int y, double value);
	bool setGridProperty(int propID, const std::vector<double> &values);
	bool clearGridProperty(int propID);

	bool setDoubleProperty(int propID, double value);
	bool clearDoubleProperty(int propID);

	bool setRecombinationModel(RecombinationModel r);
	RecombinationModel getRecombinationModel() const						{ return m_recombinationModel; }

	double getGridProperty(int prop, int x) const;
	double getGridProperty(int prop, int x, int y) const;
	const double *getGridProperty(int prop) const;
	bool getGridProperty(int prop, std::vector<double> &values) const;

	double getDoubleProperty(int propID) const;
	bool getDoubleProperty(int propID, double &value) const;

	bool isGridPropertySet(int prop) const;
	bool isDoublePropertySet(int prop) const;

	static std::string getPlotFileColumns(int dimensions);
	bool writePlotData(FILE *pFile) const;

	bool calculateCurrent1D(double &left, double &center, double &right, double &overall) const;

	bool calculateCurrent2DX(double &left, double &center, double &right, double &overall) const;
	bool calculateCurrent2DY(double &bottom, double &center, double &top, double &overall) const;

	static std::string getGridPropertyName(int propID);
private:
	void initProperties();
	bool setDoublePropertyInternal(int prop, double value);
	bool clearDoublePropertyInternal(int prop);
	bool setGridPropertyInternal(int prop, int x, int y, double value);
	bool clearGridPropertyInternal(int prop);

	double getGridPropertyInternal(int prop, int x, int y) const;
	const double *getGridPropertyInternal(int prop) const;
	bool getGridPropertyInternal(int prop, std::vector<double> &values) const;

	double getDoublePropertyInternal(int propID) const;
	bool getDoublePropertyInternal(int propID, double &value) const;

	static double integrate1D(const std::vector<double> &src, double p1, double p2, int srcWidth, bool ln);
	static bool resampleArray1D(const std::vector<double> &src, std::vector<double> &dst, 
		                      int srcWidth, int dstWidth, bool ln);
	static bool resampleArray2D(const std::vector<double> &src, std::vector<double> &dst, 
		                      std::vector<double> &tmpDst, int srcWidth, int srcHeight, 
				      int dstWidth, int dstHeight, bool ln);
	static double integrate2DH(const std::vector<double> &src, double p1, double p2, int y, int srcWidth, 
				bool ln, bool &allPos);
	static double integrate2DV(const std::vector<double> &src, double p1, double p2, int x, int srcWidth, 
		                     int srcHeight, bool ln, bool &allPos);

	bool m_init;
	int m_dimensions;
	int m_numX, m_numY;
	int m_totalPixels;
	double m_pixelWidth, m_pixelHeight;
	RecombinationModel m_recombinationModel;

	std::vector<double> m_gridProperties[SIMSTATE_GRIDPROP_MAX];
	bool m_useLogInterpolation[SIMSTATE_GRIDPROP_MAX];

	std::pair<bool, double> m_doubleProperties[SIMSTATE_PROP_MAX];

	bool m_modified;

	static uint8_t m_identifier[8];
	static uint8_t m_identifierEnd[8];
};

#endif // SIMULATIONSTATE_H
