#include "simulationstate.h"
#include <serut/fileserializer.h>
#include <stdio.h>
#include <string.h>
#include <cmath>

using namespace serut;

#define CHECK_INIT \
if (!m_init) \
{\
	setErrorString("Simulation state has not been initialized yet"); \
	return false; \
}

#define CHECK_DIMENSIONS(n) \
CHECK_INIT; \
if (n != m_dimensions) \
{\
	char str[1024];\
	sprintf(str, "Operation for %d dimensions requested, but simulation state was initialized to have %d dimensions", (int)n, m_dimensions); \
	setErrorString(str); \
	return false; \
}

#define CHECK_GRIDPROPID(id) \
if (id < 0 || id >= SIMSTATE_GRIDPROP_MAX) \
{\
	setErrorString("Specified grid property ID is invalid"); \
	return false;\
}

#define CHECK_DOUBLEPROPID(id) \
if (id < 0 || id >= SIMSTATE_PROP_MAX) \
{\
	setErrorString("Specified property ID is invalid"); \
	return false;\
}


SimulationState::SimulationState()
{
	m_init = false;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		m_useLogInterpolation[i] = false;

	m_useLogInterpolation[SIMSTATE_GRIDPROP_N] = true;
	m_useLogInterpolation[SIMSTATE_GRIDPROP_P] = true;
	m_useLogInterpolation[SIMSTATE_GRIDPROP_NI] = true;
}

SimulationState::~SimulationState()
{
	clear();
}

bool SimulationState::init(int numX, double realPixelWidth)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (numX < 3)
	{
		setErrorString("Number of pixels must be at least 3");
		return false;
	}

	if (realPixelWidth < 0)
	{
		setErrorString("Physical dimensions must be positive");
		return false;
	}

	m_dimensions = 1;
	m_numX = numX;
	m_numY = 1;
	m_totalPixels = numX;
	m_pixelWidth = realPixelWidth;
	m_pixelHeight = 0;
	m_recombinationModel = Simple;

	initProperties();

	m_init = true;

	m_modified = false;

	return true;
}

bool SimulationState::init(int numX, int numY, double realPixelWidth, double realPixelHeight)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (numX < 3 || numY < 3)
	{
		setErrorString("Number of pixels in each direction must be at least 3");
		return false;
	}

	if (realPixelWidth < 0 || realPixelHeight < 0)
	{
		setErrorString("Physical dimensions must be positive");
		return false;
	}

	m_dimensions = 2;
	m_numX = numX;
	m_numY = numY;
	m_totalPixels = numX*numY;
	m_pixelWidth = realPixelWidth;
	m_pixelHeight = realPixelHeight;
	m_recombinationModel = Simple;

	initProperties();

	m_init = true;

	m_modified = false;

	return true;
}

void SimulationState::initProperties()
{
	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		m_gridProperties[i].resize(0);

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
	{
		m_doubleProperties[i].first = false;
		m_doubleProperties[i].second = 0;
	}
}

void SimulationState::clear()
{
	if (!m_init)
		return;

	initProperties();
	m_init = false;
}

bool SimulationState::isModified() const
{
	if (!m_init)
		return false;

	return m_modified;
}

void SimulationState::setModified(bool f)
{
	if (!m_init)
		return;
	
	m_modified = f;
}

int SimulationState::getDimensions() const
{
	if (!m_init)
		return 0;
	return m_dimensions;
}

int SimulationState::getNumberOfXPixels() const
{
	if (!m_init)
		return 0;
	return m_numX;
}

int SimulationState::getNumberOfYPixels() const
{
	if (!m_init)
		return 0;
	return m_numY;
}

double SimulationState::getPixelWidth() const
{
	if (!m_init)
		return 0;
	return m_pixelWidth;
}

double SimulationState::getPixelHeight() const
{
	if (!m_init)
		return 0;
	return m_pixelHeight;
}

bool SimulationState::save(const std::string &fileName) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninialized simulation state");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::WriteOnly))
	{
		setErrorString(std::string("Couldn't save to file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return write(fSer);
}

bool SimulationState::load(const std::string &fileName)
{
	if (m_init)
	{
		setErrorString("The simulation state has already been initialized, clear it before loading other data in it");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::ReadOnly))
	{
		setErrorString(std::string("Couldn't load file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return read(fSer);
}

uint8_t SimulationState::m_identifier[8] = { 0x53, 0x49, 0x4d, 0x49, 0x53, 0x54, 0x41, 0x54 };
uint8_t SimulationState::m_identifierEnd[8] = { 0x54, 0x41, 0x54, 0x53, 0x49, 0x4d, 0x49, 0x53 };

bool SimulationState::write(serut::SerializationInterface &si) const
{
	CHECK_INIT;

	if (!si.writeBytes(m_identifier, 8))
	{
		setErrorString("Couldn't write identifier");
		return false;
	}

	int32_t intSettings[] = { m_dimensions, m_numX, m_numY, m_recombinationModel };
	double doubleSettings[] = { m_pixelWidth, m_pixelHeight };

	if (!si.writeInt32s(intSettings, 4) || !si.writeDoubles(doubleSettings, 2))
	{
		setErrorString("Error writing settings: " + si.getErrorString());
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		int32_t flag = 1;

		if (m_gridProperties[i].empty())
			flag = 0;

		if (!si.writeInt32(flag))
		{
			setErrorString("Error writing flag indicating grid property presence: " + si.getErrorString());
			return false;
		}

		if (!m_gridProperties[i].empty())
		{
			if (!si.writeDoubles(m_gridProperties[i]))
			{
				setErrorString("Error writing a grid property to file: " + si.getErrorString());
				return false;
			}
		}
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
	{
		int32_t flag = 1;

		if (!m_doubleProperties[i].first)
			flag = 0;

		if (!si.writeInt32(flag))
		{
			setErrorString("Error writing flag indicating double property presence: " + si.getErrorString());
			return false;
		}

		if (m_doubleProperties[i].first)
		{
			if (!si.writeDouble(m_doubleProperties[i].second))
			{
				setErrorString("Error writing a double property to file: " + si.getErrorString());
				return false;
			}
		}
	}
	
	if (!si.writeBytes(m_identifierEnd, 8))
	{
		setErrorString("Couldn't write end identifier");
		return false;
	}

	return true;
}

bool SimulationState::read(serut::SerializationInterface &si)
{
	if (m_init)
	{
		setErrorString("The simulation state has already been initialized, clear it before loading other data in it");
		return false;
	}

	uint8_t id[8];

	if (!si.readBytes(id, 8))
	{
		setErrorString("Unable to read simulation state identifier: " + si.getErrorString());
		return false;
	}

	if (memcmp(id, m_identifier, 8) != 0)
	{
		setErrorString("Read an invalid simulation state identifier");
		return false;
	}

	int32_t intSettings[4];
	double doubleSettings[2];

	if (!si.readInt32s(intSettings, 4) || !si.readDoubles(doubleSettings, 2))
	{
		setErrorString("Error reading settings: " + si.getErrorString());
		return false;
	}
	
	m_dimensions = intSettings[0];
	m_numX = intSettings[1];
	m_numY = intSettings[2];
	m_pixelWidth = doubleSettings[0];
	m_pixelHeight = doubleSettings[1];
	m_recombinationModel = (RecombinationModel)intSettings[3];

	if (m_dimensions < 1 || m_dimensions > 2)
	{
		setErrorString("Read an invalid number of dimensions");
		return false;
	}

	if (m_dimensions == 1)
	{
		if (m_numX <= 3)
		{
			setErrorString("Read an invalid number of pixels: number of pixels must be at least 3");
			return false;
		}
		if (m_numY != 1)
		{
			setErrorString("Read an invalid number of y pixels: must be 1 when the state is one dimensional");
			return false;
		}
		if (m_pixelWidth < 0 || m_pixelHeight != 0)
		{
			setErrorString("Read an invalid pixel width or pixel height");
			return false;
		}
	}
	else
	{
		if (m_numX <= 3 || m_numY <= 3)
		{
			setErrorString("Read an invalid number of pixels: number of pixels must be at least 3 in each dimension");
			return false;
		}
		if (m_pixelWidth < 0 || m_pixelHeight < 0)
		{
			setErrorString("Read an invalid pixel width or pixel height");
			return false;
		}
	}

	if (!(m_recombinationModel == Simple || m_recombinationModel == Braun))
	{
		setErrorString("Read an invalid recombination model");
		return false;
	}

	m_totalPixels = m_numX*m_numY;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		int32_t flag = 0;

		if (!si.readInt32(&flag))
		{
			setErrorString("Couldn't read flag indicating grid property presence: " + si.getErrorString());
			return false;
		}
		if (!flag)
			m_gridProperties[i].resize(0);
		else
		{
			m_gridProperties[i].resize(m_totalPixels);
			if (!si.readDoubles(m_gridProperties[i]))
			{
				setErrorString("Error reading grid property values: " + si.getErrorString());
				return false;
			}
		}
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
	{
		int32_t flag = 0;

		if (!si.readInt32(&flag))
		{
			setErrorString("Couldn't read flag indicating double property presence: " + si.getErrorString());
			return false;
		}

		if (flag == 0)
		{
			m_doubleProperties[i].first = false;
			m_doubleProperties[i].second = 0;
		}
		else
		{
			m_doubleProperties[i].first = true;
			
			double v = 0;

			if (!si.readDouble(&v))
			{
				setErrorString("Error reading double property value: " + si.getErrorString());
				return false;
			}

			m_doubleProperties[i].second = v;
		}
	}

	if (!si.readBytes(id, 8))
	{
		setErrorString("Unable to read simulation state end identifier: " + si.getErrorString());
		return false;
	}

	if (memcmp(id, m_identifierEnd, 8) != 0)
	{
		setErrorString("Read an invalid simulation state end identifier");
		return false;
	}

	m_init = true;
	m_modified = false;

	return true;
}

bool SimulationState::setRecombinationModel(RecombinationModel r)
{
	CHECK_INIT;

	if (r != m_recombinationModel)
	{
		m_recombinationModel = r;
		m_modified = true;
	}

	return true;
}

bool SimulationState::setGridProperty(int propID, int x, double value)
{
	CHECK_DIMENSIONS(1);
	CHECK_GRIDPROPID(propID);
	return setGridPropertyInternal(propID, x, 0, value);;
}

bool SimulationState::setGridProperty(int propID, int x, int y, double value)
{
	CHECK_DIMENSIONS(2);
	CHECK_GRIDPROPID(propID);
	return setGridPropertyInternal(propID, x, y, value);
}

bool SimulationState::clearGridProperty(int propID)
{
	CHECK_INIT;
	CHECK_GRIDPROPID(propID);
	return clearGridPropertyInternal(propID);
}

bool SimulationState::setGridPropertyInternal(int prop, int x, int y, double value)
{
	if (x < 0 || x >= m_numX || y < 0 || y >= m_numY)
	{
		setErrorString("Coordinate lies outside the grid");
		return false;
	}

	int s = m_gridProperties[prop].size();
	double *pArray = 0;

	if (s == 0)
	{
		m_gridProperties[prop].resize(m_totalPixels);

		pArray = &(m_gridProperties[prop][0]);
		for (int i = 0 ; i < m_totalPixels ; i++)
			pArray[i] = 0; // TODO: make initialization value configurable?
	}
	else
	{
		if (s != m_totalPixels)
		{
			setErrorString("Internal error: existing grid size is not equal to the size specified during initialization");
			return false;
		}

		pArray = &(m_gridProperties[prop][0]);
	}

	pArray[x+y*m_numX] = value;

	m_modified = true;

	return true;
}

bool SimulationState::clearGridPropertyInternal(int prop)
{
	m_gridProperties[prop].resize(0);
	m_modified = true;
	return true;
}

bool SimulationState::setDoubleProperty(int propID, double value)
{
	CHECK_INIT;
	CHECK_DOUBLEPROPID(propID);
	return setDoublePropertyInternal(propID, value);
}

bool SimulationState::clearDoubleProperty(int propID)
{
	CHECK_INIT;
	CHECK_DOUBLEPROPID(propID);
	return clearDoublePropertyInternal(propID);
}

bool SimulationState::setDoublePropertyInternal(int prop, double value)
{
	m_doubleProperties[prop].first = true;
	m_doubleProperties[prop].second = value;
	m_modified = true;
	return true;
}

bool SimulationState::clearDoublePropertyInternal(int prop)
{
	m_doubleProperties[prop].first = false;
	m_doubleProperties[prop].second = 0;
	m_modified = true;
	return true;
}

double SimulationState::getGridProperty(int prop, int x) const
{
	if (!m_init)
		return -12345.0;
	if (m_dimensions != 1)
		return -12345.1;
	if (prop < 0 || prop >= SIMSTATE_GRIDPROP_MAX)
		return -12345.2;
	return getGridPropertyInternal(prop, x, 0);
}

double SimulationState::getGridProperty(int prop, int x, int y) const
{
	if (!m_init)
		return -12346.0;
	if (m_dimensions != 2)
		return -12346.1;
	if (prop < 0 || prop >= SIMSTATE_GRIDPROP_MAX)
		return -12346.2;
	return getGridPropertyInternal(prop, x, y);
}

const double *SimulationState::getGridProperty(int prop) const
{
	if (!m_init)
	{
		setErrorString("Simulation state has not been initialized yet");
		return 0;
	}
	if (prop < 0 || prop >= SIMSTATE_GRIDPROP_MAX)
	{
		setErrorString("Specified grid property ID is invalid");
		return 0;
	}
	return getGridPropertyInternal(prop);
}

bool SimulationState::getGridProperty(int prop, std::vector<double> &values) const
{
	CHECK_INIT;
	CHECK_GRIDPROPID(prop);
	return getGridPropertyInternal(prop, values);
}

bool SimulationState::setGridProperty(int propID, const std::vector<double> &values)
{
	CHECK_INIT;
	CHECK_GRIDPROPID(propID);

	if (values.size() != m_totalPixels)
	{
		setErrorString("Number of specified values does not correspond to the grid size");
		return false;
	}

	if (m_gridProperties[propID].size() != m_totalPixels)
		m_gridProperties[propID].resize(m_totalPixels);

	memcpy(&(m_gridProperties[propID][0]), &(values[0]), sizeof(double)*m_totalPixels);

	return true;
}


double SimulationState::getGridPropertyInternal(int prop, int x, int y) const
{
	if (m_gridProperties[prop].empty())
		return -12347.0;
	if (x < 0 || x >= m_numX || y < 0 || y >= m_numY)
		return -12347.1;
	return m_gridProperties[prop][x+y*m_numX];
}

const double *SimulationState::getGridPropertyInternal(int prop) const
{
	if (m_gridProperties[prop].empty())
		return 0;
	return &(m_gridProperties[prop][0]);
}

bool SimulationState::getGridPropertyInternal(int prop, std::vector<double> &values) const
{
	if (m_gridProperties[prop].empty())
	{
		setErrorString("Specified grid property has not been set yet");
		return false;
	}
	
	if (values.size() != m_totalPixels)
		values.resize(m_totalPixels);

	memcpy(&(values[0]), &(m_gridProperties[prop][0]), sizeof(double)*m_totalPixels);
	return true;
}

bool SimulationState::import(const SimulationState &state)
{
	bool gridFlags[SIMSTATE_GRIDPROP_MAX];
	bool doubleFlags[SIMSTATE_PROP_MAX];

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		gridFlags[i] = true;
	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		doubleFlags[i] = true;

	return import(state, gridFlags, doubleFlags);
}

bool SimulationState::import(const SimulationState &state, bool *pGridPropFlags, bool *pDoublePropFlags)
{
	std::vector<bool> emptyMap;

	return import(state, pGridPropFlags, pDoublePropFlags, emptyMap);
}

bool SimulationState::import(const SimulationState &state, bool *pGridPropFlags, bool *pDoublePropFlags, 
		    const std::vector<bool> &excludeMap)
{
	CHECK_INIT;
	if (!state.m_init)
	{
		setErrorString("Simulation state to be imported has not been initialized yet");
		return false;
	}

	if (m_dimensions != state.m_dimensions)
	{
		setErrorString("This simulation state and the one to be imported do not have the same number of dimensions");
		return false;
	}

	if (!excludeMap.empty())
	{
		if (excludeMap.size() != m_totalPixels)
		{
			setErrorString("The exclude areas map does not have the same size as the grid");
			return false;
		}
	}

	int numDstTmp = 0;

	if (m_dimensions == 2)
		numDstTmp = state.m_numX * m_numY;

	std::vector<double> dummy3(numDstTmp);
	std::vector<double> excludeCopy;

	if (!excludeMap.empty())
		excludeCopy.resize(m_totalPixels);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (pGridPropFlags[i])
		{
			if (state.m_gridProperties[i].empty())
				m_gridProperties[i].resize(0);
			else
			{
				if (m_gridProperties[i].empty())
				{
					m_gridProperties[i].resize(m_totalPixels);
			
					if (!excludeMap.empty())
					{
						for (int i = 0 ; i < m_totalPixels ; i++)
							excludeCopy[i] = 0;
					}
				}
				else
				{
					if (!excludeMap.empty())
					{
						double *pArray = &(m_gridProperties[i][0]);
						double *pDst = &(excludeCopy[0]);

						for (int i = 0 ; i < m_totalPixels ; i++)
							pDst[i] = pArray[i];
					}
				}

				if (m_dimensions == 1)
					resampleArray1D(state.m_gridProperties[i], m_gridProperties[i], state.m_numX, m_numX, m_useLogInterpolation[i]);
				else if (m_dimensions == 2)
					resampleArray2D(state.m_gridProperties[i], m_gridProperties[i], dummy3, state.m_numX, state.m_numY, m_numX, m_numY, m_useLogInterpolation[i]);
				else
				{
					setErrorString("SimulationState::import: invalid number of dimensions");
					return false;
				}

				if (!excludeMap.empty())
				{
					double *pArray = &(m_gridProperties[i][0]);
					double *pSrc = &(excludeCopy[0]);

					for (int y = 0 ; y < m_numY ; y++)
					{
						for (int x = 0 ; x < m_numX ; x++)
						{
							int idx = x+y*m_numX;

							if (excludeMap[idx])
								pArray[idx] = pSrc[idx];
						}
					}
				}
			}
		}
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
	{
		if (pDoublePropFlags[i])
		{
			m_doubleProperties[i].first = state.m_doubleProperties[i].first;
			m_doubleProperties[i].second = state.m_doubleProperties[i].second;
		}
	}

	m_modified = true;

	return true;
}

bool SimulationState::isGridPropertySet(int prop) const
{
	if (!m_init)
		return false;
	if (prop < 0 || prop >= SIMSTATE_GRIDPROP_MAX)
		return false;
	if (m_gridProperties[prop].size() == 0)
		return false;
	return true;
}

bool SimulationState::isDoublePropertySet(int prop) const
{
	if (!m_init)
		return false;
	if (prop < 0 || prop >= SIMSTATE_PROP_MAX)
		return false;
	return m_doubleProperties[prop].first;
}

double SimulationState::getDoubleProperty(int propID) const
{
	if (!m_init)
		return -12349.0;
	if (propID < 0 || propID >= SIMSTATE_PROP_MAX)
		return -12349.1;
	return getDoublePropertyInternal(propID);
}

bool SimulationState::getDoubleProperty(int propID, double &value) const
{
	CHECK_INIT;
	CHECK_DOUBLEPROPID(propID);
	return getDoublePropertyInternal(propID, value);
}

double SimulationState::getDoublePropertyInternal(int propID) const
{
	if (!m_doubleProperties[propID].first)
		return -12350.0;
	return m_doubleProperties[propID].second;
}

bool SimulationState::getDoublePropertyInternal(int propID, double &value) const
{
	if (!m_doubleProperties[propID].first)
	{
		setErrorString("Specified property hasn't been set yet");
		return false;
	}
	value = m_doubleProperties[propID].second;
	return true;
}

std::string SimulationState::getPlotFileColumns(int dimensions)
{
	if (dimensions == 1)
		return std::string("xpixel, n, p, V, jn, jp, j, R, dissprob, ni, bg, G, rf, dn, dp, nmob, pmob, epsrel, Vn, Vp");
	else if (dimensions == 2)
		return std::string("xpixel, n, p, V, jnx, jpx, jx, jny, jpy, jy, R, dissprob, ni, bg, G, rf, dn, dp, nmob, pmob, epsrel, Vn, Vp");
	
	return std::string("ERROR: invalid number of dimensions");
}

#define DERIVEDPROP_JX -1
#define DERIVEDPROP_JY -2

bool SimulationState::writePlotData(FILE *pFile) const
{
	CHECK_INIT;
	if (m_dimensions == 1)
	{
		std::vector<int> allProps;

		allProps.push_back(SIMSTATE_GRIDPROP_N);
		allProps.push_back(SIMSTATE_GRIDPROP_P);
		allProps.push_back(SIMSTATE_GRIDPROP_V);
		allProps.push_back(SIMSTATE_GRIDPROP_JNX);
		allProps.push_back(SIMSTATE_GRIDPROP_JPX);
		allProps.push_back(DERIVEDPROP_JX);
		allProps.push_back(SIMSTATE_GRIDPROP_R);
		allProps.push_back(SIMSTATE_GRIDPROP_DISSPROB);
		allProps.push_back(SIMSTATE_GRIDPROP_NI);
		allProps.push_back(SIMSTATE_GRIDPROP_BG);
		allProps.push_back(SIMSTATE_GRIDPROP_G);
		allProps.push_back(SIMSTATE_GRIDPROP_RF);
		allProps.push_back(SIMSTATE_GRIDPROP_DN);
		allProps.push_back(SIMSTATE_GRIDPROP_DP);
		allProps.push_back(SIMSTATE_GRIDPROP_NMOB);
		allProps.push_back(SIMSTATE_GRIDPROP_PMOB);
		allProps.push_back(SIMSTATE_GRIDPROP_EPSREL);
		allProps.push_back(SIMSTATE_GRIDPROP_VNEXTRA);
		allProps.push_back(SIMSTATE_GRIDPROP_VPEXTRA);

		for (int i = 0 ; i < allProps.size() ; i++)
		{
			if (allProps[i] >= 0)
			{
				if (m_gridProperties[allProps[i]].empty())
					allProps[i] = SIMSTATE_GRIDPROP_MAX;
			}
			else // can only be total current
			{
				if (m_gridProperties[SIMSTATE_GRIDPROP_JNX].empty() ||
				    m_gridProperties[SIMSTATE_GRIDPROP_JPX].empty() )
					allProps[i] = SIMSTATE_GRIDPROP_MAX;
			}
		}

		for (int x = 0 ; x < m_numX ; x++)
		{
			fprintf(pFile, "%d", x);

			for (int i = 0 ; i < allProps.size() ; i++)
			{
				if (allProps[i] == SIMSTATE_GRIDPROP_MAX)
					fprintf(pFile, "\t?");
				else
				{
					double v = 0;
					if (allProps[i] >= 0)
						v = m_gridProperties[allProps[i]][x];
					else // can only be total current
						v = m_gridProperties[SIMSTATE_GRIDPROP_JPX][x] - m_gridProperties[SIMSTATE_GRIDPROP_JNX][x];

					fprintf(pFile, "\t%.10g", v);
				}
			}
			fprintf(pFile, "\n");
		}
	}
	else if (m_dimensions == 2)
	{
		std::vector<int> allProps;

		allProps.push_back(SIMSTATE_GRIDPROP_N);
		allProps.push_back(SIMSTATE_GRIDPROP_P);
		allProps.push_back(SIMSTATE_GRIDPROP_V);
		allProps.push_back(SIMSTATE_GRIDPROP_JNX);
		allProps.push_back(SIMSTATE_GRIDPROP_JPX);
		allProps.push_back(DERIVEDPROP_JX);
		allProps.push_back(SIMSTATE_GRIDPROP_JNY);
		allProps.push_back(SIMSTATE_GRIDPROP_JPY);
		allProps.push_back(DERIVEDPROP_JY);
		allProps.push_back(SIMSTATE_GRIDPROP_R);
		allProps.push_back(SIMSTATE_GRIDPROP_DISSPROB);
		allProps.push_back(SIMSTATE_GRIDPROP_NI);
		allProps.push_back(SIMSTATE_GRIDPROP_BG);
		allProps.push_back(SIMSTATE_GRIDPROP_G);
		allProps.push_back(SIMSTATE_GRIDPROP_RF);
		allProps.push_back(SIMSTATE_GRIDPROP_DN);
		allProps.push_back(SIMSTATE_GRIDPROP_DP);
		allProps.push_back(SIMSTATE_GRIDPROP_NMOB);
		allProps.push_back(SIMSTATE_GRIDPROP_PMOB);
		allProps.push_back(SIMSTATE_GRIDPROP_EPSREL);
		allProps.push_back(SIMSTATE_GRIDPROP_VNEXTRA);
		allProps.push_back(SIMSTATE_GRIDPROP_VPEXTRA);

		for (int i = 0 ; i < allProps.size() ; i++)
		{
			if (allProps[i] >= 0)
			{
				if (m_gridProperties[allProps[i]].empty())
					allProps[i] = SIMSTATE_GRIDPROP_MAX;
			}
			else // can only be total current
			{
				if (m_gridProperties[SIMSTATE_GRIDPROP_JNX].empty() ||
				    m_gridProperties[SIMSTATE_GRIDPROP_JPX].empty() )
					allProps[i] = SIMSTATE_GRIDPROP_MAX;
			}
		}

		for (int y = 0 ; y < m_numY ; y++)
		{
			for (int x = 0 ; x < m_numX ; x++)
			{
				int idx = x+y*m_numX;

				fprintf(pFile, "%d\t%d", x, y);

				for (int i = 0 ; i < allProps.size() ; i++)
				{
					if (allProps[i] == SIMSTATE_GRIDPROP_MAX)
						fprintf(pFile, "\t?");
					else
					{
						double v = 0;

						if (allProps[i] >= 0)
							v = m_gridProperties[allProps[i]][idx];
						else if (allProps[i] == DERIVEDPROP_JX)
							v = m_gridProperties[SIMSTATE_GRIDPROP_JPX][idx] - m_gridProperties[SIMSTATE_GRIDPROP_JNX][idx];
						else // can only be y current
							v = m_gridProperties[SIMSTATE_GRIDPROP_JPY][idx] - m_gridProperties[SIMSTATE_GRIDPROP_JNY][idx];

						fprintf(pFile, "\t%.10g", v);
					}
				}
				fprintf(pFile, "\n");
			}
			fprintf(pFile, "\n");
		}
	}
	else
	{
		setErrorString("ERROR: SimulationState::writePlotData: unexpected number of dimensions");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
	{
		fprintf(pFile, "# double property %d = ", i);
		if (m_doubleProperties[i].first)
			fprintf(pFile, "%g\n", m_doubleProperties[i].second);
		else
			fprintf(pFile, "not set\n");
	}

	fprintf(pFile, "# m_dimensions = %d\n", m_dimensions);
	fprintf(pFile, "# m_numX = %d\n", m_numX);
	fprintf(pFile, "# m_numY = %d\n", m_numY);
	fprintf(pFile, "# m_totalPixels = %d\n", m_totalPixels);
	fprintf(pFile, "# m_pixelWidth = %g\n", m_pixelWidth);
	fprintf(pFile, "# m_pixelHeight = %g\n", m_pixelHeight);
	fprintf(pFile, "# m_recombinationModel = %d\n", (int)m_recombinationModel);

	return true;
}

// TODO: this is for non-periodic boundary conditions
bool SimulationState::calculateCurrent1D(double &left, double &center, double &right, double &overall) const
{
	CHECK_DIMENSIONS(1);
	if (m_gridProperties[SIMSTATE_GRIDPROP_JNX].empty() || m_gridProperties[SIMSTATE_GRIDPROP_JPX].empty())
	{
		setErrorString("Either hole or electron current has not been set yet");
		return false;
	}

	double avg = 0;
	const double *pECur = &(m_gridProperties[SIMSTATE_GRIDPROP_JNX][0]);
	const double *pHCur = &(m_gridProperties[SIMSTATE_GRIDPROP_JPX][0]);

	for (int x = 0 ; x < m_numX - 1 ; x++)
		avg += pHCur[x]-pECur[x];

	avg /= (double)(m_numX-1);
	overall = avg;
	left = pHCur[0]-pECur[0];
	center = pHCur[m_numX/2]-pECur[m_numX/2];
	right = pHCur[m_numX-2]-pECur[m_numX-2];

	return true;
}

// TODO: this is for non-periodic boundary conditions in Y and periodic in X
bool SimulationState::calculateCurrent2DX(double &left, double &center, double &right, double &overall) const
{
	CHECK_DIMENSIONS(2);
	if (m_gridProperties[SIMSTATE_GRIDPROP_JNX].empty() || m_gridProperties[SIMSTATE_GRIDPROP_JPX].empty())
	{
		setErrorString("Either hole or electron current has not been set yet");
		return false;
	}

	const double *pECur = &(m_gridProperties[SIMSTATE_GRIDPROP_JNX][0]);
	const double *pHCur = &(m_gridProperties[SIMSTATE_GRIDPROP_JPX][0]);

	double avgL = 0;
	double avgC = 0;
	double avgR = 0;
	double avg = 0;

	int offCenter = m_numX/2;
	int offRight = m_numX-2;
	int yOff = 0;

	for (int y = 0 ; y < m_numY-1 ; y++, yOff += m_numX)
	{
		avgL += pHCur[yOff]-pECur[yOff];
		avgC += pHCur[yOff+offCenter]-pECur[yOff+offCenter];
		avgR += pHCur[yOff+offRight]-pECur[yOff+offRight];

		for (int x = 0 ; x < m_numX ; x++)
			avg += pHCur[x+y*m_numX]-pECur[x+y*m_numX];
	}

	avgL /= (double)(m_numY-1);
	avgC /= (double)(m_numY-1);
	avgR /= (double)(m_numY-1);
	avg /= (double)(m_numX*(m_numY-1));

	left = avgL;
	center = avgC;
	right = avgR;
	overall = avg;

	return true;
}

// TODO: this is for non-periodic boundary conditions in Y and periodic in X
bool SimulationState::calculateCurrent2DY(double &bottom, double &center, double &top, double &overall) const
{
	CHECK_DIMENSIONS(2);
	if (m_gridProperties[SIMSTATE_GRIDPROP_JNY].empty() || m_gridProperties[SIMSTATE_GRIDPROP_JPY].empty())
	{
		setErrorString("Either hole or electron current has not been set yet");
		return false;
	}

	const double *pECur = &(m_gridProperties[SIMSTATE_GRIDPROP_JNY][0]);
	const double *pHCur = &(m_gridProperties[SIMSTATE_GRIDPROP_JPY][0]);

	double avgB = 0;
	double avgC = 0;
	double avgT = 0;
	double avg = 0;

	int offCenter = (m_numY/2)*m_numX;
	int offTop = (m_numY-2)*m_numX;

	for (int x = 0 ; x < m_numX ; x++)
	{
		avgB += pHCur[x]-pECur[x];
		avgC += pHCur[x+offCenter]-pECur[x+offCenter];
		avgT += pHCur[x+offTop]-pECur[x+offTop];

		for (int y = 0 ; y < m_numY-1 ; y++)
			avg += pHCur[x+y*m_numX]-pECur[x+y*m_numX];
	}

	avgB /= (double)m_numX;
	avgC /= (double)m_numX;
	avgT /= (double)m_numX;
	avg /= (double)(m_numX*(m_numY-1));

	bottom = avgB;
	center = avgC;
	top = avgT;
	overall = avg;

	return true;
}

double SimulationState::integrate1D(const std::vector<double> &src, double p1, double p2, int srcWidth, bool ln)
{
	double sum = 0;
	int lowerPixel = (int)p1;
	int upperPixel = (int)(p2+1.0);

	if (!ln)
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (src[i]+src[i+1])/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = src[lowerPixel];
			double v1 = src[lowerPixel+1];
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = src[upperPixel-1];
			double v1 = src[upperPixel];
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	else
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (std::log(src[i])+std::log(src[i+1]))/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = std::log(src[lowerPixel]);
			double v1 = std::log(src[lowerPixel+1]);
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = std::log(src[upperPixel-1]);
			double v1 = std::log(src[upperPixel]);
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	
	sum /= (double)(srcWidth-1);

	return sum;
}

bool SimulationState::resampleArray1D(const std::vector<double> &src, std::vector<double> &dst, 
		                      int srcWidth, int dstWidth, bool ln)

{
	if (dstWidth > srcWidth)
	{
		for (int x = 0 ; x < dstWidth ; x++)
		{
			double xFrac = (double)x/(double)(dstWidth-1);
			double xNew = xFrac*(double)(srcWidth-1);
			int x0 = (int)xNew;
			int x1 = x0+1;

			if (x1 >= srcWidth)
				x1 = srcWidth-1;

			xFrac = xNew-(double)x0;

			double v0 = src[x0];
			double v1 = src[x1];

			if (ln)
			{
				v0 = std::log(v0);
				v1 = std::log(v1);
			}

			double v = v0*(1.0-xFrac)
				 + v1*xFrac;

			if (ln)
				v = std::exp(v);

			dst[x] = v;
		}
	}
	else if (dstWidth < srcWidth)
	{
		dst[0] = src[0];
		dst[dstWidth-1] = src[srcWidth-1];

		for (int dstPos = 1 ; dstPos < dstWidth-1 ; dstPos++)
		{
			double dstFrac1 = ((double)dstPos-0.5)/(double)(dstWidth-1);
			double dstFrac2 = ((double)dstPos+0.5)/(double)(dstWidth-1);
			double srcPos1 = dstFrac1*(double)(srcWidth-1);
			double srcPos2 = dstFrac2*(double)(srcWidth-1);

			double sum = integrate1D(src, srcPos1, srcPos2, srcWidth, ln);

			sum *= (double)(dstWidth-1);

			if (ln)
				sum = std::exp(sum);
			
			dst[dstPos] = sum;
		}
	}
	else
		dst = src;

	return true;
}

bool SimulationState::resampleArray2D(const std::vector<double> &src, std::vector<double> &dst, 
		                      std::vector<double> &tmpDst, int srcWidth, int srcHeight, 
				      int dstWidth, int dstHeight, bool ln)
{
	// First, rescale the y direction
	
	if (dstHeight > srcHeight)
	{
		for (int x = 0 ; x < srcWidth ; x++)
		{
			for (int y = 0 ; y < dstHeight ; y++)
			{
				double yFrac = (double)y/(double)(dstHeight-1);
				double yNew = yFrac*(double)(srcHeight-1);
				int y0 = (int)yNew;
				int y1 = y0+1;

				if (y1 >= srcHeight)
					y1 = srcHeight-1;

				yFrac = yNew-(double)y0;

				double v0 = src[x+y0*srcWidth];
				double v1 = src[x+y1*srcWidth];

				if (ln && v0 > 0 && v1 > 0)
				{
					v0 = std::log(v0);
					v1 = std::log(v1);
				}

				double v = v0*(1.0-yFrac)+v1*yFrac;

				if (ln && v0 > 0 && v1 > 0)
					v = std::exp(v);

				tmpDst[x+y*srcWidth] = v;
			}
		}
	}
	else if (dstHeight < srcHeight)
	{
		for (int x = 0 ; x < srcWidth ; x++)
		{
			tmpDst[x] = src[x];
			tmpDst[x+(dstHeight-1)*srcWidth] = src[x+(srcHeight-1)*srcWidth];

			for (int y = 1 ; y < dstHeight-1 ; y++)
			{
				double dstFrac1 = ((double)y-0.5)/(double)(dstHeight-1);
				double dstFrac2 = ((double)y+0.5)/(double)(dstHeight-1);
				double srcPos1 = dstFrac1*(double)(srcHeight-1);
				double srcPos2 = dstFrac2*(double)(srcHeight-1);
				bool allPos = true;
				
				double sum = integrate2DV(src, srcPos1, srcPos2, x, srcWidth, srcHeight, ln, allPos);

				sum *= (double)(dstHeight-1);

				if (ln && allPos)
					sum = std::exp(sum);
				
				tmpDst[x+y*srcWidth] = sum;

			}
		}
	}
	else
		tmpDst = src;


	// Then, rescale the x direction
	
	if (dstWidth > srcWidth)
	{
		for (int y = 0 ; y < dstHeight ; y++)
		{
			for (int x = 0 ; x < dstWidth ; x++)
			{
				double xFrac = (double)x/(double)(dstWidth);
				double xNew = xFrac*(double)(srcWidth);
				int x0 = (int)xNew;
				int x1 = (x0+1)%srcWidth;

				xFrac = xNew-(double)x0;

				double v0 = tmpDst[x0+y*srcWidth];
				double v1 = tmpDst[x1+y*srcWidth];

				if (ln && v0 > 0 && v1 > 0)
				{
					v0 = std::log(v0);
					v1 = std::log(v1);
				}

				double v = v0*(1.0-xFrac)+v1*xFrac;

				if (ln && v0 > 0 && v1 > 0)
					v = std::exp(v);

				dst[x+y*dstWidth] = v;
			}
		}
	}
	else if (dstWidth < srcWidth)
	{
		for (int y = 0 ; y < dstHeight ; y++)
		{
			for (int x = 0 ; x < dstWidth ; x++)
			{
				double dstFrac1 = ((double)x-0.5)/(double)(dstWidth);
				double dstFrac2 = ((double)x+0.5)/(double)(dstWidth);
				double srcPos1 = dstFrac1*(double)(srcWidth);
				double srcPos2 = dstFrac2*(double)(srcWidth);
				bool allPos = true;
				
				double sum = integrate2DH(tmpDst, srcPos1, srcPos2, y, srcWidth, ln, allPos);

				sum *= (double)(dstWidth);

				if (ln && allPos)
					sum = std::exp(sum);
				
				dst[x+y*dstWidth] = sum;

			}
		}
	}
	else
		dst = tmpDst;

	return true;
}

// TODO: this uses periodic boundary conditions in x-direction
double SimulationState::integrate2DH(const std::vector<double> &src, double p1, double p2, int y, int srcWidth, 
				bool ln, bool &allPos)
{
	double sum = 0;
	int lowerPixel = (int)p1;
	int upperPixel = (int)(p2+1.0);
	int offset = y*srcWidth;

	allPos = true;

#define CYC(x) (((x)+srcWidth)%srcWidth)

	for (int i = lowerPixel ; allPos && i <= upperPixel ; i++)
	{
		if (src[CYC(i)+offset] <= 0)
			allPos = false;
	}
	
	if (!(ln && allPos))
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (src[CYC(i)+offset]+src[CYC(i+1)+offset])/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = src[CYC(lowerPixel)+offset];
			double v1 = src[CYC(lowerPixel+1)+offset];
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = src[CYC(upperPixel-1)+offset];
			double v1 = src[CYC(upperPixel)+offset];
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	else
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (std::log(src[CYC(i)+offset])+std::log(src[CYC(i+1)+offset]))/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = std::log(src[CYC(lowerPixel)+offset]);
			double v1 = std::log(src[CYC(lowerPixel+1)+offset]);
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = std::log(src[CYC(upperPixel-1)+offset]);
			double v1 = std::log(src[CYC(upperPixel)+offset]);
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	
	sum /= (double)(srcWidth);

	return sum;
}

double SimulationState::integrate2DV(const std::vector<double> &src, double p1, double p2, int x, int srcWidth, 
		                     int srcHeight, bool ln, bool &allPos)
{
	double sum = 0;
	int lowerPixel = (int)p1;
	int upperPixel = (int)(p2+1.0);

	allPos = true;

	for (int i = lowerPixel ; allPos && i <= upperPixel ; i++)
	{
		if (src[i*srcWidth+x] <= 0)
			allPos = false;
	}

	if (!(ln && allPos))
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (src[i*srcWidth+x]+src[(i+1)*srcWidth+x])/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = src[lowerPixel*srcWidth+x];
			double v1 = src[(lowerPixel+1)*srcWidth+x];
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = src[(upperPixel-1)*srcWidth+x];
			double v1 = src[upperPixel*srcWidth+x];
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	else
	{
		for (int i = lowerPixel ; i < upperPixel ; i++)
		{
			sum += (std::log(src[i*srcWidth+x])+std::log(src[(i+1)*srcWidth+x]))/2.0;
		}

		// subtract lower and upper parts

		{
			double lowerFrac = p1-(double)lowerPixel;
			double v0 = std::log(src[lowerPixel*srcWidth+x]);
			double v1 = std::log(src[(lowerPixel+1)*srcWidth+x]);
			double vx = v0+(v1-v0)*lowerFrac;

			sum -= ((v0+vx)/2.0)*lowerFrac;
		}

		{
			double upperFrac = (double)upperPixel-p2;
			double v0 = std::log(src[(upperPixel-1)*srcWidth+x]);
			double v1 = std::log(src[upperPixel*srcWidth+x]);
			double vx = v1+(v0-v1)*upperFrac;

			sum -= ((vx+v1)/2.0)*upperFrac;
		}
	}
	
	sum /= (double)(srcHeight-1);

	return sum;
}

std::string SimulationState::getGridPropertyName(int propID)
{
	if (propID < 0 || propID >= SIMSTATE_GRIDPROP_MAX)
		return std::string("Invalid grid property ID");

	switch(propID)
	{
	case SIMSTATE_GRIDPROP_N:
		return std::string("electron number density");
	case SIMSTATE_GRIDPROP_P:
		return std::string("hole number density");
	case SIMSTATE_GRIDPROP_NI:
		return std::string("intrinsic electron number density");
	case SIMSTATE_GRIDPROP_BG:
		return std::string("fixed background charge number density");
	case SIMSTATE_GRIDPROP_V:
		return std::string("potential");
	case SIMSTATE_GRIDPROP_G:
		return std::string("generation rate");
	case SIMSTATE_GRIDPROP_RF:
		return std::string("recombination factor");
	case SIMSTATE_GRIDPROP_DN:
		return std::string("electron diffusion constant");
	case SIMSTATE_GRIDPROP_DP:
		return std::string("hole diffusion constant");
	case SIMSTATE_GRIDPROP_NMOB:
		return std::string("electron mobility");
	case SIMSTATE_GRIDPROP_PMOB:
		return std::string("hole mobility");
	case SIMSTATE_GRIDPROP_EPSREL:
		return std::string("relative permittivity");
	case SIMSTATE_GRIDPROP_VNEXTRA:
		return std::string("extra potential for the electrons");
	case SIMSTATE_GRIDPROP_VPEXTRA:
		return std::string("extra potential for the holes");
	case SIMSTATE_GRIDPROP_JNX:
		return std::string("electron number current in the x direction");
	case SIMSTATE_GRIDPROP_JNY:
		return std::string("electron number current in the y direction");
	case SIMSTATE_GRIDPROP_JPX:
		return std::string("hole number current in the x direction");
	case SIMSTATE_GRIDPROP_JPY:
		return std::string("hole number current in the y direction");
	case SIMSTATE_GRIDPROP_R:
		return std::string("recombination rate");
	case SIMSTATE_GRIDPROP_DISSPROB:
		return std::string("dissociation probability");
	case SIMSTATE_GRIDPROP_NMOB_GAMMA:
		return std::string("field dependent mobility parameter gamma_e");
	case SIMSTATE_GRIDPROP_PMOB_GAMMA:
		return std::string("field dependent mobility parameter gamma_h");
	default:
		// nothing to do
		;
	}

	char str[1024];
	
	sprintf(str, "unused grid property %d", propID);
	return std::string(str);
}

