#ifndef SIMULATION2DNR2_H

#define SIMULATION2DNR2_H

#include "simulation2d.h"
#include "newtonraphsonsearch.h"

class SimulationState;
class InverseMatrixPoissonSolver2D;

class Simulation2DNR2 : public Simulation2D
{
public:
	Simulation2DNR2();
	~Simulation2DNR2();

	// Set grid dimensions
	bool init(int width, int height, double realWidth, double realHeight, double T, double diffScale, double densScale, bool useLog);
	bool isInitialized() const										{ return m_init; }

	int getNumXPixels() const										{ return m_width; }
	int getNumYPixels() const										{ return m_height; }
	double getPixelWidth() const										{ return m_pixelWidth2; }
	double getPixelHeight() const										{ return m_pixelHeight2; }

	double getTemperature() const										{ return m_T; }
	double getDiffusionScale() const									{ return m_diffScale; }
	double getDensScale() const										{ return m_densScale; }
	bool isUsingLog() const											{ return m_useLog; }

	bool setState(const SimulationState &state, std::vector<std::string> &warnings);
	bool storeState(SimulationState &state) const;

	bool save(const std::string &fileName) const;
	bool load(const std::string &fileName);
	bool write(serut::SerializationInterface &s) const;
	bool read(serut::SerializationInterface &s);

	// After 'init' has been called, these functions can be used to specify a specific
	// starting situation
	void setElectronNumberDensity(int x, int y, double dens)						{ setValue(m_n, x, y, dens); }
	void setElectronNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_n, x0, x1, y0, y1, dens); }
	void setElectronNumberDensity(double dens)								{ setValues(m_n, dens); }

	void setHoleNumberDensity(int x, int y, double dens)							{ setValue(m_p, x, y, dens); }
	void setHoleNumberDensity(int x0, int x1, int y0, int y1, double dens)					{ setValues(m_p, x0, x1, y0, y1, dens); }
	void setHoleNumberDensity(double dens)									{ setValues(m_p, dens); }

	void setBackgroundNumberDensity(int x, int y, double dens)						{ setValue(m_background, x, y, dens); }
	void setBackgroundNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_background, x0, x1, y0, y1, dens); }
	void setBackgroundNumberDensity(double dens)								{ setValues(m_background, dens); }

	void setGenerationRate(int x, int y, double rate)							{ setValue(m_generationRate, x, y, rate); }
	void setGenerationRate(int x0, int x1, int y0, int y1, double rate)					{ setValues(m_generationRate, x0, x1, y0, y1, rate); }
	void setGenerationRate(double rate)									{ setValues(m_generationRate, rate); }

	// r = factor * n * p;
	void setRecombinationFactor(int x, int y, double factor)						{ setValue(m_recombinationFactor, x, y, factor); }
	void setRecombinationFactor(int x0, int x1, int y0, int y1, double factor)				{ setValues(m_recombinationFactor, x0, x1, y0, y1, factor); }
	void setRecombinationFactor(double factor)								{ setValues(m_recombinationFactor, factor); }

	// bottom = 0, top = V
	void setPotentialDifference(double V);

	void setElectronDiffusionConstant(int x, int y, double D)						{ setValue(m_De, x, y, D); }
	void setElectronDiffusionConstant(int x0, int x1, int y0, int y1, double D)				{ setValues(m_De, x0, x1, y0, y1, D); }
	void setElectronDiffusionConstant(double D)								{ setValues(m_De, D); }

	void setHoleDiffusionConstant(int x, int y, double D)							{ setValue(m_Dh, x, y, D); }
	void setHoleDiffusionConstant(int x0, int x1, int y0, int y1, double D)					{ setValues(m_Dh, x0, x1, y0, y1, D); }
	void setHoleDiffusionConstant(double D)									{ setValues(m_Dh, D); }

	void setElectronMobility(int x, int y, double mu)							{ setValue(m_eMob, x, y, mu); }
	void setElectronMobility(int x0, int x1, int y0, int y1, double mu)					{ setValues(m_eMob, x0, x1, y0, y1, mu); }
	void setElectronMobility(double mu)									{ setValues(m_eMob, mu); }

	void setHoleMobility(int x, int y, double mu)								{ setValue(m_hMob, x, y, mu); }
	void setHoleMobility(int x0, int x1, int y0, int y1, double mu)						{ setValues(m_hMob, x0, x1, y0, y1, mu); }
	void setHoleMobility(double mu)										{ setValues(m_hMob, mu); }

	void setRelativePermittivity(int x, int y, double epsRel)						{ setValue(m_epsRels, x, y, epsRel); m_epsRelChanged = true; }
	void setRelativePermittivity(int x0, int x1, int y0, int y1, double epsRel)				{ setValues(m_epsRels, x0, x1, y0, y1, epsRel); m_epsRelChanged = true; }
	void setRelativePermittivity(double epsRel)								{ setValues(m_epsRels, epsRel); m_epsRelChanged = true; }

	void setPotential(int x, int y, double v)								{ setValue(m_potential, x, y, v); }
	void setPotential(int x0, int x1, int y0, int y1, double v)						{ setValues(m_potential, x0, x1, y0, y1, v); }
	void setPotential(double v)										{ setValues(m_potential, v); }

	void setExtraElectronPotential(int x, int y, double v)							{ setValue(m_extraPotentialN, x, y, v); }
	void setExtraElectronPotential(int x0, int x1, int y0, int y1, double v)				{ setValues(m_extraPotentialN, x0, x1, y0, y1, v); }
	void setExtraElectronPotential(double v)								{ setValues(m_extraPotentialN, v); }

	void setExtraHolePotential(int x, int y, double v)							{ setValue(m_extraPotentialP, x, y, v); }
	void setExtraHolePotential(int x0, int x1, int y0, int y1, double v)					{ setValues(m_extraPotentialP, x0, x1, y0, y1, v); }
	void setExtraHolePotential(double v)									{ setValues(m_extraPotentialP, v); }

	bool start(double &sigma);
	bool start2(double &sigma);
	bool startV(double &sigma);
	bool startNP(double &sigma);

	bool runV();

	void initPotential();
	
	void calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const;
	void calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const;

	void getElectronNumberDensity(std::vector<double> &dst) const						{ getProperty(m_n, dst); }
	void getHoleNumberDensity(std::vector<double> &dst) const						{ getProperty(m_p, dst); }
	void getBackgroundNumberDensity(std::vector<double> &dst) const						{ getProperty(m_background, dst); }
	void getGenerationRate(std::vector<double> &dst) const							{ getProperty(m_generationRate, dst); }
	void getRecombinationFactor(std::vector<double> &dst) const						{ getProperty(m_recombinationFactor, dst); }
	void getPotential(std::vector<double> &dst) const							{ getProperty(m_potential, dst); }
	void getElectronDiffusionConstant(std::vector<double> &dst) const					{ getProperty(m_De, dst); }
	void getHoleDiffusionConstant(std::vector<double> &dst) const						{ getProperty(m_Dh, dst); }
	void getElectronMobility(std::vector<double> &dst) const						{ getProperty(m_eMob, dst); }
	void getHoleMobility(std::vector<double> &dst) const							{ getProperty(m_hMob, dst); }
	void getRelativePermittivity(std::vector<double> &dst) const						{ getProperty(m_epsRels, dst); }
	void getTotalElectronXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEx, dst, 1.0/m_pixelWidth2); }
	void getTotalElectronYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEy, dst, 1.0/m_pixelHeight2); }
	void getTotalHoleXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHx, dst, 1.0/m_pixelWidth2); }
	void getTotalHoleYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHy, dst, 1.0/m_pixelHeight2); }
	void getExtraElectronPotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialN, dst); }
	void getExtraHolePotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialP, dst); }

	double getPotentialDifference() const									{ return m_deltaPhi; }

	void writePlotData(FILE *pFile);
private:
	int getIndex(int x, int y) const									{ return x+y*m_width; } 
	void setValue(std::vector<double> &grid, int x, int y, double v)					{ grid[getIndex(x,y)] = v; }
	void setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v);
	void setValues(std::vector<double> &grid, double v)							{ setValues(grid, 0, m_width, 0, m_height, v); }
	void getProperty(const std::vector<double> &grid, std::vector<double> &dst) const			{ dst = grid; }
	void getProperty(const std::vector<long double> &grid, std::vector<double> &dst, long double factor) const { dst.resize(grid.size()); for (int i = 0 ; i < grid.size() ; i++) dst[i] = grid[i]*factor; }
	void resizeArrays();

	long double getFValue(int Ftype, int x, int y);
	long double getFV(int x, int y);
	long double getFN(int x, int y);
	long double getFP(int x, int y);
	long double getFJacValue(int Ftype, int x, int y, int varType, int u, int v);
	long double getFJacV(int t, int x, int y, int u, int v);
	long double getFJacN(int t, int x, int y, int u, int v);
	long double getFJacP(int t, int x, int y, int u, int v);
	long double getFJacVV(int x, int y, int u, int v);
	long double getFJacVN(int x, int y, int u, int v);
	long double getFJacVP(int x, int y, int u, int v);
	long double getFJacNV(int x, int y, int u, int v);
	long double getFJacNN(int x, int y, int u, int v);
	long double getFJacNP(int x, int y, int u, int v);
	long double getFJacPV(int x, int y, int u, int v);
	long double getFJacPN(int x, int y, int u, int v);
	long double getFJacPP(int x, int y, int u, int v);
	long double getNetGen(int x, int y);
	long double getNetGenDerivV(int x, int y, int u, int v);
	long double getNetGenDerivN(int x, int y, int u, int v);
	long double getNetGenDerivP(int x, int y, int u, int v);
	void calcFactorsAndCurrents();
	static long double calcW(long double x);
	static long double calcDerivW(long double x);
	static long double calcJ(long double x, long double d1, long double d2);
	static long double calcDerivJx(long double x, long double d1, long double d2);
	static long double calcDerivJd1(long double x, long double d1, long double d2);
	static long double calcDerivJd2(long double x, long double d1, long double d2);

	void calcCurrentHorizontal(int x, int y);
	void calcCurrentVertical(int x, int y);
	void dummyGetMatrix();
	void dummyGetMatrixV();
	void dummyGetMatrixNP();

	bool m_init;
	int m_width, m_height; // ok
	int m_totalPixels;
	
	double m_pixelWidth2, m_pixelHeight2;

	std::vector<double> m_n, m_p, m_background;
	std::vector<double> m_generationRate, m_recombinationFactor; // ok
	std::vector<double> m_De, m_Dh, m_eMob, m_hMob; // ok
	std::vector<double> m_epsRels, m_potential;
	std::vector<long double> m_numCurTotEx, m_numCurTotEy, m_numCurTotHx, m_numCurTotHy; // ok

	std::vector<double> m_extraPotentialN;
	std::vector<double> m_extraPotentialP;

	long double m_DL2, m_DLx2, m_DLy2;
	int m_numVariablePixels;

	double m_deltaPhi; // ok
	long double m_diffScale, m_kTev, m_densScale;
	double m_T;

	bool m_useLog;

	class Search : public NewtonRaphsonSearch
	{
	public:
		Search(Simulation2DNR2 *pInstance);
		~Search();

		int getMaxNonZeroRows(int col);
		void startColumn(int col);
		int getNextNonZeroRow();
		double getFunctionValue(int Findex);
		double getDerivative(int Findex, int varIndex);
	private:
		void getVariableCoordinates(int idx, int &t, int &u, int &v);
		void getFunctionCoordinates(int idx, int &t, int &x, int &y);
		int getFunctionIndex(int t, int x, int y);

		Simulation2DNR2 *m_pSim;
		int m_rowIdx;
		std::vector<int> m_rows;
	};

	class SearchV : public NewtonRaphsonSearch
	{
	public:
		SearchV(Simulation2DNR2 *pInstance);
		~SearchV();

		int getMaxNonZeroRows(int col);
		void startColumn(int col);
		int getNextNonZeroRow();
		double getFunctionValue(int Findex);
		double getDerivative(int Findex, int varIndex);
	private:
		void getVariableCoordinates(int idx, int &u, int &v);
		void getFunctionCoordinates(int idx, int &x, int &y);
		int getFunctionIndex(int x, int y);

		Simulation2DNR2 *m_pSim;
		int m_rowIdx;
		std::vector<int> m_rows;
	};

	class SearchNP : public NewtonRaphsonSearch
	{
	public:
		SearchNP(Simulation2DNR2 *pInstance);
		~SearchNP();

		int getMaxNonZeroRows(int col);
		void startColumn(int col);
		int getNextNonZeroRow();
		double getFunctionValue(int Findex);
		double getDerivative(int Findex, int varIndex);
	private:
		void getVariableCoordinates(int idx, int &t, int &u, int &v);
		void getFunctionCoordinates(int idx, int &t, int &x, int &y);
		int getFunctionIndex(int t, int x, int y);

		Simulation2DNR2 *m_pSim;
		int m_rowIdx;
		std::vector<int> m_rows;
	};

	Search *m_pNRSearch;
	SearchV *m_pNRSearchV;
	SearchNP *m_pNRSearchNP;

	friend class Search;
	friend class SearchV;
	friend class SearchNP;

	InverseMatrixPoissonSolver2D *m_pPoissonSolver;	
	bool m_epsRelChanged;
};

inline void Simulation2DNR2::setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v)
{
	if (x1 < x0)
		return;
	if (y1 < y0)
		return;

	for (int y = y0 ; y != y1 ; y++)
		for (int x = x0 ; x != x1 ; x++)
			setValue(grid, x, y, v);
}

#endif // SIMULATION2DNR2_H

