#include "simulation2dnr2.h"
#include "simulationstate.h"
#include "inversematrixpoissonsolver2d.h"
#include "klu.h"
#include <serut/fileserializer.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <iostream>
#include <cmath>
#include <math.h>
#include <climits>

#ifdef WIN32
#define ISFINITE _finite
#else
#define ISFINITE std::isfinite
#endif // WIN32

using namespace serut;

// NOTE: All units are SI units. Shouldn't be a problem for 'double' representation of real numbers.

class DensScaleStuff2D
{
public:
	DensScaleStuff2D(long double *pScale, std::vector<double> &n, std::vector<double> &p, int w, int h)
	{
		m_scaleBackup = *pScale;
		m_pScale = pScale;

//		std::cout << "Saving scale " << m_scaleBackup << std::endl;

		if (m_scaleBackup < 0)
		{
			double newScale = n[0];

			for (int x = 0 ; x < w ; x++)
			{
				double N1 = n[x];
				double N2 = n[x+(h-1)*w];
				double P1 = p[x];
				double P2 = p[x+(h-1)*w];

				if (N1 > newScale)
					newScale = N1;
				if (N2 > newScale)
					newScale = N2;
				if (P1 > newScale)
					newScale = P1;
				if (P2 > newScale)
					newScale = P2;
			}

			*pScale = newScale;

	//		std::cout << "Storing new scale " << newScale << std::endl;
		}
	}

	~DensScaleStuff2D()
	{
		*m_pScale = m_scaleBackup;
	//	std::cout << "Restoring old scale " << m_scaleBackup << std::endl;
	}
private:
	long double m_scaleBackup;
	long double *m_pScale;
};

class DiffScaleStuff2D
{
public:
	DiffScaleStuff2D(long double *pScale, std::vector<double> &dn, std::vector<double> &dp, int w, int h)
	{
		m_scaleBackup = *pScale;
		m_pScale = pScale;

		if (m_scaleBackup < 0)
		{
			/*
			double newScale = dn[0];

			for (int y = 0 ; y < h ; y++)
			{
				for (int x = 0 ; x < w ; x++)
				{
					int idx = x+y*w;
					double dN = dn[idx];
					double dP = dp[idx];

					if (dN > newScale)
						newScale = dN;
					if (dP > newScale)
						newScale = dP;
				}
			}
			*/
			double newScale = 0;

			for (int y = 0 ; y < h ; y++)
			{
				for (int x = 0 ; x < w ; x++)
				{
					int idx = x+y*w;
					double dN = dn[idx];
					double dP = dp[idx];

					newScale += dN;
					newScale += dP;
				}
			}
			newScale /= (double)(w*h*2);

			*pScale = newScale;
	
			//std::cout << "Storing new diffusion scale " << newScale << std::endl;
		}
	}

	~DiffScaleStuff2D()
	{
		*m_pScale = m_scaleBackup;
	}
private:
	long double m_scaleBackup;
	long double *m_pScale;
};

Simulation2DNR2::Simulation2DNR2()
{
	m_init = false;

	m_width = 0;
	m_height = 0;
	m_totalPixels = 0;
	m_pixelWidth2 = 0;
	m_pixelHeight2 = 0;
	m_deltaPhi = 0;

	m_DL2 = 0;
	m_DLx2 = 0;
	m_DLy2 = 0;
	m_numVariablePixels = 0;

	m_pNRSearch = 0;
	m_pNRSearchV = 0;
	m_pNRSearchNP = 0;
	m_pPoissonSolver = 0;
}

Simulation2DNR2::~Simulation2DNR2()
{
	if (m_pNRSearch)
		delete m_pNRSearch;
	if (m_pNRSearchV)
		delete m_pNRSearchV;
	if (m_pNRSearchNP)
		delete m_pNRSearchNP;
	if (m_pPoissonSolver)
		delete m_pPoissonSolver;
}

bool Simulation2DNR2::init(int width, int height, double realWidth, double realHeight, double T, double diffScale, double densScale, bool useLog)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (width < 3 || height < 3)
	{
		setErrorString("Invalid grid dimensions");
		return false;
	}
	// Calculate the number of pixels on the highest resolution grid.

	int xPixels = width;
	int yPixels = height;

	m_width = xPixels;
	m_height = yPixels;
	m_totalPixels = m_width*m_height;

	m_pixelWidth2 = realWidth/(double)xPixels;
	m_pixelHeight2 = realHeight/(double)(yPixels-1);

	m_DL2 = m_pixelWidth2 * m_pixelHeight2;
	m_DLx2 = m_DL2/(m_pixelWidth2*m_pixelWidth2);
	m_DLy2 = m_DL2/(m_pixelHeight2*m_pixelHeight2);
	m_numVariablePixels = m_width*(m_height-2);

	// Allocate enough memory for various arrays
	resizeArrays();

	m_deltaPhi = 0;

	m_pNRSearch = new Search(this);
	if (!m_pNRSearch->init(m_numVariablePixels*3, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver: " + m_pNRSearch->getErrorString());
		delete m_pNRSearch;
		m_pNRSearch = 0;
		return false;
	}

	m_pNRSearchV = new SearchV(this);
	if (!m_pNRSearchV->init(m_numVariablePixels, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver for V: " + m_pNRSearchV->getErrorString());
		delete m_pNRSearchV;
		delete m_pNRSearch;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		return false;
	}

	m_pNRSearchNP = new SearchNP(this);
	if (!m_pNRSearchNP->init(m_numVariablePixels*2, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver for N & P: " + m_pNRSearchNP->getErrorString());
		delete m_pNRSearchNP;
		delete m_pNRSearchV;
		delete m_pNRSearch;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		m_pNRSearchNP = 0;
		return false;
	}

	m_pPoissonSolver = new InverseMatrixPoissonSolver2D();
	if (!m_pPoissonSolver->init(m_width, m_height, m_pixelWidth2*(double)m_width, m_pixelHeight2*(double)(m_height-1), &(m_potential[0])))
	{
		setErrorString("Couldn't initialze 2D inverse matrix poisson solver: " + m_pPoissonSolver->getErrorString());
		delete m_pNRSearchNP;
		delete m_pNRSearchV;
		delete m_pNRSearch;
		delete m_pPoissonSolver;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		m_pNRSearchNP = 0;
		m_pPoissonSolver = 0;
		return false;
	}

	m_init = true;
	m_T = T;
	m_kTev = T*CONST_K/CHARGE_ELECTRON;
	m_diffScale = diffScale;
	m_densScale = densScale;
	m_useLog = useLog;
	m_epsRelChanged = true;
	//std::cout << "m_densScale = " << m_densScale << std::endl;

	return true;
}

#define SIMULATION2DNR_STARTID	0x4e4d4953
#define SIMULATION2DNR_VERSION	1
#define SIMULATION2DNR_ENDID	0x444e4f43

bool Simulation2DNR2::write(SerializationInterface &s) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninitialized simulation");
		return false;
	}

	s.writeInt32(SIMULATION2DNR_STARTID);
	s.writeInt32(SIMULATION2DNR_VERSION);

	s.writeInt32(m_width);
	s.writeInt32(m_height);
	s.writeDouble(m_pixelWidth2);
	s.writeDouble(m_pixelHeight2);

	int32_t tmp;

	s.writeDouble(m_T);
	s.writeDouble(m_diffScale);
	s.writeDouble(m_densScale);
	
	tmp = (m_useLog)?1:0;

	s.writeInt32(tmp);

	s.writeDoubles(m_n);
	s.writeDoubles(m_p);
	s.writeDoubles(m_background);
	s.writeDoubles(m_generationRate);
	s.writeDoubles(m_recombinationFactor);
	s.writeDoubles(m_De);
	s.writeDoubles(m_Dh);
	s.writeDoubles(m_eMob);
	s.writeDoubles(m_hMob);
	s.writeDoubles(m_epsRels);
	s.writeDoubles(m_potential);
	s.writeDoubles(m_extraPotentialN);
	s.writeDoubles(m_extraPotentialP);

	s.writeLongDoubles(m_numCurTotEx);
	s.writeLongDoubles(m_numCurTotEy);
	s.writeLongDoubles(m_numCurTotHx);
	s.writeLongDoubles(m_numCurTotHy);

	s.writeDouble(m_deltaPhi);

	if (!s.writeInt32(SIMULATION2DNR_ENDID))
	{
		setErrorString(std::string("Error writing to stream: ") + s.getErrorString());
		return false;
	}

	return true;
}

bool Simulation2DNR2::read(SerializationInterface &s)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	int32_t tmp;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file start identifier: ") + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DNR_STARTID)
	{
		setErrorString("Read an invalid data file start identifier");
		return false;
	}
	if (!s.readInt32(&tmp))
	{
		setErrorString("Unable to read file format version: " + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DNR_VERSION)
	{
		setErrorString("Incompatible file format version");
		return false;
	}

	int32_t w, h;
	double pixW, pixH;

	if (!(s.readInt32(&w) && s.readInt32(&h) && s.readDouble(&pixW) && s.readDouble(&pixH)))
	{
		setErrorString("Error reading grid dimensions");
		return false;
	}
	
	m_width = w;
	m_height = h;
	m_pixelWidth2 = pixW;
	m_pixelHeight2 = pixH;
	m_totalPixels = m_width*m_height;
	m_DL2 = m_pixelWidth2 * m_pixelHeight2;
	m_DLx2 = m_DL2/(m_pixelWidth2*m_pixelWidth2);
	m_DLy2 = m_DL2/(m_pixelHeight2*m_pixelHeight2);
	m_numVariablePixels = m_width*(m_height-2);

	double T, diffScale, densScale;
	int32_t flag;

	if (!(s.readDouble(&T) && s.readDouble(&diffScale) && s.readDouble(&densScale) && s.readInt32(&flag)))
	{
		setErrorString("Error reading simulation scales");
		return false;
	}

	m_T = T;
	m_kTev = T*CONST_K/CHARGE_ELECTRON;
	m_diffScale = diffScale;
	m_densScale = densScale;

	if (flag)
		m_useLog = true;
	else
		m_useLog = false;

	resizeArrays();

	s.readDoubles(m_n);
	s.readDoubles(m_p);
	s.readDoubles(m_background);
	s.readDoubles(m_generationRate);
	s.readDoubles(m_recombinationFactor);
	s.readDoubles(m_De);
	s.readDoubles(m_Dh);
	s.readDoubles(m_eMob);
	s.readDoubles(m_hMob);
	s.readDoubles(m_epsRels);
	s.readDoubles(m_potential);
	s.readDoubles(m_extraPotentialN);
	s.readDoubles(m_extraPotentialP);

	s.readLongDoubles(m_numCurTotEx);
	s.readLongDoubles(m_numCurTotEy);
	s.readLongDoubles(m_numCurTotHx);
	s.readLongDoubles(m_numCurTotHy);

	s.readDouble(&m_deltaPhi);

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file end identifier: ") + s.getErrorString());
		return false;
	}

	if (tmp != SIMULATION2DNR_ENDID)
	{
		setErrorString("Read an invalid data file end identifier");
		return false;
	}

	m_pNRSearch = new Search(this);
	if (!m_pNRSearch->init(m_numVariablePixels*3, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver: " + m_pNRSearch->getErrorString());
		delete m_pNRSearch;
		m_pNRSearch = 0;
		return false;
	}

	m_pNRSearchV = new SearchV(this);
	if (!m_pNRSearchV->init(m_numVariablePixels, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver for V: " + m_pNRSearchV->getErrorString());
		delete m_pNRSearchV;
		delete m_pNRSearch;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		return false;
	}

	m_pNRSearchNP = new SearchNP(this);
	if (!m_pNRSearchNP->init(m_numVariablePixels*2, true, true))
	{
		setErrorString("Couldn't initialize Newton-Raphson solver for N & P: " + m_pNRSearchNP->getErrorString());
		delete m_pNRSearchNP;
		delete m_pNRSearchV;
		delete m_pNRSearch;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		m_pNRSearchNP = 0;
		return false;
	}

	m_pPoissonSolver = new InverseMatrixPoissonSolver2D();
	if (!m_pPoissonSolver->init(m_width, m_height, m_pixelWidth2*(double)m_width, m_pixelHeight2*(double)(m_height-1), &(m_potential[0])))
	{
		setErrorString("Couldn't initialze 2D inverse matrix poisson solver: " + m_pPoissonSolver->getErrorString());
		delete m_pNRSearchNP;
		delete m_pNRSearchV;
		delete m_pNRSearch;
		delete m_pPoissonSolver;
		m_pNRSearch = 0;
		m_pNRSearchV = 0;
		m_pNRSearchNP = 0;
		m_pPoissonSolver = 0;
		return false;
	}

	m_init = true;
	m_epsRelChanged = true;

	//std::cout << "m_densScale = " << m_densScale << std::endl;

	return true;
}

bool Simulation2DNR2::save(const std::string &fileName) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninialized simulation");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::WriteOnly))
	{
		setErrorString(std::string("Couldn't save to file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return write(fSer);
}

bool Simulation2DNR2::load(const std::string &fileName)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::ReadOnly))
	{
		setErrorString(std::string("Couldn't load file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return read(fSer);
}

bool Simulation2DNR2::start(double &sigma)
{
	if (!m_init)
	{
		setErrorString("Simulation hasn't been initialized yet");
		return false;
	}

	DensScaleStuff2D ds(&m_densScale, m_n, m_p, m_width, m_height);
	DiffScaleStuff2D diffscale(&m_diffScale, m_De, m_Dh, m_width, m_height);

	// force boundary conditions on potential
	
	for (int x = 0 ; x < m_width ; x++)
	{
		m_potential[x + 0 * m_width] = 0;
		m_potential[x + (m_height-1)*m_width] = m_deltaPhi;
	}

	// calculate factors and currents needed later
	
	calcFactorsAndCurrents();
	
	{
		double sum1 = 0;
		double sum2 = 0;
		double sum3 = 0;

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFV(x,y);
				//std::cout << "dV(" << x << "," << y << ") = " << d << std::endl;
				sum1 += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFN(x,y);
				//std::cout << "dN(" << x << "," << y << ") = " << d << std::endl;
				sum2 += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFP(x,y);
				//std::cout << "dP(" << x << "," << y << ") = " << d << std::endl;
				sum3 += d*d;
			}
		}

		double sum = sum1+sum2+sum3;

		double newSigma = std::sqrt(sum/(double)(m_numVariablePixels*3));
	
		//std::cout << "START: (" << sum1 << "," << sum2 << "," << sum3 << ") => " << newSigma << std::endl;
	}
	//exit(-1);

	std::vector<long double> curEX = m_numCurTotEx;
	std::vector<long double> curEY = m_numCurTotEy;
	std::vector<long double> curHX = m_numCurTotHx;
	std::vector<long double> curHY = m_numCurTotHy;

	if (!m_pNRSearch->step(sigma))
	{
		setErrorString("Couldn't run Newton-Raphson algorithm: " + m_pNRSearch->getErrorString());
		return false;
	}

	//dummyGetMatrix();

	const double *pDiffs = m_pNRSearch->getDifferences();
	int dim = m_pNRSearch->getDimension();

	// TODO: need a better search for the best value here

	int maxIt = 256;
	double factor = 1.0;
	//double div = 2.0;
	double div = 1.1;
	double prevSigma = -1;
	bool done = false;
	double bestFactor = 0;
	double bestSigma = 1e200;

	//std::cout << "initial sigma is " << sigma << std::endl;

	std::vector<double> v = m_potential;
	std::vector<double> n = m_n;
	std::vector<double> p = m_p;

	int it = 0;

	while (!done && it++ < maxIt)
	{
		if (m_useLog)
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

				m_potential[idx] = v[idx] + pDiffs[i]*factor;
				m_n[idx] = n[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
				m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels*2]*factor/m_kTev);
			}
		}
		else
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

				m_potential[idx] = v[idx] + pDiffs[i]*factor;
				m_n[idx] = n[idx] + pDiffs[i+m_numVariablePixels]*factor;
				m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels*2]*factor;
			}
		}

		calcFactorsAndCurrents();

		double newSigma = 0;
		double sum1 = 0;
		double sum2 = 0;
		double sum3 = 0;

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFV(x,y);
				sum1 += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFN(x,y);
				sum2 += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFP(x,y);
				sum3 += d*d;
			}
		}

		double sum = sum1+sum2+sum3;

		newSigma = std::sqrt(sum/(double)(m_numVariablePixels*3));
		//std::cout << it << ". factor = " << factor << " newSigma = " << newSigma << std::endl;

		//std::cout << it << ": (" << sum1 << "," << sum2 << "," << sum3 << ") => " << newSigma << std::endl;
		
		if (ISFINITE(newSigma))
		{
			if (bestSigma > newSigma)
			{
				bestFactor = factor;
				bestSigma = newSigma;
			}
		}

		if (prevSigma >= 0)
		{
			if (newSigma >= prevSigma) // need the previous setting
			{
				done = true;
			}
			else
			{
				// keep looking
			}
		}

		if (ISFINITE(newSigma))
			prevSigma = newSigma;

		factor /= div;		
	}

	m_numCurTotEx = curEX;
	m_numCurTotEy = curEY;
	m_numCurTotHx = curHX;
	m_numCurTotHy = curHY;

	if (!done)
		std::cerr << "Warning: max count reached" << std::endl;

	if (bestSigma > sigma)
		factor = 0;
	else
		factor = bestFactor;
	//std::cout << "Best factor = " << factor << std::endl;

	if (m_useLog)
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			m_potential[idx] = v[idx] + pDiffs[i]*factor;
			m_n[idx] = n[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
			m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels*2]*factor/m_kTev);
		}
	}
	else
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			m_potential[idx] = v[idx] + pDiffs[i]*factor;
			m_n[idx] = n[idx] + pDiffs[i+m_numVariablePixels]*factor;
			m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels*2]*factor;
		}
	}

	return true;
}

bool Simulation2DNR2::startV(double &sigma)
{
	if (!m_init)
	{
		setErrorString("Simulation hasn't been initialized yet");
		return false;
	}

	DensScaleStuff2D ds(&m_densScale, m_n, m_p, m_width, m_height);
	DiffScaleStuff2D diffscale(&m_diffScale, m_De, m_Dh, m_width, m_height);

	// force boundary conditions on potential
	
	for (int x = 0 ; x < m_width ; x++)
	{
		m_potential[x + 0 * m_width] = 0;
		m_potential[x + (m_height-1)*m_width] = m_deltaPhi;
	}

	calcFactorsAndCurrents();

	/*
	std::vector<long double> curEX = m_numCurTotEx;
	std::vector<long double> curEY = m_numCurTotEy;
	std::vector<long double> curHX = m_numCurTotHx;
	std::vector<long double> curHY = m_numCurTotHy;
	*/

	if (!m_pNRSearchV->step(sigma))
	{
		setErrorString("Couldn't run Newton-Raphson algorithm for V: " + m_pNRSearchV->getErrorString());
		return false;
	}

	//dummyGetMatrix();

	const double *pDiffs = m_pNRSearchV->getDifferences();
	int dim = m_pNRSearchV->getDimension();

	// TODO: need a better search for the best value here

	int maxIt = 256;
	double factor = 1.0;
	//double div = 2.0;
	double div = 1.1;
	double prevSigma = -1;
	bool done = false;
	double bestFactor = 0;
	double bestSigma = 1e200;

	//std::cout << "initial sigma is " << sigma << std::endl;

	std::vector<double> v = m_potential;

	int it = 0;

	while (!done && it++ < maxIt)
	{
		if (m_useLog)
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

				m_potential[idx] = v[idx] + pDiffs[i]*factor;
//				m_n[idx] = n[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
//				m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels*2]*factor/m_kTev);
			}
		}
		else
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

				m_potential[idx] = v[idx] + pDiffs[i]*factor;
//				m_n[idx] = n[idx] + pDiffs[i+m_numVariablePixels]*factor;
//				m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels*2]*factor;
			}
		}

//		calcFactorsAndCurrents();

		double newSigma = 0;
		double sum = 0;

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFV(x,y);
				sum += d*d;
			}
		}

		/*
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFN(x,y);
				sum += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFP(x,y);
				sum += d*d;
			}
		}
		*/

		newSigma = std::sqrt(sum/(double)(m_numVariablePixels));
		//std::cout << it << ". factor = " << factor << " newSigma = " << newSigma << std::endl;
		
		if (ISFINITE(newSigma))
		{
			if (bestSigma > newSigma)
			{
				bestFactor = factor;
				bestSigma = newSigma;
			}
		}

		if (prevSigma >= 0)
		{
			if (newSigma >= prevSigma) // need the previous setting
			{
				done = true;
			}
			else
			{
				// keep looking
			}
		}

		if (ISFINITE(newSigma))
			prevSigma = newSigma;

		factor /= div;		
	}

	/*
	m_numCurTotEx = curEX;
	m_numCurTotEy = curEY;
	m_numCurTotHx = curHX;
	m_numCurTotHy = curHY;
	*/

	if (!done)
		std::cerr << "Warning: max count reached" << std::endl;

	if (bestSigma > sigma)
		factor = 0;
	else
		factor = bestFactor;
	//std::cout << "Best factor = " << factor << std::endl;

	if (m_useLog)
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			m_potential[idx] = v[idx] + pDiffs[i]*factor;
//			m_n[idx] = n[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
//			m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels*2]*factor/m_kTev);
		}
	}
	else
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			m_potential[idx] = v[idx] + pDiffs[i]*factor;
//			m_n[idx] = n[idx] + pDiffs[i+m_numVariablePixels]*factor;
//			m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels*2]*factor;
		}
	}

	return true;
}

bool Simulation2DNR2::startNP(double &sigma)
{
	if (!m_init)
	{
		setErrorString("Simulation hasn't been initialized yet");
		return false;
	}

	DensScaleStuff2D ds(&m_densScale, m_n, m_p, m_width, m_height);
	DiffScaleStuff2D diffscale(&m_diffScale, m_De, m_Dh, m_width, m_height);

	// force boundary conditions on potential
	
	for (int x = 0 ; x < m_width ; x++)
	{
		m_potential[x + 0 * m_width] = 0;
		m_potential[x + (m_height-1)*m_width] = m_deltaPhi;
	}

	// calculate factors and currents needed later
	
	calcFactorsAndCurrents();

	std::vector<long double> curEX = m_numCurTotEx;
	std::vector<long double> curEY = m_numCurTotEy;
	std::vector<long double> curHX = m_numCurTotHx;
	std::vector<long double> curHY = m_numCurTotHy;

	if (!m_pNRSearchNP->step(sigma))
	{
		setErrorString("Couldn't run Newton-Raphson algorithm: " + m_pNRSearchNP->getErrorString());
		return false;
	}

	//dummyGetMatrix();

	const double *pDiffs = m_pNRSearchNP->getDifferences();
	int dim = m_pNRSearchNP->getDimension();

	// TODO: need a better search for the best value here

	int maxIt = 256;
	double factor = 1.0;
	//double div = 2.0;
	double div = 1.1;
	double prevSigma = -1;
	bool done = false;
	double bestFactor = 0;
	double bestSigma = 1e200;

	//std::cout << "initial sigma is " << sigma << std::endl;

	// std::vector<double> v = m_potential;
	std::vector<double> n = m_n;
	std::vector<double> p = m_p;

	int it = 0;

	while (!done && it++ < maxIt)
	{
		if (m_useLog)
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

	//			m_potential[idx] = v[idx] + pDiffs[i]*factor;
				m_n[idx] = n[idx] * std::exp(pDiffs[i]*factor/m_kTev);
				m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
			}
		}
		else
		{
			for (int i = 0 ; i < m_numVariablePixels ; i++)
			{
				int x = i%m_width;
				int y = i/m_width+1;
				int idx = x+y*m_width;

	//			m_potential[idx] = v[idx] + pDiffs[i]*factor;
				m_n[idx] = n[idx] + pDiffs[i]*factor;
				m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels]*factor;
			}
		}

		calcFactorsAndCurrents();

		double newSigma = 0;
		double sum = 0;

		/*
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFV(x,y);
				sum += d*d;
			}
		}*/
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFN(x,y);
				sum += d*d;
			}
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				double d = 0;

				d = getFP(x,y);
				sum += d*d;
			}
		}

		newSigma = std::sqrt(sum/(double)(m_numVariablePixels*2));
		//std::cout << it << ". factor = " << factor << " newSigma = " << newSigma << std::endl;
		
		if (ISFINITE(newSigma))
		{
			if (bestSigma > newSigma)
			{
				bestFactor = factor;
				bestSigma = newSigma;
			}
		}

		if (prevSigma >= 0)
		{
			if (newSigma >= prevSigma) // need the previous setting
			{
				done = true;
			}
			else
			{
				// keep looking
			}
		}

		if (ISFINITE(newSigma))
			prevSigma = newSigma;

		factor /= div;		
	}

	m_numCurTotEx = curEX;
	m_numCurTotEy = curEY;
	m_numCurTotHx = curHX;
	m_numCurTotHy = curHY;

	if (!done)
		std::cerr << "Warning: max count reached" << std::endl;

	if (bestSigma > sigma)
		factor = 0;
	else
		factor = bestFactor;
	//std::cout << "Best factor = " << factor << std::endl;

	if (m_useLog)
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			//m_potential[idx] = v[idx] + pDiffs[i]*factor;
			m_n[idx] = n[idx] * std::exp(pDiffs[i]*factor/m_kTev);
			m_p[idx] = p[idx] * std::exp(pDiffs[i+m_numVariablePixels]*factor/m_kTev);
		}
	}
	else
	{
		for (int i = 0 ; i < m_numVariablePixels ; i++)
		{
			int x = i%m_width;
			int y = i/m_width+1;
			int idx = x+y*m_width;

			//m_potential[idx] = v[idx] + pDiffs[i]*factor;
			m_n[idx] = n[idx] + pDiffs[i]*factor;
			m_p[idx] = p[idx] + pDiffs[i+m_numVariablePixels]*factor;
		}
	}

	return true;
}

bool Simulation2DNR2::runV()
{
	if (!m_init)
	{
		setErrorString("Simulation hasn't been initialized yet");
		return false;
	}

	if (m_epsRelChanged)
	{
		m_epsRelChanged = false;
		if (!m_pPoissonSolver->setRelativePermittivity(&(m_epsRels[0])))
		{
			setErrorString("Couldn't set relative permittivity in potential solver: " + m_pPoissonSolver->getErrorString());
			return false;
		}
	}

	if (!m_pPoissonSolver->setPotentialDifference(m_deltaPhi))
	{
		setErrorString("Couldn't set potential difference in potential solver: " + m_pPoissonSolver->getErrorString());
		return false;
	}

	if (!m_pPoissonSolver->findPotential(&(m_p[0]), &(m_n[0]), &(m_background[0])))
	{
		setErrorString("Couldn't solve for potential: " + m_pPoissonSolver->getErrorString());
		return false;
	}

	return true;
}

void Simulation2DNR2::initPotential()
{
	for (int y = 0 ; y < m_height ; y++)
	{
		double V = (double)y/(double)(m_height-1) * m_deltaPhi;

		for (int x = 0 ; x < m_width ; x++)
			m_potential[x+y*m_width] = V;
	}
	
	/*
	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			double lN = (double)y/(double)(m_height-1) * (std::log(m_n[x + (m_height-1)*m_width]) 
					                            - std::log(m_n[x + 0*m_width])) + std::log(m_n[x + 0*m_width]);

			m_n[x+y*m_width] = std::exp(lN);

			double lP = (double)y/(double)(m_height-1) * (std::log(m_p[x + (m_height-1)*m_width]) 
					                            - std::log(m_p[x + 0*m_width])) + std::log(m_p[x + 0*m_width]);

			m_p[x+y*m_width] = std::exp(lP);

		}
	}
	*/
}

// Helper function to calculate the current in the X-direction at the left side, the right side
// and averaged over the entire grid

void Simulation2DNR2::calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int y = 1 ; y < m_height-1 ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;

			double JexCur = m_numCurTotEx[idx];
			double JhxCur = m_numCurTotHx[idx];
			
			sum += -JexCur+JhxCur;
		}

		int idx = 0+y*m_width;

		double JexCur = m_numCurTotEx[idx];
		double JhxCur = m_numCurTotHx[idx];

		sum2 += -JexCur + JhxCur;

		idx = m_width-1+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum3 += -JexCur + JhxCur;

		idx = m_width/2+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum4 += -JexCur + JhxCur;
	}

	sum /= m_width*(m_height-2);
	sum2 /= m_height-2;
	sum3 /= m_height-2;
	sum4 /= m_height-2;

	sum /= m_pixelWidth2;
	sum2 /= m_pixelWidth2;
	sum3 /= m_pixelWidth2;
	sum4 /= m_pixelWidth2;

	leftAvg = sum2;
	rightAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

// Helper function to calculate the current in the Y-direction at the bottom side, the top side
// and averaged over the entire grid

void Simulation2DNR2::calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{
			int idx = x+y*m_width;

			double JeyCur = m_numCurTotEy[idx];
			double JhyCur = m_numCurTotHy[idx];
			
			sum += -JeyCur+JhyCur;
		}

		int idx = x+0*m_width;

		double JeyCur = m_numCurTotEy[idx];
		double JhyCur = m_numCurTotHy[idx];

		sum2 += -JeyCur + JhyCur;

		idx = x+(m_height-2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum3 += -JeyCur + JhyCur;

		idx = x+(m_height/2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum4 += -JeyCur + JhyCur;

	}

	sum /= m_width*(m_height-1);
	sum2 /= m_width;
	sum3 /= m_width;
	sum4 /= m_width;

	sum /= m_pixelHeight2;
	sum2 /= m_pixelHeight2;
	sum3 /= m_pixelHeight2;
	sum4 /= m_pixelHeight2;

	bottomAvg = sum2;
	topAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

void Simulation2DNR2::resizeArrays()
{
	m_n.resize(m_totalPixels);
	m_p.resize(m_totalPixels);
	m_background.resize(m_totalPixels);
	m_generationRate.resize(m_totalPixels);
	m_recombinationFactor.resize(m_totalPixels);
	m_De.resize(m_totalPixels);
	m_Dh.resize(m_totalPixels);
	m_eMob.resize(m_totalPixels);
	m_hMob.resize(m_totalPixels);
	m_epsRels.resize(m_totalPixels);
	m_potential.resize(m_totalPixels);

	// Initialize the values of some arrays

	setValues(m_n, 0);
	setValues(m_p, 0);
	setValues(m_background, 0);
	setValues(m_generationRate, 0);
	setValues(m_recombinationFactor, 0);
	setValues(m_De, 0);
	setValues(m_Dh, 0);
	setValues(m_eMob, 0);
	setValues(m_hMob, 0);
	setValues(m_epsRels, 1);

	m_epsRelChanged = true;

	setValues(m_background, 0);

	// Allocate memory for the arrays which will contain the number currents
	// due to diffusion and electric field respectively

	m_numCurTotEx.resize(m_totalPixels);
	m_numCurTotEy.resize(m_totalPixels);
	m_numCurTotHx.resize(m_totalPixels);
	m_numCurTotHy.resize(m_totalPixels);

	m_extraPotentialN.resize(m_totalPixels);
	m_extraPotentialP.resize(m_totalPixels);

	setValues(m_extraPotentialN, 0);
	setValues(m_extraPotentialP, 0);
}

void Simulation2DNR2::writePlotData(FILE *pFile)
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x + y*m_width;
			double j1 = 0;
			double j2 = 0;

			if (y == m_height-1)
			{
				j1 = m_numCurTotHx[idx-m_width]-m_numCurTotEx[idx-m_width];
				j2 = m_numCurTotHy[idx-m_width]-m_numCurTotEy[idx-m_width];
			}
			else
			{
				j1 = m_numCurTotHx[idx]-m_numCurTotEx[idx];
				j2 = m_numCurTotHy[idx]-m_numCurTotEy[idx];
			}

			fprintf(pFile, "%d\t%d\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\t%.30g\n", x, y, 
					         (double)m_n[idx], (double)m_p[idx], (double)m_potential[idx], 
					         (double)(m_recombinationFactor[idx]*m_n[idx]*m_p[idx]),
						 (double)(j1/m_pixelWidth2), (double)(j2/m_pixelHeight2),
						 (double)m_generationRate[idx], (double)m_recombinationFactor[idx],
						 (double)m_De[idx], (double)m_Dh[idx], (double)m_eMob[idx], (double)m_hMob[idx]);
		}
		fprintf(pFile, "\n");
	}
	fprintf(pFile, "\n");

	fprintf(pFile, "# %d\n# %d\n# %d\n", m_width, m_height, m_totalPixels);
	fprintf(pFile, "# %g\n# %g\n# %g\n# %g\n# %g\n",  m_pixelWidth2, m_pixelHeight2, (double)m_DL2, (double)m_DLx2, (double)m_DLy2);
	fprintf(pFile, "# %d\n# %g\n",  m_numVariablePixels, m_deltaPhi);
	fprintf(pFile, "# %g\n# %g\n# %g\n", (double)m_diffScale, (double)m_kTev, (double)m_densScale);
	fprintf(pFile, "# %g\n", m_T);

}

long double Simulation2DNR2::getFValue(int Ftype, int xPix, int yPix)
{
	long double value = -12345;

	switch(Ftype)
	{
	case 0:
		value = getFV(xPix, yPix);
		break;
	case 1:
		value = getFN(xPix, yPix);
		break;
	case 2:
		value = getFP(xPix, yPix);
		break;
	default:
		std::cerr << "Simulation2DNR2::getFValue: invalid type " << Ftype << std::endl;
	}

	return value;
}

long double Simulation2DNR2::getFV(int x, int y)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = -2.0 * (m_DLy2 + m_DLx2) * m_epsRels[idx] * m_potential[idx];

	sum += m_potential[rightIndex] * m_DLx2 * 0.5 * (m_epsRels[idx] + m_epsRels[rightIndex]);
	sum += m_potential[leftIndex] * m_DLx2 * 0.5 * (m_epsRels[idx] + m_epsRels[leftIndex]);
	sum += m_potential[upIndex] * m_DLy2 * 0.5 * (m_epsRels[idx] + m_epsRels[upIndex]);
	sum += m_potential[downIndex] * m_DLy2 * 0.5 * (m_epsRels[idx] + m_epsRels[downIndex]);

	sum += m_DL2*(m_p[idx] - m_n[idx] + m_background[idx])*CHARGE_ELECTRON/CONST_EPSILON0;

	return sum;
}

long double Simulation2DNR2::getFN(int x, int y)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int leftIndex = xPrev + y*m_width;
	int downIndex = idx - m_width;

	long double sum = getNetGen(x, y)*m_DL2;

	sum -= m_DLx2*(m_numCurTotEx[idx] - m_numCurTotEx[leftIndex]);
	sum -= m_DLy2*(m_numCurTotEy[idx] - m_numCurTotEy[downIndex]);

	if (m_useLog)
		return sum/m_n[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation2DNR2::getFP(int x, int y)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = getNetGen(x, y)*m_DL2;

	sum -= m_DLx2*(m_numCurTotHx[idx] - m_numCurTotHx[leftIndex]);
	sum -= m_DLy2*(m_numCurTotHy[idx] - m_numCurTotHy[downIndex]);

	if (m_useLog)
		return sum/m_p[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation2DNR2::getFJacValue(int Ftype, int xPix, int yPix, int varType, int u, int v)
{
	int xPix2 = u;
	int yPix2 = v;

	long double value = -12345;

	switch(Ftype)
	{
	case 0:
		value = getFJacV(varType, xPix, yPix, xPix2, yPix2);
		break;
	case 1:
		value = getFJacN(varType, xPix, yPix, xPix2, yPix2);
		break;
	case 2:
		value = getFJacP(varType, xPix, yPix, xPix2, yPix2);
		break;
	default:
		std::cerr << "Simulation2DNR2::getFJacValue: invalid type " << Ftype << std::endl;
	}

	return value;
}

long double Simulation2DNR2::getFJacV(int t, int x, int y, int u, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacVV(x, y, u, v);
		break;
	case 1:
		value = getFJacVN(x, y, u, v);
		break;
	case 2:
		value = getFJacVP(x, y, u, v);
		break;
	default:
		std::cerr << "Simulation2DNR2::getFJacV: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation2DNR2::getFJacN(int t, int x, int y, int u, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacNV(x, y, u, v);
		break;
	case 1:
		value = getFJacNN(x, y, u, v);
		break;
	case 2:
		value = getFJacNP(x, y, u, v);
		break;
	default:
		std::cerr << "Simulation2DNR2::getFJacN: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation2DNR2::getFJacP(int t, int x, int y, int u, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacPV(x, y, u, v);
		break;
	case 1:
		value = getFJacPN(x, y, u, v);
		break;
	case 2:
		value = getFJacPP(x, y, u, v);
		break;
	default:
		std::cerr << "Simulation2DNR2::getFJacP: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation2DNR2::getFJacVV(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = 0;

	if (x == u)
	{
		if (y == v)
			sum += -2.0*(m_DLx2+m_DLy2)*m_epsRels[idx];
		if (y+1 == v)
			sum += m_DLy2*0.5*(m_epsRels[idx]+m_epsRels[upIndex]);
		if (y-1 == v)
			sum += m_DLy2*0.5*(m_epsRels[idx]+m_epsRels[downIndex]);
	}
	if (y == v)
	{
		if (xNext == u)
			sum += m_DLx2*0.5*(m_epsRels[idx]+m_epsRels[rightIndex]);
		if (xPrev == u)
			sum += m_DLx2*0.5*(m_epsRels[idx]+m_epsRels[leftIndex]);
	}

	return sum;
}

long double Simulation2DNR2::getFJacVN(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	long double sum = 0;
	
	if (x == u && y == v)
		sum += -m_DL2*CHARGE_ELECTRON/CONST_EPSILON0;

	if (m_useLog)
		return sum*m_n[idx]/m_kTev;
	return sum;
}

long double Simulation2DNR2::getFJacVP(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	double sum = 0;

	if (x == u && y == v)
		sum += m_DL2*CHARGE_ELECTRON/CONST_EPSILON0;

	if (m_useLog)
		return sum*m_p[idx]/m_kTev;
	return sum;
}

long double Simulation2DNR2::getFJacNV(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivV(x, y, u, v);

	if (y == v)
	{
		int d1 = (xNext == u) - (x == u);
		int d2 = (x == u) - (xPrev == u);

		if (d1 || d2)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double xRight = ((m_eMob[idx]+m_eMob[rightIndex])/(m_De[idx]+m_De[rightIndex])) * ( (m_potential[rightIndex]+m_extraPotentialN[rightIndex]) - (m_potential[idx]+m_extraPotentialN[idx]) );
			long double xLeft = ((m_eMob[idx]+m_eMob[leftIndex])/(m_De[idx]+m_De[leftIndex])) * ( (m_potential[idx]+m_extraPotentialN[idx]) - (m_potential[leftIndex]+m_extraPotentialN[leftIndex]) );
			sum -= m_DLx2*( 0.5*(m_eMob[idx]+m_eMob[rightIndex])*calcDerivJx(xRight, m_n[idx], m_n[rightIndex])*F1
				       -0.5*(m_eMob[idx]+m_eMob[leftIndex])*calcDerivJx(xLeft, m_n[leftIndex], m_n[idx])*F2);
		}
	}
	if (x == u)
	{
		int d1 = (y+1 == v) - (y == v);
		int d2 = (y == v) - (y-1 == v);

		if (d1 || d2)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double xUp = ((m_eMob[idx]+m_eMob[upIndex])/(m_De[idx]+m_De[upIndex])) * ( (m_potential[upIndex]+m_extraPotentialN[upIndex]) - (m_potential[idx]+m_extraPotentialN[idx]) );
			long double xDown = ((m_eMob[idx]+m_eMob[downIndex])/(m_De[idx]+m_De[downIndex])) * ( (m_potential[idx]+m_extraPotentialN[idx]) - (m_potential[downIndex]+m_extraPotentialN[downIndex]) );

			sum -= m_DLy2*( 0.5*(m_eMob[idx]+m_eMob[upIndex])*calcDerivJx(xUp, m_n[idx], m_n[upIndex])*F1
				       -0.5*(m_eMob[idx]+m_eMob[downIndex])*calcDerivJx(xDown, m_n[downIndex], m_n[idx])*F2);
		}
	}

	if (m_useLog)
		return sum/m_n[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation2DNR2::getFJacNN(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivN(x, y, u, v);

	if (y == v)
	{
		int d1 = (x == u);
		int d2 = (xNext == u);
		int d3 = (xPrev == u);

		if (d1 || d2 || d3)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double F3 = (long double)d3;

			long double xRight = ((m_eMob[idx]+m_eMob[rightIndex])/(m_De[idx]+m_De[rightIndex])) * ( (m_potential[rightIndex]+m_extraPotentialN[rightIndex]) - (m_potential[idx]+m_extraPotentialN[idx]) );
			long double xLeft = ((m_eMob[idx]+m_eMob[leftIndex])/(m_De[idx]+m_De[leftIndex])) * ( (m_potential[idx]+m_extraPotentialN[idx]) - (m_potential[leftIndex]+m_extraPotentialN[leftIndex]) );

			sum -= m_DLx2*(
				0.5*(m_De[idx]+m_De[rightIndex])
				   *(F1*calcDerivJd1(xRight, m_n[idx], m_n[rightIndex])
			            +F2*calcDerivJd2(xRight, m_n[idx], m_n[rightIndex]) )
			       -0.5*(m_De[idx]+m_De[leftIndex])
			           *(F3*calcDerivJd1(xLeft, m_n[leftIndex], m_n[idx])
			            +F1*calcDerivJd2(xLeft, m_n[leftIndex], m_n[idx]) )
				);
		}
	}

	if (x == u)
	{
		int d1 = (y == v);
		int d2 = (y+1 == v);
		int d3 = (y-1 == v);

		if (d1 || d2 || d3)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double F3 = (long double)d3;

			long double xUp = ((m_eMob[idx]+m_eMob[upIndex])/(m_De[idx]+m_De[upIndex])) * ( (m_potential[upIndex]+m_extraPotentialN[upIndex]) - (m_potential[idx]+m_extraPotentialN[idx]) );
			long double xDown = ((m_eMob[idx]+m_eMob[downIndex])/(m_De[idx]+m_De[downIndex])) * ( (m_potential[idx]+m_extraPotentialN[idx]) - (m_potential[downIndex]+m_extraPotentialN[downIndex]) );

			sum -= m_DLy2*(
				0.5*(m_De[idx]+m_De[upIndex])
				   *(F1*calcDerivJd1(xUp, m_n[idx], m_n[upIndex])
			            +F2*calcDerivJd2(xUp, m_n[idx], m_n[upIndex]) )
			       -0.5*(m_De[idx]+m_De[downIndex])
			           *(F3*calcDerivJd1(xDown, m_n[downIndex], m_n[idx])
			            +F1*calcDerivJd2(xDown, m_n[downIndex], m_n[idx]) )
				);
		}

	}

	if (m_useLog)
	{
		if (sum != 0)
		{
			int idx2 = u+v*m_width;

			sum *= (m_n[idx2]/m_n[idx]);
			sum /= m_diffScale;
		}

		if (x == u && y == v)
			sum -= getFN(x, y)/m_kTev; // getFN has already been divided by diffscale

		return sum;
	}
	else
		sum = sum/m_densScale*m_kTev/m_diffScale;

	return sum;
}

long double Simulation2DNR2::getFJacNP(int x, int y, int u, int v)
{
	long double val = m_DL2*getNetGenDerivP(x, y, u, v);
	int idx2 = u+v*m_width;
	int idx = x+y*m_width;

	if (m_useLog)
	{
		if (val != 0)
			val *= (m_p[idx2]/m_n[idx])/m_diffScale;
	}
	else
		val = val/m_densScale*m_kTev/m_diffScale;

	return val;
}

long double Simulation2DNR2::getFJacPV(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivV(x, y, u, v);

	if (y == v)
	{
		int d1 = (xNext == u) - (x == u);
		int d2 = (x == u) - (xPrev == u);

		if (d1 || d2)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double xRight = -((m_hMob[idx]+m_hMob[rightIndex])/(m_Dh[idx]+m_Dh[rightIndex])) * ( (m_potential[rightIndex]+m_extraPotentialP[rightIndex]) - (m_potential[idx]+m_extraPotentialP[idx]) );
			long double xLeft = -((m_hMob[idx]+m_hMob[leftIndex])/(m_Dh[idx]+m_Dh[leftIndex])) * ( (m_potential[idx]+m_extraPotentialP[idx]) - (m_potential[leftIndex]+m_extraPotentialP[leftIndex]) );

			sum -= m_DLx2*(-0.5*(m_hMob[idx]+m_hMob[rightIndex])*calcDerivJx(xRight, m_p[idx], m_p[rightIndex])*F1
				       +0.5*(m_hMob[idx]+m_hMob[leftIndex])*calcDerivJx(xLeft, m_p[leftIndex], m_p[idx])*F2);
		}
	}
	if (x == u)
	{
		int d1 = (y+1 == v) - (y == v);
		int d2 = (y == v) - (y-1 == v);

		if (d1 || d2)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double xUp = -((m_hMob[idx]+m_hMob[upIndex])/(m_Dh[idx]+m_Dh[upIndex])) * ( (m_potential[upIndex]+m_extraPotentialP[upIndex]) - (m_potential[idx]+m_extraPotentialP[idx]) );
			long double xDown = -((m_hMob[idx]+m_hMob[downIndex])/(m_Dh[idx]+m_Dh[downIndex])) * ( (m_potential[idx]+m_extraPotentialP[idx]) - (m_potential[downIndex]+m_extraPotentialP[downIndex]) );

			sum -= m_DLy2*(-0.5*(m_hMob[idx]+m_hMob[upIndex])*calcDerivJx(xUp, m_p[idx], m_p[upIndex])*F1
				       +0.5*(m_hMob[idx]+m_hMob[downIndex])*calcDerivJx(xDown, m_p[downIndex], m_p[idx])*F2);
		}
	}

	if (m_useLog)
		return sum/m_p[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation2DNR2::getFJacPN(int x, int y, int u, int v)
{
	long double val = m_DL2*getNetGenDerivN(x, y, u, v);
	int idx2 = u+v*m_width;
	int idx = x+y*m_width;

	if (m_useLog)
	{
		if (val != 0)
			val *= (m_n[idx2]/m_p[idx])/m_diffScale;
	}
	else
		val = val/m_densScale*m_kTev/m_diffScale;

	return val;
}

long double Simulation2DNR2::getFJacPP(int x, int y, int u, int v)
{
	int idx = x + y*m_width;
	int xPrev = (x-1+m_width)%m_width;
	int xNext = (x+1)%m_width;
	int rightIndex = xNext + y*m_width;
	int leftIndex = xPrev + y*m_width;
	int upIndex  = idx + m_width;
	int downIndex = idx - m_width;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivP(x, y, u, v);

	if (y == v)
	{
		int d1 = (x == u);
		int d2 = (xNext == u);
		int d3 = (xPrev == u);

		if (d1 || d2 || d3)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double F3 = (long double)d3;

			long double xRight = -((m_hMob[idx]+m_hMob[rightIndex])/(m_Dh[idx]+m_Dh[rightIndex])) * ( (m_potential[rightIndex]+m_extraPotentialP[rightIndex]) - (m_potential[idx]+m_extraPotentialP[idx]) );
			long double xLeft = -((m_hMob[idx]+m_hMob[leftIndex])/(m_Dh[idx]+m_Dh[leftIndex])) * ( (m_potential[idx]+m_extraPotentialP[idx]) - (m_potential[leftIndex]+m_extraPotentialP[leftIndex]) );

			sum -= m_DLx2*(
				0.5*(m_Dh[idx]+m_Dh[rightIndex])
				   *(F1*calcDerivJd1(xRight, m_p[idx], m_p[rightIndex])
			            +F2*calcDerivJd2(xRight, m_p[idx], m_p[rightIndex]) )
			       -0.5*(m_Dh[idx]+m_Dh[leftIndex])
			           *(F3*calcDerivJd1(xLeft, m_p[leftIndex], m_p[idx])
			            +F1*calcDerivJd2(xLeft, m_p[leftIndex], m_p[idx]) )
				);
		}
	}

	if (x == u)
	{
		int d1 = (y == v);
		int d2 = (y+1 == v);
		int d3 = (y-1 == v);

		if (d1 || d2 || d3)
		{
			long double F1 = (long double)d1;
			long double F2 = (long double)d2;
			long double F3 = (long double)d3;

			long double xUp = -((m_hMob[idx]+m_hMob[upIndex])/(m_Dh[idx]+m_Dh[upIndex])) * ( (m_potential[upIndex]+m_extraPotentialP[upIndex]) - (m_potential[idx]+m_extraPotentialP[idx]) );
			long double xDown = -((m_hMob[idx]+m_hMob[downIndex])/(m_Dh[idx]+m_Dh[downIndex])) * ( (m_potential[idx]+m_extraPotentialP[idx]) - (m_potential[downIndex]+m_extraPotentialP[downIndex]) );

			sum -= m_DLy2*(
				0.5*(m_Dh[idx]+m_Dh[upIndex])
				   *(F1*calcDerivJd1(xUp, m_p[idx], m_p[upIndex])
			            +F2*calcDerivJd2(xUp, m_p[idx], m_p[upIndex]) )
			       -0.5*(m_Dh[idx]+m_Dh[downIndex])
			           *(F3*calcDerivJd1(xDown, m_p[downIndex], m_p[idx])
			            +F1*calcDerivJd2(xDown, m_p[downIndex], m_p[idx]) )
				);
		}

	}

	if (m_useLog)
	{
		if (sum != 0)
		{
			int idx2 = u+v*m_width;

			sum *= (m_p[idx2]/m_p[idx]);
			sum /= m_diffScale;
		}

		if (x == u && y == v)
			sum -= getFP(x, y)/m_kTev;  // getFP has already been divided by diffScale
	}
	else
		sum = sum/m_densScale*m_kTev/m_diffScale;

	return sum;

}

long double Simulation2DNR2::getNetGen(int x, int y)
{
	int idx = x + y*m_width;

	return m_generationRate[idx] - m_recombinationFactor[idx] * m_n[idx] * m_p[idx];
}

long double Simulation2DNR2::getNetGenDerivV(int x, int y, int u, int v)
{
	return 0;
}

long double Simulation2DNR2::getNetGenDerivN(int x, int y, int u, int v)
{
	if (! (x == u && y == v))
		return 0;

	int idx = x + y*m_width;

	return -m_recombinationFactor[idx]*m_p[idx];
}

long double Simulation2DNR2::getNetGenDerivP(int x, int y, int u, int v)
{
	if (! (x == u && y == v))
		return 0;

	int idx = x + y*m_width;

	return -m_recombinationFactor[idx]*m_n[idx];
}

void Simulation2DNR2::calcCurrentHorizontal(int x, int y)
{
	int idx = getIndex(x, y);
	int nextX = (x+1)%m_width; // for periodic boundary conditions
	int nextIdx = getIndex(nextX, y);

	long double De = (m_De[idx]+m_De[nextIdx])/2.0;
	long double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
	long double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
	long double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

	long double dPhiN = (m_potential[nextIdx]+m_extraPotentialN[nextIdx]) - (m_potential[idx]+m_extraPotentialN[idx]);
	long double dPhiP = (m_potential[nextIdx]+m_extraPotentialP[nextIdx]) - (m_potential[idx]+m_extraPotentialP[idx]);

	// Note: we won't divide by dx or dy here, we'll do this at another stage
	m_numCurTotEx[idx] = De * calcJ( dPhiN*eMob/De, m_n[idx], m_n[nextIdx]);
	m_numCurTotHx[idx] = Dh * calcJ( -dPhiP*hMob/Dh, m_p[idx], m_p[nextIdx]);
}

void Simulation2DNR2::calcCurrentVertical(int x, int y)
{
	int idx = getIndex(x, y);
	int nextIdx = getIndex(x, y+1);

	long double De = (m_De[idx]+m_De[nextIdx])/2.0;
	long double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
	long double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
	long double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

	long double dPhiN = (m_potential[nextIdx]+m_extraPotentialN[nextIdx]) - (m_potential[idx]+m_extraPotentialN[idx]);
	long double dPhiP = (m_potential[nextIdx]+m_extraPotentialP[nextIdx]) - (m_potential[idx]+m_extraPotentialP[idx]);

	// Note: we won't divide by dx or dy here, we'll do this at another stage
	m_numCurTotEy[idx] = De * calcJ( dPhiN*eMob/De, m_n[idx], m_n[nextIdx]);
	m_numCurTotHy[idx] = Dh * calcJ( -dPhiP*hMob/Dh, m_p[idx], m_p[nextIdx]);
}

void Simulation2DNR2::calcFactorsAndCurrents()
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
			calcCurrentHorizontal(x, y);
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
			calcCurrentVertical(x, y);
	}
}


long double Simulation2DNR2::calcW(long double x)
{
	long double val = 0;

	if (x > 0.01)
	{
		val = x/(1.0-std::exp(-x));
	}
	else if (x < -0.01)
	{
		long double f = std::exp(x);
		
		val = x*f/(f-1.0);
	}
	else
	{
		long double x2 = x*x;
		long double x4 = x2*x2;

		val = 1.0+0.5*x+1.0/12.0*x2-1.0/720.0*x4;
	}
	return val;
}

long double Simulation2DNR2::calcDerivW(long double x)
{
	long double val = 0;

	if (x > 0.01)
	{
		long double f = std::exp(-x);
		long double oneMinusF = 1.0-f;

		val = (1.0-x*f/oneMinusF)/oneMinusF;
	}
	else if (x < -0.01)
	{
		long double f = std::exp(x);
		long double fMinusOne = f-1.0;

		val = f/fMinusOne*(1.0-x/fMinusOne);
	}
	else
	{
		long double x2 = x*x;
		long double x3 = x2*x;

		val = 0.5+1.0/6.0*x-1.0/180.0*x3;
	}

	return val;
}

long double Simulation2DNR2::calcJ(long double x, long double d1, long double d2)
{
	long double Wx = calcW(x);

	return d1*Wx+d2*(x-Wx);
}

long double Simulation2DNR2::calcDerivJx(long double x, long double d1, long double d2)
{
	long double dWdx = calcDerivW(x);

	return d1*dWdx + d2*(1.0-dWdx);
}

long double Simulation2DNR2::calcDerivJd1(long double x, long double d1, long double d2)
{
	return calcW(x);
}

long double Simulation2DNR2::calcDerivJd2(long double x, long double d1, long double d2)
{
	return x - calcW(x);
}

void Simulation2DNR2::setPotentialDifference(double V)
{
	m_deltaPhi = V;

	for (int x = 0 ; x < m_width ; x++)
	{
		m_potential[x + 0 * m_width] = 0;
		m_potential[x + (m_height-1)*m_width] = m_deltaPhi;
	}
}


Simulation2DNR2::Search::Search(Simulation2DNR2 *pInstance)
{
	m_pSim = pInstance;
	m_rows.resize(16);
	m_rowIdx = 0;

	for (int i = 0 ; i < m_rows.size() ; i++)
		m_rows[i] = -1;
}

Simulation2DNR2::Search::~Search()
{
}

int Simulation2DNR2::Search::getMaxNonZeroRows(int col)
{
	// total dimension = m_numVariablePixels*3;
	// max total nonzero is 5*m_numVariablePixels*9 = 5*(m_numVariablePixels*3)*3;
	// so per col: max 5*3
	return 15;
}

void Simulation2DNR2::Search::startColumn(int col)
{
//	std::cout << "Starting col " << col << std::endl;

	m_rowIdx = 0;

	/*
	int relPos1[5][2] = { { 0, -1} , {-1, 0} , { 0, 0} , { 1, 0} , { 0, 1} };
	int relPos2[5][2] = { { 0, -1} , { 0, 0} , { 1, 0} , { -1, 0}, { 0, 1} };
	int relPos3[5][2] = { { 0, -1} , { 1, 0} , { -1, 0} , { 0, 0}, { 0, 1} };
	*/
	int relPos1[] = { 0, -1 , -1, 0 ,  0, 0 ,  1, 0 ,  0, 1 };
	int relPos2[] = { 0, -1 ,  0, 0 ,  1, 0 ,  -1, 0,  0, 1 };
	int relPos3[] = { 0, -1 ,  1, 0 ,  -1, 0 ,  0, 0,  0, 1 };

	int varType, u, v;
	int w = m_pSim->m_width;
	int h = m_pSim->m_height;

	getVariableCoordinates(col, varType, u, v);

	int *relPos = relPos1;

	if (u == 0)
		relPos = relPos2;
	else if (u == w-1)
		relPos = relPos3;

	int pos = 0;
	for (int t = 0 ; t < 3 ; t++) // function type
	{
		for (int k = 0 ; k < 5 ; k++)
		{
			int x = (u + relPos[k*2+0] + w)%w;
			int y = v + relPos[k*2+1];

			if (y > 0 && y < h-1)
			{
				m_rows[pos] = getFunctionIndex(t, x, y);

	//			std::cout << "row " << m_rows[pos] << std::endl;

				pos++;
			}
		}
	}
	m_rows[pos] = -1;
}

int Simulation2DNR2::Search::getNextNonZeroRow()
{
	int val = m_rows[m_rowIdx];
	m_rowIdx++;
	return val;
}

double Simulation2DNR2::Search::getFunctionValue(int Findex)
{
	int Ftype, x, y;

	getFunctionCoordinates(Findex, Ftype, x, y);

	return m_pSim->getFValue(Ftype, x, y);
}

double Simulation2DNR2::Search::getDerivative(int Findex, int varIndex)
{
	int Ftype, x, y, vtype, u, v;

	getVariableCoordinates(varIndex, vtype, u, v);
	getFunctionCoordinates(Findex, Ftype, x, y);

	return m_pSim->getFJacValue(Ftype, x, y, vtype, u, v);
}

void Simulation2DNR2::Search::getVariableCoordinates(int idx, int &t, int &u, int &v)
{
	int num = m_pSim->m_numVariablePixels;
	int w = m_pSim->m_width;
	int mod = idx%num;

	t = idx/num;
	u = mod%w;
	v = mod/w + 1;
}

void Simulation2DNR2::Search::getFunctionCoordinates(int idx, int &t, int &x, int &y)
{
	int num = m_pSim->m_numVariablePixels;
	int w = m_pSim->m_width;
	int mod = idx%num;

	t = idx/num;
	x = mod%w;
	y = mod/w + 1;
}

int Simulation2DNR2::Search::getFunctionIndex(int t, int x, int y)
{
	return t*(m_pSim->m_numVariablePixels) + (y-1)*(m_pSim->m_width) + x;
}

void Simulation2DNR2::dummyGetMatrix()
{
	int numVars = m_numVariablePixels*3;
	std::vector<double> matrix(numVars*numVars);
	std::vector<double> oldVals(numVars);

	for (int x = 0 ; x < numVars ; x++)
	{
		int varType = x/m_numVariablePixels;
		int varIdx = x%m_numVariablePixels;
		int varX = varIdx%m_width;
		int varY = varIdx/m_width+1;

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			oldVals[y] = getFValue(fType, fX, fY);
		}

		double diff = 0.00000000001;

		int prevX = (varX-1+m_width)%m_width;
		int prevY = varY - 1;

		double backup = 0;

		if (varType == 0)
		{
			backup = m_potential[varX+varY*m_width];
			m_potential[varX+varY*m_width] += diff;
		}
		else if (varType == 1)
		{
			backup = m_n[varX+varY*m_width];
			if (m_useLog)
				m_n[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_n[varX+varY*m_width];
				m_n[varX+varY*m_width] += diff;
			}
		}
		else if (varType == 2)
		{
			backup = m_p[varX+varY*m_width];
			if (m_useLog)
				m_p[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_p[varX+varY*m_width];
				m_p[varX+varY*m_width] += diff;
			}
		}
		else
		{
			std::cerr << "Invalid varType " << varType << std::endl;
		}

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			double newVal = getFValue(fType, fX, fY);
			double fDiff = newVal-oldVals[y];

			matrix[x+y*numVars] = fDiff/diff;
		}

		if (varType == 0)
			m_potential[varX+varY*m_width] = backup;
		else if (varType == 1)
			m_n[varX+varY*m_width] = backup;
		else if (varType == 2)
			m_p[varX+varY*m_width] = backup;

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);
	}

	std::vector<double> nrMatrix(numVars*numVars);

	m_pNRSearch->getMatrix(nrMatrix);

	for (int y = 0 ; y < numVars ; y++)
	{
		for (int x = 0 ; x < numVars ; x++)
		{
			int varType = x/m_numVariablePixels;
			int varIdx = x%m_numVariablePixels;
			int varX = varIdx%m_width;
			int varY = varIdx/m_width+1;

			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			int idx = x+y*numVars;
			if (! (nrMatrix[idx] == 0 && matrix[idx] == 0))
			{
				if (fType == 0)
					std::cout << "FV(";
				else if (fType == 1)
					std::cout << "FN(";
				else if (fType == 2)
					std::cout << "FP(";
				std::cout << fX << "," << fY << ")/";

				if (varType == 0)
					std::cout << "V(";
				else if (varType == 1)
					std::cout << "N(";
				else if (varType == 2)
					std::cout << "P(";

				std::cout << varX << "," << varY << ") = \t";
				std::cout << nrMatrix[idx]-matrix[idx] << "\t" << nrMatrix[idx] << "\t" << matrix[idx] << std::endl;
			}
		}
		std::cout << std::endl;
	}
	std::cout << std::endl;
}




Simulation2DNR2::SearchV::SearchV(Simulation2DNR2 *pInstance)
{
	m_pSim = pInstance;
	m_rows.resize(16);
	m_rowIdx = 0;

	for (int i = 0 ; i < m_rows.size() ; i++)
		m_rows[i] = -1;
}

Simulation2DNR2::SearchV::~SearchV()
{
}

int Simulation2DNR2::SearchV::getMaxNonZeroRows(int col)
{
	// total dimension = m_numVariablePixels*3;
	// max total nonzero is 5*m_numVariablePixels*9 = 5*(m_numVariablePixels*3)*3;
	// so per col: max 5*3
	return 15;
}

void Simulation2DNR2::SearchV::startColumn(int col)
{
//	std::cout << "Starting col " << col << std::endl;

	m_rowIdx = 0;

	/*
	int relPos1[5][2] = { { 0, -1} , {-1, 0} , { 0, 0} , { 1, 0} , { 0, 1} };
	int relPos2[5][2] = { { 0, -1} , { 0, 0} , { 1, 0} , { -1, 0}, { 0, 1} };
	int relPos3[5][2] = { { 0, -1} , { 1, 0} , { -1, 0} , { 0, 0}, { 0, 1} };
	*/
	int relPos1[] = { 0, -1 , -1, 0 ,  0, 0 ,  1, 0 ,  0, 1 };
	int relPos2[] = { 0, -1 ,  0, 0 ,  1, 0 ,  -1, 0,  0, 1 };
	int relPos3[] = { 0, -1 ,  1, 0 ,  -1, 0 ,  0, 0,  0, 1 };

	int u, v;
	int w = m_pSim->m_width;
	int h = m_pSim->m_height;

	getVariableCoordinates(col, u, v);

	int *relPos = relPos1;

	if (u == 0)
		relPos = relPos2;
	else if (u == w-1)
		relPos = relPos3;

	int pos = 0;
	for (int k = 0 ; k < 5 ; k++)
	{
		int x = (u + relPos[k*2+0] + w)%w;
		int y = v + relPos[k*2+1];

		if (y > 0 && y < h-1)
		{
			m_rows[pos] = getFunctionIndex(x, y);

//			std::cout << "row " << m_rows[pos] << std::endl;

			pos++;
		}
	}
	m_rows[pos] = -1;
}

int Simulation2DNR2::SearchV::getNextNonZeroRow()
{
	int val = m_rows[m_rowIdx];
	m_rowIdx++;
	return val;
}

double Simulation2DNR2::SearchV::getFunctionValue(int Findex)
{
	int x, y;

	getFunctionCoordinates(Findex, x, y);
	return m_pSim->getFValue(0, x, y);
}

double Simulation2DNR2::SearchV::getDerivative(int Findex, int varIndex)
{
	int x, y, u, v;

	getVariableCoordinates(varIndex, u, v);
	getFunctionCoordinates(Findex, x, y);

	return m_pSim->getFJacValue(0, x, y, 0, u, v);
}

void Simulation2DNR2::SearchV::getVariableCoordinates(int idx, int &u, int &v)
{
	int w = m_pSim->m_width;

	u = idx%w;
	v = idx/w + 1;
}

void Simulation2DNR2::SearchV::getFunctionCoordinates(int idx, int &x, int &y)
{
	int w = m_pSim->m_width;

	x = idx%w;
	y = idx/w + 1;
}

int Simulation2DNR2::SearchV::getFunctionIndex(int x, int y)
{
	return (y-1)*(m_pSim->m_width) + x;
}

void Simulation2DNR2::dummyGetMatrixV()
{
	int numVars = m_numVariablePixels;
	std::vector<double> matrix(numVars*numVars);
	std::vector<double> oldVals(numVars);

	for (int x = 0 ; x < numVars ; x++)
	{
		int varType = x/m_numVariablePixels;
		int varIdx = x%m_numVariablePixels;
		int varX = varIdx%m_width;
		int varY = varIdx/m_width+1;

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			oldVals[y] = getFValue(fType, fX, fY);
		}

		double diff = 0.00000000001;

		int prevX = (varX-1+m_width)%m_width;
		int prevY = varY - 1;

		double backup = 0;

		if (varType == 0)
		{
			backup = m_potential[varX+varY*m_width];
			m_potential[varX+varY*m_width] += diff;
		}
		else if (varType == 1)
		{
			backup = m_n[varX+varY*m_width];
			if (m_useLog)
				m_n[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_n[varX+varY*m_width];
				m_n[varX+varY*m_width] += diff;
			}
		}
		else if (varType == 2)
		{
			backup = m_p[varX+varY*m_width];
			if (m_useLog)
				m_p[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_p[varX+varY*m_width];
				m_p[varX+varY*m_width] += diff;
			}
		}
		else
		{
			std::cerr << "Invalid varType " << varType << std::endl;
		}

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			double newVal = getFValue(fType, fX, fY);
			double fDiff = newVal-oldVals[y];

			matrix[x+y*numVars] = fDiff/diff;
		}

		if (varType == 0)
			m_potential[varX+varY*m_width] = backup;
		else if (varType == 1)
			m_n[varX+varY*m_width] = backup;
		else if (varType == 2)
			m_p[varX+varY*m_width] = backup;

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);
	}

	std::vector<double> nrMatrix(numVars*numVars);

	m_pNRSearchV->getMatrix(nrMatrix);

	for (int y = 0 ; y < numVars ; y++)
	{
		for (int x = 0 ; x < numVars ; x++)
		{
			int varType = x/m_numVariablePixels;
			int varIdx = x%m_numVariablePixels;
			int varX = varIdx%m_width;
			int varY = varIdx/m_width+1;

			int fType = y/m_numVariablePixels;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			int idx = x+y*numVars;
			if (! (nrMatrix[idx] == 0 && matrix[idx] == 0))
			{
				if (fType == 0)
					std::cout << "FV(";
				else if (fType == 1)
					std::cout << "FN(";
				else if (fType == 2)
					std::cout << "FP(";
				std::cout << fX << "," << fY << ")/";

				if (varType == 0)
					std::cout << "V(";
				else if (varType == 1)
					std::cout << "N(";
				else if (varType == 2)
					std::cout << "P(";

				std::cout << varX << "," << varY << ") = \t";
				std::cout << nrMatrix[idx]-matrix[idx] << "\t" << nrMatrix[idx] << "\t" << matrix[idx] << std::endl;
			}
		}
		std::cout << std::endl;
	}
	std::cout << std::endl;
}





Simulation2DNR2::SearchNP::SearchNP(Simulation2DNR2 *pInstance)
{
	m_pSim = pInstance;
	m_rows.resize(16);
	m_rowIdx = 0;

	for (int i = 0 ; i < m_rows.size() ; i++)
		m_rows[i] = -1;
}

Simulation2DNR2::SearchNP::~SearchNP()
{
}

int Simulation2DNR2::SearchNP::getMaxNonZeroRows(int col)
{
	// total dimension = m_numVariablePixels*3;
	// max total nonzero is 5*m_numVariablePixels*9 = 5*(m_numVariablePixels*3)*3;
	// so per col: max 5*3
	return 15;
}

void Simulation2DNR2::SearchNP::startColumn(int col)
{
//	std::cout << "Starting col " << col << std::endl;

	m_rowIdx = 0;

	/*
	int relPos1[5][2] = { { 0, -1} , {-1, 0} , { 0, 0} , { 1, 0} , { 0, 1} };
	int relPos2[5][2] = { { 0, -1} , { 0, 0} , { 1, 0} , { -1, 0}, { 0, 1} };
	int relPos3[5][2] = { { 0, -1} , { 1, 0} , { -1, 0} , { 0, 0}, { 0, 1} };
	*/
	int relPos1[] = { 0, -1 , -1, 0 ,  0, 0 ,  1, 0 ,  0, 1 };
	int relPos2[] = { 0, -1 ,  0, 0 ,  1, 0 ,  -1, 0,  0, 1 };
	int relPos3[] = { 0, -1 ,  1, 0 ,  -1, 0 ,  0, 0,  0, 1 };

	int varType, u, v;
	int w = m_pSim->m_width;
	int h = m_pSim->m_height;

	getVariableCoordinates(col, varType, u, v);

	int *relPos = relPos1;

	if (u == 0)
		relPos = relPos2;
	else if (u == w-1)
		relPos = relPos3;

	int pos = 0;
	for (int t = 0 ; t < 2 ; t++) // function type
	{
		for (int k = 0 ; k < 5 ; k++)
		{
			int x = (u + relPos[k*2+0] + w)%w;
			int y = v + relPos[k*2+1];

			if (y > 0 && y < h-1)
			{
				m_rows[pos] = getFunctionIndex(t, x, y);

	//			std::cout << "row " << m_rows[pos] << std::endl;

				pos++;
			}
		}
	}
	m_rows[pos] = -1;
}

int Simulation2DNR2::SearchNP::getNextNonZeroRow()
{
	int val = m_rows[m_rowIdx];
	m_rowIdx++;
	return val;
}

double Simulation2DNR2::SearchNP::getFunctionValue(int Findex)
{
	int Ftype, x, y;

	getFunctionCoordinates(Findex, Ftype, x, y);

	return m_pSim->getFValue(Ftype+1, x, y);
}

double Simulation2DNR2::SearchNP::getDerivative(int Findex, int varIndex)
{
	int Ftype, x, y, vtype, u, v;

	getVariableCoordinates(varIndex, vtype, u, v);
	getFunctionCoordinates(Findex, Ftype, x, y);

	return m_pSim->getFJacValue(Ftype+1, x, y, vtype+1, u, v);
}

void Simulation2DNR2::SearchNP::getVariableCoordinates(int idx, int &t, int &u, int &v)
{
	int num = m_pSim->m_numVariablePixels;
	int w = m_pSim->m_width;
	int mod = idx%num;

	t = idx/num;
	u = mod%w;
	v = mod/w + 1;
}

void Simulation2DNR2::SearchNP::getFunctionCoordinates(int idx, int &t, int &x, int &y)
{
	int num = m_pSim->m_numVariablePixels;
	int w = m_pSim->m_width;
	int mod = idx%num;

	t = idx/num;
	x = mod%w;
	y = mod/w + 1;
}

int Simulation2DNR2::SearchNP::getFunctionIndex(int t, int x, int y)
{
	return t*(m_pSim->m_numVariablePixels) + (y-1)*(m_pSim->m_width) + x;
}

void Simulation2DNR2::dummyGetMatrixNP()
{
	int numVars = m_numVariablePixels*2;
	std::vector<double> matrix(numVars*numVars);
	std::vector<double> oldVals(numVars);

	for (int x = 0 ; x < numVars ; x++)
	{
		int varType = x/m_numVariablePixels + 1;
		int varIdx = x%m_numVariablePixels;
		int varX = varIdx%m_width;
		int varY = varIdx/m_width+1;

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels + 1;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			oldVals[y] = getFValue(fType, fX, fY);
		}

		double diff = 0.00000000001;

		int prevX = (varX-1+m_width)%m_width;
		int prevY = varY - 1;

		double backup = 0;

		if (varType == 0)
		{
			backup = m_potential[varX+varY*m_width];
			m_potential[varX+varY*m_width] += diff;
		}
		else if (varType == 1)
		{
			backup = m_n[varX+varY*m_width];
			if (m_useLog)
				m_n[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_n[varX+varY*m_width];
				m_n[varX+varY*m_width] += diff;
			}
		}
		else if (varType == 2)
		{
			backup = m_p[varX+varY*m_width];
			if (m_useLog)
				m_p[varX+varY*m_width] *= std::exp(diff/m_kTev);
			else
			{
				diff *= m_p[varX+varY*m_width];
				m_p[varX+varY*m_width] += diff;
			}
		}
		else
		{
			std::cerr << "Invalid varType " << varType << std::endl;
		}

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);

		for (int y = 0 ; y < numVars ; y++)
		{
			int fType = y/m_numVariablePixels + 1;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			double newVal = getFValue(fType, fX, fY);
			double fDiff = newVal-oldVals[y];

			matrix[x+y*numVars] = fDiff/diff;
		}

		if (varType == 0)
			m_potential[varX+varY*m_width] = backup;
		else if (varType == 1)
			m_n[varX+varY*m_width] = backup;
		else if (varType == 2)
			m_p[varX+varY*m_width] = backup;

		calcCurrentHorizontal(varX, varY);
		calcCurrentHorizontal(prevX, varY);
		calcCurrentVertical(varX, varY);
		calcCurrentVertical(varX, prevY);
	}

	std::vector<double> nrMatrix(numVars*numVars);

	m_pNRSearchNP->getMatrix(nrMatrix);

	for (int y = 0 ; y < numVars ; y++)
	{
		for (int x = 0 ; x < numVars ; x++)
		{
			int varType = x/m_numVariablePixels + 1;
			int varIdx = x%m_numVariablePixels;
			int varX = varIdx%m_width;
			int varY = varIdx/m_width+1;

			int fType = y/m_numVariablePixels + 1;
			int fIdx = y%m_numVariablePixels;
			int fX = fIdx%m_width;
			int fY = fIdx/m_width+1;

			int idx = x+y*numVars;
			if (! (nrMatrix[idx] == 0 && matrix[idx] == 0))
			{
				if (fType == 0)
					std::cout << "FV(";
				else if (fType == 1)
					std::cout << "FN(";
				else if (fType == 2)
					std::cout << "FP(";
				std::cout << fX << "," << fY << ")/";

				if (varType == 0)
					std::cout << "V(";
				else if (varType == 1)
					std::cout << "N(";
				else if (varType == 2)
					std::cout << "P(";

				std::cout << varX << "," << varY << ") = \t";
				std::cout << nrMatrix[idx]-matrix[idx] << "\t" << nrMatrix[idx] << "\t" << matrix[idx] << std::endl;
			}
		}
		std::cout << std::endl;
	}
	std::cout << std::endl;
}

bool Simulation2DNR2::setState(const SimulationState &state, std::vector<std::string> &warnings)
{
	if (!m_init)
	{
		setErrorString("Simulation not initialized");
		return false;
	}
	if (!state.isInitialized())
	{
		setErrorString("State to be used is not initialized");
		return false;
	}
	if (state.getDimensions() != 2)
	{
		setErrorString("State is not two-dimensional");
		return false;
	}

	if (state.getNumberOfXPixels() != m_width || state.getNumberOfYPixels() != m_height)
	{
		setErrorString("State does not have same dimensions as simulation");
		return false;
	}

	std::vector<bool> m_input(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_output(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_optional(SIMSTATE_GRIDPROP_MAX);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		m_input[i] = false;
		m_output[i] = false;
		m_optional[i] = false;
	}

	m_input[SIMSTATE_GRIDPROP_N] = true;
	m_input[SIMSTATE_GRIDPROP_P] = true;
	m_input[SIMSTATE_GRIDPROP_V] = true;
	m_input[SIMSTATE_GRIDPROP_DN] = true;
	m_input[SIMSTATE_GRIDPROP_DP] = true;
	m_input[SIMSTATE_GRIDPROP_NMOB] = true;
	m_input[SIMSTATE_GRIDPROP_PMOB] = true;
	m_input[SIMSTATE_GRIDPROP_EPSREL] = true;

	m_output[SIMSTATE_GRIDPROP_N] = true;
	m_output[SIMSTATE_GRIDPROP_P] = true;
	m_output[SIMSTATE_GRIDPROP_JNX] = true;
	m_output[SIMSTATE_GRIDPROP_JPX] = true;
	m_output[SIMSTATE_GRIDPROP_JNY] = true;
	m_output[SIMSTATE_GRIDPROP_JPY] = true;
	m_output[SIMSTATE_GRIDPROP_R] = true;

	m_optional[SIMSTATE_GRIDPROP_BG] = true;
	m_optional[SIMSTATE_GRIDPROP_G] = true;
	m_optional[SIMSTATE_GRIDPROP_RF] = true;
	m_optional[SIMSTATE_GRIDPROP_VNEXTRA] = true;
	m_optional[SIMSTATE_GRIDPROP_VPEXTRA] = true;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (m_input[i])
		{
			if (!state.isGridPropertySet(i))
			{
				setErrorString("Grid property '" + state.getGridPropertyName(i) + "' is required, but has not been set");
				return false;
			}
		}
		else
		{
			if (state.isGridPropertySet(i))
			{
				if (!m_output[i] && !m_optional[i])
					warnings.push_back("Property '" + state.getGridPropertyName(i) + "' is set but not used");
			}
		}
	}

	state.getGridProperty(SIMSTATE_GRIDPROP_N, m_n);
	state.getGridProperty(SIMSTATE_GRIDPROP_P, m_p);
	state.getGridProperty(SIMSTATE_GRIDPROP_BG, m_background);
	state.getGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate);
	state.getGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor);
	state.getGridProperty(SIMSTATE_GRIDPROP_DN, m_De);
	state.getGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh);
	state.getGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob);
	state.getGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob);
	state.getGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels);

	m_epsRelChanged = true;

	state.getGridProperty(SIMSTATE_GRIDPROP_V, m_potential);
	state.getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, m_extraPotentialN);
	state.getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, m_extraPotentialP);

	double vDiff;

	if (!state.getDoubleProperty(SIMSTATE_PROP_VDIFF, vDiff))
	{
		setErrorString("Potential difference has not been set");
		return false;
	}

	m_deltaPhi = vDiff;

	if (state.getRecombinationModel() == SimulationState::Braun)
	{
		setErrorString("Braun recombination model selected, but not supported in this simulation type");
		return false;
	}
	else
	{
		double a, kf;

		if (state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
			warnings.push_back("'Pair distance' parameter is set, but is only used in the braun model");
		if (state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
			warnings.push_back("'Dissociation rate' parameter 'kf' is set, but is only used in the braun model");
	}

	//FILE *f = fopen("log.txt","wt");
	//writePlotData(f);
	//fclose(f);

	return true;
}

bool Simulation2DNR2::storeState(SimulationState &state) const
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	if (state.getNumberOfXPixels() != m_width || state.getNumberOfYPixels() != m_height)
	{
		setErrorString("State does not have same dimensions as simulation");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		state.clearDoubleProperty(i);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		state.clearGridProperty(i);

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_N, m_n) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_P, m_p) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_V, m_potential))
	{
		setErrorString("Unable to store n,p or V");
		return false;
	}

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_BG, m_background) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DN, m_De) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, m_extraPotentialN) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, m_extraPotentialP) )
	{
		setErrorString("Unable to store fixed simulation items");
		return false;
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			double jnx = m_numCurTotEx[x+y*m_width]/m_pixelWidth2;
			double jpx = m_numCurTotHx[x+y*m_width]/m_pixelWidth2;
			double jny = m_numCurTotEy[x+y*m_width]/m_pixelHeight2;
			double jpy = m_numCurTotHy[x+y*m_width]/m_pixelHeight2;

			if (!state.setGridProperty(SIMSTATE_GRIDPROP_JNX, x, y, jnx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPX, x, y, jpx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JNY, x, y, jny) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPY, x, y, jpy))
			{
				setErrorString("Unable to store the number current");
				return false;
			}
		}
	}

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;
			long double R = m_recombinationFactor[idx] * (m_n[idx] * m_p[idx]);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, x, y, (double)R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
		}
	}

	state.setRecombinationModel(SimulationState::Simple);	

	if (!state.setDoubleProperty(SIMSTATE_PROP_VDIFF, m_deltaPhi))
	{
		setErrorString("Couldn't store potential difference");
		return false;
	}

	if (!state.setDoubleProperty(SIMSTATE_PROP_TEMP, m_T))
	{
		setErrorString("Couldn't store temperature");
		return false;
	}

	return true;
}

