#include "simulation2ddoublerel.h"
#include "simulationstate.h"
#include <serut/fileserializer.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <iostream>
#include <cmath>

using namespace serut;

//#define CHECK(value, idx) checkArray(value, idx%m_width, idx/m_width, #value, __LINE__)
#define CHECK(value, idx) 

#ifndef WIN32
inline void checkArray(double val, int x, int y, const char *pName, int lineNumber)
{
	if (std::isnan(val))
	{
		if (x >= 0 && y >= 0)
			std::cerr << "Detected NaN " << pName << " at location (" << x << ", " << y << ") on line " << lineNumber << std::endl;
		else
			std::cerr << "Detected NaN " << pName << " on line " << lineNumber << std::endl;
		exit(-1);
	}
}
#endif // !WIN32

#define LIM 1e-10

inline double Simulation2DDoubleRel::getSGCurrent(double v, double D, double delta, double n1, double n2)
{
	double vDelta = v*delta;
	double x = vDelta/D;
	double j = 0;

	if (x < -LIM)
	{
		double factor = std::exp(x);
		double factorMinusOne = factor - 1.0f;
		double n1factor = n1*factor;
		double n1factorMinusN2 = n1factor - n2;
		double n1factorMinusN2OverFactorMinusOne = n1factorMinusN2/factorMinusOne;

		j = v*n1factorMinusN2OverFactorMinusOne;
	}
	else if (x > LIM)
	{
		double factor = std::exp(-x);
		double n2factor = n2*factor;
		double oneMinusFactor = 1.0f-factor;
		double n1MinusN2Factor = n1-n2factor;
		double n1MinusN2FactorOverOneMinusFactor = n1MinusN2Factor/oneMinusFactor;

		j = v*n1MinusN2FactorOverOneMinusFactor;

	}
	else
	{
		x /= 2.0f;

		double twoD = 2.0f*D;
		double factor = twoD/delta;
		double n1MinusN2 = n1-n2;
		double n1PlusN2 = n1+n2;
		double term1 = 0.5f*n1MinusN2;
		double term2Part = 0.5f*n1PlusN2;
		double term2 = term2Part*x;
		double xSquared = x*x;
		double term3Part = n1MinusN2*xSquared;
		double term3 = term3Part/6.0f;

		double seriesPart = term1 + term2;
		double series = seriesPart + term3;

		j = factor*series;
	}

	return j;
}

Simulation2DDoubleRel::Simulation2DDoubleRel()
{
	m_init = false;

	m_width = 0;
	m_height = 0;
	m_totalPixels = 0;
	m_pixelWidth = 0;
	m_pixelHeight = 0;
	m_pixelFrac = 1;
	m_pixelFracInv = 1;
	m_deltaPhi = 0;

	m_phiMethodThreshold = 100;
	m_phiBackupBetterCounter = 0;
	m_phiMethodAdditionalCheckCount = 1000;
}

Simulation2DDoubleRel::~Simulation2DDoubleRel()
{
}

bool Simulation2DDoubleRel::init(int lowestWidth, int lowestHeight, double realWidth, double realHeight, int scaleSteps, int *pNumX, int *pNumY)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (lowestWidth < 2 || lowestHeight < 3)
	{
		setErrorString("Invalid grid dimensions");
		return false;
	}

	if (scaleSteps < 1)
	{
		setErrorString("Invalid number of multigrid steps");
		return false;
	}

	// Calculate the number of pixels on the highest resolution grid.

	int multiplier = 1 << (scaleSteps-1); // bitshift to accomplish 2^(scaleSteps-1)
	int xPixels = lowestWidth*multiplier;
	int yPixels = lowestHeight*multiplier;

	*pNumX = xPixels;
	*pNumY = yPixels;

	m_multiScaleSteps = scaleSteps;

	m_width = xPixels;
	m_height = yPixels;
	m_totalPixels = m_width*m_height;

	m_pixelWidth = realWidth/(double)xPixels;
	m_pixelHeight = realHeight/(double)(yPixels-1);
	m_pixelFrac = m_pixelWidth/m_pixelHeight;
	m_pixelFracInv = m_pixelHeight/m_pixelWidth;

	// Allocate enough memory for various arrays
	resizeArrays();

	m_deltaPhi = 0;

	m_init = true;
	m_initPotential = true;
	m_epsChanged = true;
	m_extraPotentialChanged = true;

	m_npDensFactor = 1.0*CONST_EPSILON0/(CHARGE_ELECTRON*m_pixelWidth*m_pixelWidth);
	m_chargeMultiplier = (CHARGE_ELECTRON*m_pixelWidth*m_pixelWidth/CONST_EPSILON0) * m_npDensFactor;
	m_timeFactor = 1.0e-10;

	//std::cout << "m_npDensFactor = " << m_npDensFactor << std::endl;
	//std::cout << "m_chargeMultiplier = " << m_chargeMultiplier << std::endl;
	//std::cout << "m_timeFactor = " << m_timeFactor << std::endl;
	//std::cout << "m_pixelWidth = " << m_pixelWidth << std::endl;

	return true;
}

#define SIMULATION2DDOUBLEREL_STARTID	0x524d4953
#define SIMULATION2DDOUBLEREL_VERSION	1
#define SIMULATION2DDOUBLEREL_ENDID	0x444e4f43

bool Simulation2DDoubleRel::write(SerializationInterface &s) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninitialized simulation");
		return false;
	}

	s.writeInt32(SIMULATION2DDOUBLEREL_STARTID);
	s.writeInt32(SIMULATION2DDOUBLEREL_VERSION);
	s.writeInt32(m_width);
	s.writeInt32(m_height);
	s.writeInt32(m_multiScaleSteps);
	s.writeDouble(m_pixelWidth);
	s.writeDouble(m_pixelHeight);

	s.writeDouble(m_npDensFactor);
	s.writeDouble(m_chargeMultiplier);
	s.writeDouble(m_timeFactor);

	s.writeDoubles(m_n);
	s.writeDoubles(m_p);
	s.writeDoubles(m_background);

	s.writeDoubles(m_generationRate);
	s.writeDoubles(m_recombinationFactor);
	s.writeDoubles(m_De);
	s.writeDoubles(m_Dh);
	s.writeDoubles(m_eMob);
	s.writeDoubles(m_hMob);

	s.writeDoubles(m_epsRels[0]);
	s.writeDoubles(m_backupPotential);
	s.writeDouble(m_deltaPhi);

	s.writeDoubles(m_potential[0]);
	s.writeDoubles(m_potential[m_multiScaleSteps-1]);
	s.writeDoubles(m_numCurTotEx);
	s.writeDoubles(m_numCurTotEy);
	s.writeDoubles(m_numCurTotHx);
	s.writeDoubles(m_numCurTotHy);

	s.writeDoubles(m_extraPotentialN);
	s.writeDoubles(m_extraPotentialP);

	int32_t flag = (m_initPotential)?1:0;

	s.writeInt32(flag);

	if (!s.writeInt32(SIMULATION2DDOUBLEREL_ENDID))
	{
		setErrorString(std::string("Error writing to stream: ") + s.getErrorString());
		return false;
	}

	return true;
}

bool Simulation2DDoubleRel::read(SerializationInterface &s)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	int32_t tmp;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file start identifier: ") + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DDOUBLEREL_STARTID)
	{
		setErrorString("Read an invalid data file start identifier");
		return false;
	}
	if (!s.readInt32(&tmp))
	{
		setErrorString("Unable to read file format version: " + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DDOUBLEREL_VERSION)
	{
		setErrorString("Incompatible file format version");
		return false;
	}

	int32_t w, h, steps;
	double pixelWidth, pixelHeight;
	double npDens, chargeMult, timeFact;

	if (!(s.readInt32(&w) && s.readInt32(&h) && s.readInt32(&steps) && s.readDouble(&pixelWidth) && s.readDouble(&pixelHeight)
	    && s.readDouble(&npDens) && s.readDouble(&chargeMult) && s.readDouble(&timeFact) ) )
	{
		setErrorString("Error reading grid dimensions and base units");
		return false;
	}

	m_width = w;
	m_height = h;
	m_multiScaleSteps = steps;

	m_pixelWidth = pixelWidth;
	m_pixelHeight = pixelHeight;
	m_pixelFrac = m_pixelWidth/m_pixelHeight;
	m_pixelFracInv = m_pixelHeight/m_pixelWidth;

	m_npDensFactor = npDens;
	m_chargeMultiplier = chargeMult;
	m_timeFactor = timeFact;

	int multiplier = 1 << (m_multiScaleSteps-1);
	int lowestWidth = m_width/multiplier;
	int lowestHeight = m_height/multiplier;

	if (!(lowestWidth*multiplier == m_width && lowestHeight*multiplier == m_height))
	{
		setErrorString("Detected inconsistent dimension settings");
		return false;
	}

	m_totalPixels = m_width*m_height;
	
	resizeArrays();

	s.readDoubles(m_n);
	s.readDoubles(m_p);
	s.readDoubles(m_background);

	s.readDoubles(m_generationRate);
	s.readDoubles(m_recombinationFactor);
	s.readDoubles(m_De);
	s.readDoubles(m_Dh);
	s.readDoubles(m_eMob);
	s.readDoubles(m_hMob);

	s.readDoubles(m_epsRels[0]);
	s.readDoubles(m_backupPotential);
	s.readDouble(&m_deltaPhi);

	s.readDoubles(m_potential[0]);
	s.readDoubles(m_potential[m_multiScaleSteps-1]);
	s.readDoubles(m_numCurTotEx);
	s.readDoubles(m_numCurTotEy);
	s.readDoubles(m_numCurTotHx);
	s.readDoubles(m_numCurTotHy);

	s.readDoubles(m_extraPotentialN);
	s.readDoubles(m_extraPotentialP);

	int32_t flag = 0;

	s.readInt32(&flag);
	if (flag == 1)
		m_initPotential = true;
	else
		m_initPotential = false;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file end identifier: ") + s.getErrorString());
		return false;
	}

	if (tmp != SIMULATION2DDOUBLEREL_ENDID)
	{
		setErrorString("Read an invalid data file end identifier");
		return false;
	}

	m_init = true;
	m_epsChanged = true;
	m_extraPotentialChanged = true;

	m_phiBackupBetterCounter = 0;

	return true;
}

bool Simulation2DDoubleRel::save(const std::string &fileName) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninialized simulation");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::WriteOnly))
	{
		setErrorString(std::string("Couldn't save to file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return write(fSer);
}

bool Simulation2DDoubleRel::load(const std::string &fileName)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::ReadOnly))
	{
		setErrorString(std::string("Couldn't load file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return read(fSer);
}

bool Simulation2DDoubleRel::start(int steps, double dt, bool inverseMatrixSolver)
{
	if (inverseMatrixSolver)
	{
		setErrorString("Inverse matrix Poisson solver is not available for this type of simulation");
		return false;
	}

	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	double scaledDt = (double)(dt/m_timeFactor);
	
	initializePotentialFinder();

	prepareRelativeCalculation();	

	for (int N = 0 ; N < steps ; N++)
	{
		potentialFinder();

		calcNumberCurrents();

		updateDensities(scaledDt);
	}

	mergeRelativeResults();

	return true;
}

bool Simulation2DDoubleRel::start(int seconds, double dt, int &steps, bool inverseMatrixSolver)
{
	if (inverseMatrixSolver)
	{
		setErrorString("Inverse matrix Poisson solver is not available for this type of simulation");
		return false;
	}

	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	double scaledDt = (double)(dt/m_timeFactor);

	initializePotentialFinder();

	prepareRelativeCalculation();

	time_t startTime = time(0);
	time_t endTime = startTime;
	int count = 0;
	int maxSteps = steps;

	endTime += seconds;

	do
	{
		potentialFinder();

		calcNumberCurrents();

		updateDensities(scaledDt);

		count++;

	} while (time(0) <= endTime && count < maxSteps);

	steps = count;

	mergeRelativeResults();

	return true;
}

void Simulation2DDoubleRel::calcNumberCurrents()
{

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = getIndex(x, y);
			int nextX = (x+1)%m_width; // for periodic boundary conditions
			int nextIdx = getIndex(nextX, y);

			double De = (m_De[idx]+m_De[nextIdx])/2.0;
			double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
			double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
			double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

			double ve = -eMob*(m_electicFieldx[idx]+m_extraElecticFieldNx[idx]);
			double vh = +hMob*(m_electicFieldx[idx]+m_extraElecticFieldPx[idx]);

			CHECK(De, idx);
			CHECK(Dh, idx);
			CHECK(eMob, idx);
			CHECK(hMob, idx);
			CHECK(ve, idx);
			CHECK(vh, idx);

			// densities in number currents are in same units as m_n, m_p and m_background

			m_numCurTotEx[idx] = getSGCurrent(ve, De, 1.0, m_n[idx] + m_nRel[idx], m_n[nextIdx] + m_nRel[nextIdx]);
			m_numCurTotHx[idx] = getSGCurrent(vh, Dh, 1.0, m_p[idx] + m_pRel[idx], m_p[nextIdx] + m_pRel[nextIdx]);

			CHECK(m_numCurTotEx[idx], idx);
			CHECK(m_numCurTotHx[idx], idx);
		}
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{	
			int idx = getIndex(x, y);
			int nextIdx = getIndex(x, y+1);

			double De = (m_De[idx]+m_De[nextIdx])/2.0;
			double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
			double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
			double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

			double ve = -eMob*(m_electicFieldy[idx]+m_extraElecticFieldNy[idx]);
			double vh = +hMob*(m_electicFieldy[idx]+m_extraElecticFieldPy[idx]);

			CHECK(De, idx);
			CHECK(Dh, idx);
			CHECK(eMob, idx);
			CHECK(hMob, idx);
			CHECK(ve, idx);
			CHECK(vh, idx);

			// densities in number currents are in same units as m_n, m_p and m_background

			m_numCurTotEy[idx] = getSGCurrent(ve, De, m_pixelFracInv, m_n[idx]+m_nRel[idx], m_n[nextIdx] + m_nRel[nextIdx]);
			m_numCurTotHy[idx] = getSGCurrent(vh, Dh, m_pixelFracInv, m_p[idx]+m_pRel[idx], m_p[nextIdx] + m_pRel[nextIdx]);

			CHECK(m_numCurTotEy[idx], idx);
			CHECK(m_numCurTotHy[idx], idx);
		}
	}
}

// Based on the currents, update the densities
// NOTE: these are number currents, not charge currents

void Simulation2DDoubleRel::updateDensities(double scaledDt)
{
	// Calculate dn/dt and dp/dt first

	for (int x = 0 ; x < m_width ; x++)
	{
		int prevX = (x-1+m_width)%m_width; // For periodic boundary conditions

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			int prevY = y-1;
			int idx = getIndex(x, y);
			int leftIdx = getIndex(prevX, y);
			int belowIdx = getIndex(x, prevY);

			double JexCur = m_numCurTotEx[idx];
			double JexPrev = m_numCurTotEx[leftIdx];
			double JeyCur = m_numCurTotEy[idx];
			double JeyPrev = m_numCurTotEy[belowIdx];

			double JhxCur = m_numCurTotHx[idx];
			double JhxPrev = m_numCurTotHx[leftIdx];
			double JhyCur = m_numCurTotHy[idx];
			double JhyPrev = m_numCurTotHy[belowIdx];

			double Jexx = (JexCur - JexPrev);
			double Jeyy = (JeyCur - JeyPrev)*m_pixelFrac;
			double Jhxx = (JhxCur - JhxPrev);
			double Jhyy = (JhyCur - JhyPrev)*m_pixelFrac;

			double netGeneration = m_GR[idx] - m_rn[idx]*m_pRel[idx] - m_rp[idx]*m_nRel[idx] - m_recombinationFactor[idx]*m_nRel[idx]*m_pRel[idx];

			m_dndt[idx] = netGeneration - ( Jexx+Jeyy );
			m_dpdt[idx] = netGeneration - ( Jhxx+Jhyy );
		}
	}

	// Then, perform the actual update of the densities

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			int idx = getIndex(x, y);

			m_nRel[idx] += m_dndt[idx]*(double)scaledDt;
			m_pRel[idx] += m_dpdt[idx]*(double)scaledDt;
		}
	}
}

// Helper function to calculate the current in the X-direction at the left side, the right side
// and averaged over the entire grid

void Simulation2DDoubleRel::calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int y = 1 ; y < m_height-1 ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;

			double JexCur = m_numCurTotEx[idx];
			double JhxCur = m_numCurTotHx[idx];
			
			sum += -JexCur+JhxCur;
		}

		int idx = 0+y*m_width;

		double JexCur = m_numCurTotEx[idx];
		double JhxCur = m_numCurTotHx[idx];

		sum2 += -JexCur + JhxCur;

		idx = m_width-1+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum3 += -JexCur + JhxCur;

		idx = m_width/2+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum4 += -JexCur + JhxCur;
	}

	sum /= m_width*(m_height-2);
	sum2 /= m_height-2;
	sum3 /= m_height-2;
	sum4 /= m_height-2;

	double factor = m_npDensFactor*m_pixelWidth/m_timeFactor;

	sum *= factor;
	sum2 *= factor;
	sum3 *= factor;
	sum4 *= factor;

	leftAvg = sum2;
	rightAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

// Helper function to calculate the current in the Y-direction at the bottom side, the top side
// and averaged over the entire grid

void Simulation2DDoubleRel::calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{
			int idx = x+y*m_width;

			double JeyCur = m_numCurTotEy[idx];
			double JhyCur = m_numCurTotHy[idx];
			
			sum += -JeyCur+JhyCur;
		}

		int idx = x+0*m_width;

		double JeyCur = m_numCurTotEy[idx];
		double JhyCur = m_numCurTotHy[idx];

		sum2 += -JeyCur + JhyCur;

		idx = x+(m_height-2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum3 += -JeyCur + JhyCur;

		idx = x+(m_height/2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum4 += -JeyCur + JhyCur;
	}

	sum /= m_width*(m_height-1);
	sum2 /= m_width;
	sum3 /= m_width;
	sum4 /= m_width;

	double factor = m_npDensFactor*m_pixelWidth/m_timeFactor;

	sum *= factor;
	sum2 *= factor;
	sum3 *= factor;
	sum4 *= factor;

	bottomAvg = sum2;
	topAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

// 
// The rest of the file contains functions for calculating the potential. This is
// probably the most complex part of the calculation, mostly because a multigrid
// method is used.
//
// To calculate the potential, the following inputs are used:
//  - The relative permittivity at each grid point
//  - The electron density at each grid point
//  - The hole density at each grid point
//  - A fixed background charge density
//  - Potential at the top contact (bottom contact is assumed to be at 0 V

// This function initializes some parts of the routine which determines the potential
// (if necessary).

void Simulation2DDoubleRel::initializePotentialFinder()
{
	// Initialize the potential at the coarsest scale

	if (m_initPotential)
	{
		m_initPotential = false;

		for (int y = 0 ; y < m_height ; y++)
		{
			double V = ((double)y/(double)(m_height-1))*m_deltaPhi;

			for (int x = 0 ; x < m_width ; x++)
			{
				int idx = x+y*m_width;

				m_potential[0][idx] = V;
			}
		}
		
		m_phiBackupBetterCounter = 0;
	}
	
	// Build scaled arrays of permittivity at each scale

	if (m_epsChanged)
	{
		m_epsChanged = false;

		int w = m_width;
		int h = m_height;

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				for (int y = 0 ; y < h ; y += 2)
				{
					double epsSum = m_epsRels[i][x+y*w] + m_epsRels[i][x+1+y*w] + m_epsRels[i][x+(y+1)*w] + m_epsRels[i][x+1+(y+1)*w];

					m_epsRels[i+1][x/2+(y/2)*w2] = epsSum/4.0f; // average it
				}
			}

			w /= 2;
			h /= 2;
		}

		w = m_width;
		h = m_height;

		double frac2 = m_pixelFrac*m_pixelFrac;

		for (int i = 0 ; i < m_multiScaleSteps ; i++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				for (int x = 0 ; x < w ; x++)
				{
					// Initialize using periodic boundary conditions in both directions

					int xNext = (x+1)%w;
					int xPrev = (x-1+w)%w;
					int yNext = (y+1)%h;
					int yPrev = (y-1+h)%h;

					int index = x+y*w;
					int leftIndex = xPrev+y*w; 
					int rightIndex = xNext+y*w;
					int upIndex = x+yNext*w;
					int downIndex = x+yPrev*w;

					m_a1s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][rightIndex]);
					m_a2s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][leftIndex]);
					m_a3s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][upIndex])*frac2;
					m_a4s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][downIndex])*frac2;
					m_a0s[i][index] = m_a1s[i][index] + m_a2s[i][index] + m_a3s[i][index] + m_a4s[i][index];
				}
			}

			w /= 2;
			h /= 2;
		}
	
		m_phiBackupBetterCounter = 0;
	}

	if (m_extraPotentialChanged)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				int idx = getIndex(x, y);
				int nextX = (x+1)%m_width;
				int nextIdx = getIndex(nextX, y);

				m_extraElecticFieldNx[idx] = -(m_extraPotentialN[nextIdx] - m_extraPotentialN[idx]);
				m_extraElecticFieldPx[idx] = -(m_extraPotentialP[nextIdx] - m_extraPotentialP[idx]);
			}
		}

		for (int x = 0 ; x < m_width ; x++)
		{
			for (int y = 0 ; y < m_height-1 ; y++)
			{	
				int idx = getIndex(x, y);
				int nextIdx = getIndex(x, y+1);

				m_extraElecticFieldNy[idx] = -(m_extraPotentialN[nextIdx]-m_extraPotentialN[idx])*m_pixelFrac;
				m_extraElecticFieldPy[idx] = -(m_extraPotentialP[nextIdx]-m_extraPotentialP[idx])*m_pixelFrac;
			}

			// NOTE: the last row is not needed (and cannot be calculated by the way)
		}

		m_extraPotentialChanged = false;
	}

}

void Simulation2DDoubleRel::potentialFinder()
{
	if (m_phiBackupBetterCounter < m_phiMethodThreshold || m_phiBackupBetterCounter%m_phiMethodAdditionalCheckCount == 0)
	{
		// First create scaled grids of the charge

		//std::cout << m_phiBackupBetterCounter << " Trying both " << std::endl;

		int w = m_width;
		int h = m_height;

		for (int x = 0 ; x < w ; x++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				int idx = x+y*w;

				m_chargeSums[0][idx] = (m_pRel[idx]-m_nRel[idx])*m_chargeMultiplier;
			}
		}

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				for (int y = 0 ; y < h ; y += 2)
				{
					double chargeSum = m_chargeSums[i][x+y*w] + m_chargeSums[i][x+1+y*w] + m_chargeSums[i][x+(y+1)*w] + m_chargeSums[i][x+1+(y+1)*w];

					m_chargeSums[i+1][x/2+(y/2)*w2] = chargeSum; // sum it
				}
			}

			w /= 2;
			h /= 2;
		}

		// Width and height should now contain the dimensions of the coarsest grid
		
		// Calculate the potential for the lowest resolution grid
		double error, error2;
		
		error = blackRed(&(m_potential[m_multiScaleSteps-1][0]), m_multiScaleSteps-1, w, h, 100, 1.8f);

		for (int i = m_multiScaleSteps-2 ; i >= 0 ; i--)
		{
			int w2 = w;
			int h2 = h;

			w *= 2;
			h *= 2;

			// Interpolate the result for a first estimate of the higher resolution result

			for (int y = 1 ; y < h-1 ; y++) // leave the boundaries alone
			{
				double yFrac = (double)y/(double)(h-1);
				double y1d = yFrac*(h2-1);
				int y1 = (int)y1d;
				int y2 = y1+1;

				if (y2 == h2)
					y2 = h2-1;

				double t = y1d-(double)y1;

				for (int x = 0 ; x < w ; x++)
				{
					double xFrac = (double)x/(double)(w-1);
					double x1d = xFrac*(w2-1);
					int x1 = (int)x1d;
					int x2 = (x1+1)%w2;

					double s = x1d-(double)x1;

					double v1 = m_potential[i+1][x1+y1*w2];
					double v2 = m_potential[i+1][x2+y1*w2];
					double v3 = m_potential[i+1][x1+y2*w2];
					double v4 = m_potential[i+1][x2+y2*w2];

					// Bilinear interpolation
					m_potential[i][x+y*w] = v1*(1.0f-t)*(1.0f-s) + v2*(1.0f-t)*s + v3*t*(1.0f-s) + v4*t*s;
				}
			}

			// Optimize at new scale

			error = blackRed(&(m_potential[i][0]), i, w, h, 10, 1.8f);
		}

		// We shall also continuously update another potential field using another
		// method

		error2 = blackRed(&(m_backupPotential[0]), 0, m_width, m_height, 4, 1.0f);

		//std::cerr << "error = " << error << " error2 = " << error2 << std::endl;

		// We'll use the best result of the two approaches
		if (error2 >= error)
		{
			m_phiBackupBetterCounter = 0;
			memcpy(&(m_backupPotential[0]),&(m_potential[0][0]), sizeof(double)*m_totalPixels);
		}
		else
		{
			m_phiBackupBetterCounter++;
			memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
		}
	}
	else
	{
		// First grid of total charge

		int w = m_width;
		int h = m_height;

		for (int x = 0 ; x < w ; x++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				int idx = x+y*w;

				m_chargeSums[0][idx] = (m_pRel[idx]-m_nRel[idx])*m_chargeMultiplier;
			}
		}

		blackRed(&(m_backupPotential[0]), 0, m_width, m_height, 4, 1.0f);
		m_phiBackupBetterCounter++;
		memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
	}
	
	// Determine electric field from potential field

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = getIndex(x, y);
			int nextX = (x+1)%m_width;
			int nextIdx = getIndex(nextX, y);

			m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx]) + m_fieldBaseX[idx];
		}
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{	
			int idx = getIndex(x, y);
			int nextIdx = getIndex(x, y+1);

			m_electicFieldy[idx] = -(m_potential[0][nextIdx]-m_potential[0][idx])*m_pixelFrac + m_fieldBaseY[idx];
		}

		// NOTE: the last row is not needed (and cannot be calculated by the way)
	}

}

double Simulation2DDoubleRel::blackRed(double *pSrc, int aIndex, int width, int height, int steps, double w)
{
	double error = -1;

	for (int i = 0 ; i < steps ; i++)
	{
		error = 0;

		// Alternating updates in the same array seems to be much more stable!

		for (int loop = 0 ; loop < 2 ; loop++)
		{
			int yStart = 1;
			int yStop = height-1;

			for (int y = yStart ; y < yStop ; y++)
			{
				int xOff = (y+loop)%2;

				for (int x = xOff ; x < width ; x += 2)
				{
					int xNext = (x+1)%width;
					int xPrev = (x-1+width)%width;
					int yNext = y+1;
					int yPrev = y-1;

					int index = x + y*width;
					int leftIndex = xPrev + y*width;
					int rightIndex = xNext + y*width;
					int upIndex = x + yNext*width;
					int downIndex = x+ yPrev*width;

					double prediction = (m_a1s[aIndex][index]*pSrc[rightIndex] + m_a2s[aIndex][index]*pSrc[leftIndex]
							  +  m_a3s[aIndex][index]*pSrc[upIndex]  + m_a4s[aIndex][index]*pSrc[downIndex]
							  +  m_chargeSums[aIndex][index])/m_a0s[aIndex][index];

					double curValue = pSrc[index];

					double diff = m_VpredSub[aIndex][index] + prediction - curValue;

					pSrc[index] += w*diff;

					// update error

					error += diff*diff;
				}
			}
		}
	}

	return std::sqrt(error/(double)(width*height));
}

void Simulation2DDoubleRel::resizeArrays()
{
	m_n.resize(m_totalPixels);
	m_p.resize(m_totalPixels);
	m_background.resize(m_totalPixels);
	m_generationRate.resize(m_totalPixels);
	m_recombinationFactor.resize(m_totalPixels);
	m_De.resize(m_totalPixels);
	m_Dh.resize(m_totalPixels);
	m_eMob.resize(m_totalPixels);
	m_hMob.resize(m_totalPixels);

	int tmpSize = m_totalPixels;

	// More arrays need to be allocated for the multigrid method

	m_epsRels.resize(m_multiScaleSteps);
	m_chargeSums.resize(m_multiScaleSteps);
	m_potential.resize(m_multiScaleSteps);
	m_a0s.resize(m_multiScaleSteps);
	m_a1s.resize(m_multiScaleSteps);
	m_a2s.resize(m_multiScaleSteps);
	m_a3s.resize(m_multiScaleSteps);
	m_a4s.resize(m_multiScaleSteps);
	m_Vbase.resize(m_multiScaleSteps);
	m_VpredSub.resize(m_multiScaleSteps);
	m_chargeSumBase.resize(m_multiScaleSteps);

	m_backupPotential.resize(m_totalPixels);

	for (int i = 0 ; i < m_multiScaleSteps ; i++)
	{
		m_epsRels[i].resize(tmpSize);
		m_chargeSums[i].resize(tmpSize);
		m_potential[i].resize(tmpSize);
		m_a0s[i].resize(tmpSize);
		m_a1s[i].resize(tmpSize);
		m_a2s[i].resize(tmpSize);
		m_a3s[i].resize(tmpSize);
		m_a4s[i].resize(tmpSize);
		m_Vbase[i].resize(tmpSize);
		m_VpredSub[i].resize(tmpSize);
		m_chargeSumBase[i].resize(tmpSize);
		tmpSize /= 4;
	}

	m_electicFieldx.resize(m_totalPixels);
	m_electicFieldy.resize(m_totalPixels);

	// Initialize the values of some arrays

	setValues(m_n, 0);
	setValues(m_p, 0);
	setValues(m_background, 0);
	setValues(m_generationRate, 0);
	setValues(m_recombinationFactor, 0);
	setValues(m_De, 0);
	setValues(m_Dh, 0);
	setValues(m_eMob, 0);
	setValues(m_hMob, 0);
	setValues(m_epsRels[0], 1);
	setValues(m_background, 0);

	// Allocate memory for the arrays which will contain the number currents
	// due to diffusion and electric field respectively

	m_numCurTotEx.resize(m_totalPixels);
	m_numCurTotEy.resize(m_totalPixels);
	m_numCurTotHx.resize(m_totalPixels);
	m_numCurTotHy.resize(m_totalPixels);

	m_dndt.resize(m_totalPixels);
	m_dpdt.resize(m_totalPixels);

	m_nRel.resize(m_totalPixels);
	m_pRel.resize(m_totalPixels);
	m_GR.resize(m_totalPixels);
	m_rp.resize(m_totalPixels);
	m_rn.resize(m_totalPixels);
	
	m_fieldBaseX.resize(m_totalPixels);
	m_fieldBaseY.resize(m_totalPixels);

	m_extraPotentialN.resize(m_totalPixels);
	m_extraPotentialP.resize(m_totalPixels);
	m_extraElecticFieldNx.resize(m_totalPixels);
	m_extraElecticFieldNy.resize(m_totalPixels);
	m_extraElecticFieldPx.resize(m_totalPixels);
	m_extraElecticFieldPy.resize(m_totalPixels);

	setValues(m_extraPotentialN, 0);
	setValues(m_extraPotentialP, 0);
}

void Simulation2DDoubleRel::writePlotData(FILE *pFile)
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x + y*m_width;

			double j1 = 0;
			double j2 = 0;

			if (y == m_height-1)
			{
				j1 = m_numCurTotHx[idx-m_width]-m_numCurTotEx[idx-m_width];
				j2 = m_numCurTotHy[idx-m_width]-m_numCurTotEy[idx-m_width];
			}
			else
			{
				j1 = m_numCurTotHx[idx]-m_numCurTotEx[idx];
				j2 = m_numCurTotHy[idx]-m_numCurTotEy[idx];
			}

			fprintf(pFile, "%d %d %g %g %g %g %g %g  %g %g %g %g %g %g\n", x, y, 
					         (double)m_npDensFactor*(double)m_n[idx], 
						 (double)m_npDensFactor*(double)m_p[idx], 
						 (double)m_potential[0][idx], 
					         (double)(1.0/(m_npDensFactor*m_timeFactor))*(double)m_recombinationFactor[idx]*(double)m_n[idx]*(double)m_p[idx]*m_npDensFactor*m_npDensFactor,
						 (double)j1*(double)(m_npDensFactor*m_pixelWidth/m_timeFactor), 
						 (double)j2*(double)(m_npDensFactor*m_pixelWidth/m_timeFactor),
						 (double)m_generationRate[idx]*(double)(m_npDensFactor/m_timeFactor), 
						 (double)m_recombinationFactor[idx]*(double)(1.0/(m_npDensFactor*m_timeFactor)),
						 (double)m_De[idx]*(double)(m_pixelWidth*m_pixelWidth/m_timeFactor), 
						 (double)m_Dh[idx]*(double)(m_pixelWidth*m_pixelWidth/m_timeFactor), 
						 (double)m_eMob[idx]*(double)(m_pixelWidth*m_pixelWidth/m_timeFactor), 
						 (double)m_hMob[idx]*(double)(m_pixelWidth*m_pixelWidth/m_timeFactor));
		}

		fprintf(pFile, "\n");
	}
	fprintf(pFile, "\n");
}

void Simulation2DDoubleRel::prepareRelativeCalculation()
{
	memset(&(m_nRel[0]), 0, sizeof(double)*m_totalPixels);
	memset(&(m_pRel[0]), 0, sizeof(double)*m_totalPixels);

	for (int y = 1 ; y < m_height-1 ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x + y*m_width;
			double rf = m_recombinationFactor[idx];
			double p = m_p[idx];
			double n = m_n[idx];

			m_GR[idx] = m_generationRate[idx] - rf*p*n;
			m_rn[idx] = rf*n;
			m_rp[idx] = rf*p;

			m_n[idx] += m_nRel[idx];
			m_p[idx] += m_pRel[idx];
		}
	}

	memcpy(&(m_Vbase[0][0]), &(m_potential[0][0]), m_totalPixels*sizeof(double));

	int w = m_width;
	int h = m_height;

	// set boundary conditions (just to be safe)
	{
		int yOff = (h-1)*w;
		for (int x = 0 ; x < w ; x++)
		{
			m_Vbase[0][x] = 0;
			m_Vbase[0][x+yOff] = m_deltaPhi;
			m_potential[0][x] = 0;
			m_potential[0][x+yOff] = 0;
		}
	}

	for (int i = 1 ; i < m_multiScaleSteps ; i++) // TODO: improve this very simple interpolation?
	{
		int w2 = w/2;
		int h2 = h/2;

		for (int y = 0 ; y < h2 ; y++)
		{
			double ySrc = ((double)y/(double)(h2-1))*(double)(h-1);
			int y0 = (int)ySrc;
			int y1 = y0+1;
			if (y0 == h-1)
				y1 = h-1;

			double yFrac = ySrc - (double)y0;

			for (int x = 0 ; x < w2 ; x++)
			{
				double xSrc = ((double)x/(double)(w2-1))*(double)(w-1);
				int x0 = (int)xSrc;
				int x1 = (x0+1)%w;

				double xFrac = xSrc - (double)x0;
				double v00 = m_Vbase[i-1][x0+y0*w];
				double v01 = m_Vbase[i-1][x0+y1*w];
				double v10 = m_Vbase[i-1][x1+y0*w];
				double v11 = m_Vbase[i-1][x1+y1*w];

				double val = (1.0-xFrac)*(1.0-yFrac)*v00
					   + xFrac*(1.0-yFrac)*v10
					   + (1.0-xFrac)*yFrac*v01
					   + xFrac*yFrac*v11;

				m_Vbase[i][x+y*w2] = val;
			}
		}
		w = w2;
		h = h2;
	
		// set boundary conditions (just to be safe)
		{
			int yOff = (h-1)*w;
			for (int x = 0 ; x < w ; x++)
			{
				m_Vbase[i][x] = 0;
				m_Vbase[i][x+yOff] = m_deltaPhi;
				m_potential[i][x] = 0;
				m_potential[i][x+yOff] = 0;
			}
		}
	}

	// base charge sums

	w = m_width;
	h = m_height;

	for (int x = 0 ; x < w ; x++)
	{
		for (int y = 0 ; y < h ; y++)
		{
			int idx = x+y*w;

			m_chargeSumBase[0][idx] = (m_p[idx]-m_n[idx]+m_background[idx])*m_chargeMultiplier;
		}
	}

	for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
	{
		int w2 = w/2;

		for (int x = 0 ; x < w ; x += 2)
		{
			for (int y = 0 ; y < h ; y += 2)
			{
				double chargeSum = m_chargeSumBase[i][x+y*w] + m_chargeSumBase[i][x+1+y*w] + m_chargeSumBase[i][x+(y+1)*w] + m_chargeSumBase[i][x+1+(y+1)*w];

				m_chargeSumBase[i+1][x/2+(y/2)*w2] = chargeSum; // sum it
			}
		}

		w /= 2;
		h /= 2;
	}

	// Calculations that can already be done 

	w = m_width;
	h = m_height;

	for (int i = 0 ; i < m_multiScaleSteps ; i++)
	{
		for (int y = 1 ; y < h-1 ; y++)
		{
			for (int x = 0 ; x < w ; x++)
			{
				int xNext = (x+1)%w;
				int xPrev = (x-1+w)%w;
				int yNext = y+1;
				int yPrev = y-1;

				int index = x + y*w;
				int leftIndex = xPrev + y*w;
				int rightIndex = xNext + y*w;
				int upIndex = x + yNext*w;
				int downIndex = x+ yPrev*w;

				double D = (m_a1s[i][index]*m_Vbase[i][leftIndex] + m_a2s[i][index]*m_Vbase[i][rightIndex]
					   + m_a3s[i][index]*m_Vbase[i][upIndex] + m_a4s[i][index]*m_Vbase[i][downIndex] + m_chargeSumBase[i][index] )/m_a0s[i][index]
					 - m_Vbase[i][index];

				m_VpredSub[i][index] = D;
			}
		}

		w /= 2;
		h /= 2;
	}

	// m_potential[0][0], or m_Vbase[0] already contains the new estimate of the next potential
	memset(&(m_backupPotential[0]), 0, sizeof(double)*m_totalPixels);

	// Calculate base field corresponding to base potential

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = getIndex(x, y);
			int nextX = (x+1)%m_width;
			int nextIdx = getIndex(nextX, y);

			m_fieldBaseX[idx] = -(m_Vbase[0][nextIdx] - m_Vbase[0][idx]);
		}
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{	
			int idx = getIndex(x, y);
			int nextIdx = getIndex(x, y+1);

			m_fieldBaseY[idx] = -(m_Vbase[0][nextIdx]-m_Vbase[0][idx])*m_pixelFrac;
		}

		// NOTE: the last row is not needed (and cannot be calculated by the way)
	}
}

void Simulation2DDoubleRel::mergeRelativeResults()
{
	for (int y = 1 ; y < m_height-1 ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;

			m_n[idx] += m_nRel[idx];
			m_p[idx] += m_pRel[idx];
		}
	}

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
			m_potential[0][x+y*m_width] += m_Vbase[0][x+y*m_width];
	}
}

bool Simulation2DDoubleRel::setState(const SimulationState &state, std::vector<std::string> &warnings)
{
	if (!m_init)
	{
		setErrorString("Simulation not initialized");
		return false;
	}
	if (!state.isInitialized())
	{
		setErrorString("State to be used is not initialized");
		return false;
	}
	if (state.getDimensions() != 2)
	{
		setErrorString("State is not one-dimensional");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels() || state.getNumberOfYPixels() != getNumYPixels())
	{
		setErrorString("State does not have same dimensions as simulation");
		return false;
	}

	std::vector<bool> m_input(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_output(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_optional(SIMSTATE_GRIDPROP_MAX);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		m_input[i] = false;
		m_output[i] = false;
		m_optional[i] = false;
	}

	m_input[SIMSTATE_GRIDPROP_N] = true;
	m_input[SIMSTATE_GRIDPROP_P] = true;
	m_input[SIMSTATE_GRIDPROP_DN] = true;
	m_input[SIMSTATE_GRIDPROP_DP] = true;
	m_input[SIMSTATE_GRIDPROP_NMOB] = true;
	m_input[SIMSTATE_GRIDPROP_PMOB] = true;
	m_input[SIMSTATE_GRIDPROP_EPSREL] = true;

	m_output[SIMSTATE_GRIDPROP_N] = true;
	m_output[SIMSTATE_GRIDPROP_P] = true;
	m_output[SIMSTATE_GRIDPROP_V] = true;
	m_output[SIMSTATE_GRIDPROP_JNX] = true;
	m_output[SIMSTATE_GRIDPROP_JPX] = true;
	m_output[SIMSTATE_GRIDPROP_JNY] = true;
	m_output[SIMSTATE_GRIDPROP_JPY] = true;
	m_output[SIMSTATE_GRIDPROP_R] = true;

	m_optional[SIMSTATE_GRIDPROP_BG] = true;
	m_optional[SIMSTATE_GRIDPROP_G] = true;
	m_optional[SIMSTATE_GRIDPROP_RF] = true;
	m_optional[SIMSTATE_GRIDPROP_VNEXTRA] = true;
	m_optional[SIMSTATE_GRIDPROP_VPEXTRA] = true;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (m_input[i])
		{
			if (!state.isGridPropertySet(i))
			{
				setErrorString("Grid property '" + state.getGridPropertyName(i) + "' is required, but has not been set");
				return false;
			}
		}
		else
		{
			if (state.isGridPropertySet(i))
			{
				if (!m_output[i] && !m_optional[i])
					warnings.push_back("Property '" + state.getGridPropertyName(i) + "' is set but not used");
			}
		}
	}

	int numX = getNumXPixels();
	int numY = getNumYPixels();

	for (int y = 0 ; y < numY ; y++)
	{
		for (int x = 0 ; x < numX ; x++)
		{
			setElectronNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_N, x, y));
			setHoleNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_P, x, y));

			setHoleMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_P, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_BG))
				setBackgroundNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_BG, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_G))
				setGenerationRate(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_G, x, y));
			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_RF))
				setRecombinationFactor(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_RF, x, y));

			setElectronDiffusionConstant(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_DN, x, y));
			setHoleDiffusionConstant(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_DP, x, y));
			setElectronMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_NMOB, x, y));
			setHoleMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_PMOB, x, y));
			setRelativePermittivity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_EPSREL, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_V))
				setPotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_V, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VNEXTRA))
				setExtraElectronPotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VPEXTRA))
				setExtraHolePotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, x, y));
		}
	}

	double vDiff;

	if (!state.getDoubleProperty(SIMSTATE_PROP_VDIFF, vDiff))
	{
		setErrorString("Potential difference has not been set");
		return false;
	}

	setPotentialDifference(vDiff);

	if (state.getRecombinationModel() == SimulationState::Braun)
	{
		setErrorString("Braun recombination model selected, but not supported in this simulation type");
		return false;
	}
	else
	{
		double a, kf;

		if (state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
			warnings.push_back("'Pair distance' parameter is set, but is only used in the braun model");
		if (state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
			warnings.push_back("'Dissociation rate' parameter 'kf' is set, but is only used in the braun model");
	}

	//FILE *f = fopen("log.txt","wt");
	//writePlotData(f);
	//fclose(f);

	return true;
}

bool Simulation2DDoubleRel::storeState(SimulationState &state) const
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels() ||
	    state.getNumberOfYPixels() != getNumYPixels() )
	{
		setErrorString("Number of pixels is not same as in simulation state");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		state.clearDoubleProperty(i);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		state.clearGridProperty(i);

	std::vector<double> tmp(getNumXPixels() * getNumYPixels());

	if (!copyProperty(state, SIMSTATE_GRIDPROP_N, tmp, m_n, m_npDensFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_P, tmp, m_p, m_npDensFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_V, m_potential[0]))
	{
		setErrorString("Unable to store n,p or V");
		return false;
	}

	if (!copyProperty(state, SIMSTATE_GRIDPROP_BG, tmp, m_background, m_npDensFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_G, tmp, m_generationRate, m_npDensFactor/m_timeFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_RF, tmp, m_recombinationFactor, 1.0/(m_npDensFactor*m_timeFactor)) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_DN, tmp, m_De, m_pixelWidth*m_pixelWidth/m_timeFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_DP, tmp, m_Dh, m_pixelWidth*m_pixelWidth/m_timeFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_NMOB, tmp, m_eMob, m_pixelWidth*m_pixelWidth/m_timeFactor) ||
	    !copyProperty(state, SIMSTATE_GRIDPROP_PMOB, tmp, m_hMob, m_pixelWidth*m_pixelWidth/m_timeFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels[0]) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, m_extraPotentialN) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, m_extraPotentialP) )
	{
		setErrorString("Unable to store fixed simulation items");
		return false;
	}

	double jFactor = m_npDensFactor*m_pixelWidth/m_timeFactor;
	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			double jnx = m_numCurTotEx[x+y*m_width]*jFactor;
			double jpx = m_numCurTotHx[x+y*m_width]*jFactor;
			double jny = m_numCurTotEy[x+y*m_width]*jFactor;
			double jpy = m_numCurTotHy[x+y*m_width]*jFactor;

			if (!state.setGridProperty(SIMSTATE_GRIDPROP_JNX, x, y, jnx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPX, x, y, jpx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JNY, x, y, jny) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPY, x, y, jpy))
			{
				setErrorString("Unable to store the number current");
				return false;
			}
		}
	}

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;
			double R = m_recombinationFactor[idx] * (m_n[idx] * m_p[idx]) * m_npDensFactor/m_timeFactor;
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, x, y, R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
		}
	}

	state.setRecombinationModel(SimulationState::Simple);	

	if (!state.setDoubleProperty(SIMSTATE_PROP_VDIFF, m_deltaPhi))
	{
		setErrorString("Couldn't store potential difference");
		return false;
	}

	return true;
}

bool Simulation2DDoubleRel::copyProperty(SimulationState &dst, int propID, std::vector<double> &tmp, const std::vector<double> &grid, double multiplier)
{
	int num = grid.size();

	for (int i = 0 ; i < num ; i++)
		tmp[i] = grid[i]*multiplier;

	if (!dst.setGridProperty(propID, tmp))
		return false;

	return true;
}

