#ifndef SIMULATION2DDOUBLEGPUREL_H

#define SIMULATION2DDOUBLEGPUREL_H

#include "simulation2d.h"
#include "openclkernel.h"

class SimulationState;

class Simulation2DDoubleGPURel : public Simulation2D
{
public:
	Simulation2DDoubleGPURel();
	~Simulation2DDoubleGPURel();

	// Set grid dimensions
	bool init(int lowestWidth, int lowestHeight, double realWidth, double realHeight, int scaleSteps, 
		  int *pNumX, int *pNumY);
	bool isInitialized() const										{ return m_init; }

	int getNumXPixels() const										{ return m_width; }
	int getNumYPixels() const										{ return m_height; }
	double getPixelWidth() const										{ return m_pixelWidth; }
	double getPixelHeight() const										{ return m_pixelHeight; }

	bool save(const std::string &fileName) const;
	bool load(const std::string &fileName);
	bool write(serut::SerializationInterface &s) const;
	bool read(serut::SerializationInterface &s);

	bool setState(const SimulationState &state, std::vector<std::string> &warnings);
	bool storeState(SimulationState &state) const;

	// After 'init' has been called, these functions can be used to specify a specific
	// starting situation
	void setElectronNumberDensity(int x, int y, double dens)						{ setValue(m_n, x, y, (double)(dens/m_npDensFactor)); }
	void setElectronNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_n, x0, x1, y0, y1, (double)(dens/m_npDensFactor)); }
	void setElectronNumberDensity(double dens)								{ setValues(m_n, (double)(dens/m_npDensFactor)); }

	void setHoleNumberDensity(int x, int y, double dens)							{ setValue(m_p, x, y, (double)(dens/m_npDensFactor)); }
	void setHoleNumberDensity(int x0, int x1, int y0, int y1, double dens)					{ setValues(m_p, x0, x1, y0, y1, (double)(dens/m_npDensFactor)); }
	void setHoleNumberDensity(double dens)									{ setValues(m_p, (double)(dens/m_npDensFactor)); }

	void setBackgroundNumberDensity(int x, int y, double dens)						{ setValue(m_background, x, y, (double)(dens/m_npDensFactor)); }
	void setBackgroundNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_background, x0, x1, y0, y1, (double)(dens/m_npDensFactor)); }
	void setBackgroundNumberDensity(double dens)								{ setValues(m_background, (double)(dens/m_npDensFactor)); }

	void setGenerationRate(int x, int y, double rate)							{ setValue(m_generationRate, x, y, (double)(rate/m_npDensFactor*m_timeFactor)); }
	void setGenerationRate(int x0, int x1, int y0, int y1, double rate)					{ setValues(m_generationRate, x0, x1, y0, y1, (double)(rate/m_npDensFactor*m_timeFactor)); }
	void setGenerationRate(double rate)									{ setValues(m_generationRate, (double)(rate/m_npDensFactor*m_timeFactor)); }

	// r = factor * n * p;
	void setRecombinationFactor(int x, int y, double factor)						{ double v = (double)(factor*m_npDensFactor*m_timeFactor); setValue(m_recombinationFactor, x, y, v); setValue(m_recombinationFactorFloat, x, y, v); }
	void setRecombinationFactor(int x0, int x1, int y0, int y1, double factor)				{ double v = (double)(factor*m_npDensFactor*m_timeFactor); setValues(m_recombinationFactor, x0, x1, y0, y1, v);  setValues(m_recombinationFactorFloat, x0, x1, y0, y1, v); }
	void setRecombinationFactor(double factor)								{ double v = (double)(factor*m_npDensFactor*m_timeFactor); setValues(m_recombinationFactor, v); setValues(m_recombinationFactorFloat, v); }

	// bottom = 0, top = V
	void setPotentialDifference(double V)									{ m_deltaPhi = V; m_phiBackupBetterCounter = 0; }

	void setElectronDiffusionConstant(int x, int y, double D)						{ setValue(m_De, x, y, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setElectronDiffusionConstant(int x0, int x1, int y0, int y1, double D)				{ setValues(m_De, x0, x1, y0, y1, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setElectronDiffusionConstant(double D)								{ setValues(m_De, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }

	void setHoleDiffusionConstant(int x, int y, double D)							{ setValue(m_Dh, x, y, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setHoleDiffusionConstant(int x0, int x1, int y0, int y1, double D)					{ setValues(m_Dh, x0, x1, y0, y1, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setHoleDiffusionConstant(double D)									{ setValues(m_Dh, (double)(D*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }

	void setElectronMobility(int x, int y, double mu)							{ setValue(m_eMob, x, y, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setElectronMobility(int x0, int x1, int y0, int y1, double mu)					{ setValues(m_eMob, x0, x1, y0, y1, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setElectronMobility(double mu)									{ setValues(m_eMob, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }

	void setHoleMobility(int x, int y, double mu)								{ setValue(m_hMob, x, y, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setHoleMobility(int x0, int x1, int y0, int y1, double mu)						{ setValues(m_hMob, x0, x1, y0, y1, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }
	void setHoleMobility(double mu)										{ setValues(m_hMob, (double)(mu*m_timeFactor/(m_pixelWidth*m_pixelWidth))); }

	void setRelativePermittivity(int x, int y, double epsRel)						{ setValue(m_epsRels[0], x, y, epsRel); m_epsChanged = true; }
	void setRelativePermittivity(int x0, int x1, int y0, int y1, double epsRel)				{ setValues(m_epsRels[0], x0, x1, y0, y1, epsRel); m_epsChanged = true; }
	void setRelativePermittivity(double epsRel)								{ setValues(m_epsRels[0], epsRel); m_epsChanged = true; }

	void setPotential(int x, int y, double v)								{ setValue(m_Vbase[0], x, y, v); m_phiBackupBetterCounter = 0; }
	void setPotential(int x0, int x1, int y0, int y1, double v)						{ setValues(m_Vbase[0], x0, x1, y0, y1, v); m_phiBackupBetterCounter = 0; }
	void setPotential(double v)										{ setValues(m_Vbase[0], v); m_phiBackupBetterCounter = 0; }

	void setExtraElectronPotential(int x, int y, double v)							{ setValue(m_extraPotentialN, x, y, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(int x0, int x1, int y0, int y1, double v)				{ setValues(m_extraPotentialN, x0, x1, y0, y1, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(double v)								{ setValues(m_extraPotentialN, v); m_extraPotentialChanged = true; }

	void setExtraHolePotential(int x, int y, double v)							{ setValue(m_extraPotentialP, x, y, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(int x0, int x1, int y0, int y1, double v)					{ setValues(m_extraPotentialP, x0, x1, y0, y1, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(double v)									{ setValues(m_extraPotentialP, v); m_extraPotentialChanged = true; }

	bool start(int steps, double dt, bool inverseMatrixSolver);
	bool start(int seconds, double dt, int &steps, bool inverseMatrixSolver);
	
	void calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const;
	void calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const;

	void getElectronNumberDensity(std::vector<double> &dst) const						{ getProperty(m_n, dst, m_npDensFactor); }
	void getHoleNumberDensity(std::vector<double> &dst) const						{ getProperty(m_p, dst, m_npDensFactor); }
	void getBackgroundNumberDensity(std::vector<double> &dst) const						{ getProperty(m_background, dst, m_npDensFactor); }
	void getGenerationRate(std::vector<double> &dst) const							{ getProperty(m_generationRate, dst, m_npDensFactor/m_timeFactor); }
	void getRecombinationFactor(std::vector<double> &dst) const						{ getProperty(m_recombinationFactor, dst, 1.0/(m_npDensFactor*m_timeFactor)); }
	void getPotential(std::vector<double> &dst) const							{ getProperty(m_Vbase[0], dst, 1.0); }
	void getElectronDiffusionConstant(std::vector<double> &dst) const					{ getProperty(m_De, dst, m_pixelWidth*m_pixelWidth/m_timeFactor); }
	void getHoleDiffusionConstant(std::vector<double> &dst) const						{ getProperty(m_Dh, dst, m_pixelWidth*m_pixelWidth/m_timeFactor); }
	void getElectronMobility(std::vector<double> &dst) const						{ getProperty(m_eMob, dst, m_pixelWidth*m_pixelWidth/m_timeFactor); }
	void getHoleMobility(std::vector<double> &dst) const							{ getProperty(m_hMob, dst, m_pixelWidth*m_pixelWidth/m_timeFactor); }
	void getRelativePermittivity(std::vector<double> &dst) const						{ getProperty(m_epsRels[0], dst, 1.0); }
	void getTotalElectronXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEx, dst, m_npDensFactor*m_pixelWidth/m_timeFactor); }
	void getTotalElectronYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEy, dst, m_npDensFactor*m_pixelWidth/m_timeFactor); }
	void getTotalHoleXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHx, dst, m_npDensFactor*m_pixelWidth/m_timeFactor); }
	void getTotalHoleYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHy, dst, m_npDensFactor*m_pixelWidth/m_timeFactor); }
	void getExtraElectronPotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialN, dst, 1.0); }
	void getExtraHolePotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialP, dst, 1.0); }

	double getPotentialDifference() const									{ return m_deltaPhi; }

	void writePlotData(FILE *pFile);
protected:
	int getIndex(int x, int y) const									{ return x+y*m_width; } 
	void setValue(std::vector<double> &grid, int x, int y, double v)						{ grid[getIndex(x,y)] = v; }
	void setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v);
	void setValues(std::vector<double> &grid, double v)							{ setValues(grid, 0, m_width, 0, m_height, v); }
	void setValue(std::vector<float> &grid, int x, int y, double v)						{ grid[getIndex(x,y)] = (float)v; }
	void setValues(std::vector<float> &grid, int x0, int x1, int y0, int y1, double v);
	void setValues(std::vector<float> &grid, double v)							{ setValues(grid, 0, m_width, 0, m_height, v); }
	void getProperty(const std::vector<double> &grid, std::vector<double> &dst, double multiplier) const;
	void getProperty(const std::vector<float> &grid, std::vector<double> &dst, double multiplier) const;
	static bool copyProperty(SimulationState &dst, int propID, std::vector<double> &tmp, const std::vector<double> &grid, double multiplier);
	static bool copyProperty(SimulationState &dst, int propID, std::vector<double> &tmp, const std::vector<float> &grid, double multiplier);

	float blackRed(float *pSrc, int aIndex, int width, int height, int steps, float w);
	double getSGCurrent(double v, double D, double delta, double n1, double n2);

	bool resizeArrays();
	void initializePotentialFinder();
	void potentialFinder();
	void calcNumberCurrents();
	void updateDensities(float scaledDt);

	void prepareRelativeCalculation();
	void mergeRelativeResults();

	void clearCLMemory();
	void zeroCLMemory();
	bool allocateCLMemory();
	bool initGPU();
	template <class T> bool commonStart(T &extraCheck, int &steps, double dt);
	static std::string getOpenCLProgram();

	bool m_init;
	bool m_initPotential;
	bool m_epsChanged;
	int m_width, m_height; 
	int m_totalPixels;
	int m_multiScaleSteps; 
	
	double m_pixelWidth;
	double m_pixelHeight;
	double m_pixelFrac;
	double m_pixelFracInv;

	std::vector<double> m_n, m_p, m_background;
	std::vector<float> m_nFloat, m_pFloat;

	std::vector<double> m_generationRate, m_recombinationFactor;
	std::vector<float> m_recombinationFactorFloat;
	std::vector<float> m_De, m_Dh, m_eMob, m_hMob;
	
	std::vector<std::vector<float> > m_chargeSums;
	std::vector<std::vector<double> > m_epsRels;
	std::vector<std::vector<double> > m_a0s, m_a1s, m_a2s, m_a3s, m_a4s;
	std::vector<float> m_backupPotential; // ok
	std::vector<float> m_electicFieldx, m_electicFieldy; // ok
	std::vector<float> m_numCurTotExRel, m_numCurTotEyRel, m_numCurTotHxRel, m_numCurTotHyRel; // ok
	std::vector<double> m_numCurTotEx, m_numCurTotEy, m_numCurTotHx, m_numCurTotHy; // ok

	std::vector<float> m_dndt, m_dpdt; // ok

	std::vector<float> m_nRel, m_pRel;
	std::vector<float> m_GRminJDivE, m_GRminJDivH, m_rn, m_rp;

	std::vector<std::vector<double> > m_Vbase;
	std::vector<std::vector<double> > m_chargeSumBase;
	std::vector<std::vector<float> > m_VpredSub;
	std::vector<float> m_fieldBaseXN, m_fieldBaseYN;
	std::vector<float> m_fieldBaseXP, m_fieldBaseYP;

	std::vector<std::vector<float> > m_potential, m_a0sFloat, m_a1sFloat, m_a2sFloat, m_a3sFloat, m_a4sFloat;

	std::vector<double> m_extraPotentialN;
	std::vector<double> m_extraPotentialP;
	std::vector<double> m_extraElecticFieldNx, m_extraElecticFieldNy; 
	std::vector<double> m_extraElecticFieldPx, m_extraElecticFieldPy; 
	bool m_extraPotentialChanged;

	double m_deltaPhi;

	int m_phiBackupBetterCounter;
	int m_phiMethodThreshold;
	int m_phiMethodAdditionalCheckCount;

	double m_npDensFactor;
	double m_chargeMultiplier;
	float m_chargeMultiplierFloat;
	double m_timeFactor;

	OpenCLKernel m_gpu;
	cl_mem m_clPotential, m_clVSubPred, m_clExBaseN, m_clEyBaseN,
	       m_clExBaseP, m_clEyBaseP,
	       m_clDe, m_clDh, m_clEMob, m_clHMob,
	       m_clN, m_clP, m_clNBase, m_clPBase,
	       m_clECurX, m_clECurY, m_clHCurX, m_clHCurY,
	       m_clGRminJDivE, m_clGRminJDivH,
	       m_clRN, m_clRP, m_clRecFactor, 
	       m_clN2, m_clP2, 
	       m_clA0, m_clA1, m_clA2, m_clA3, m_clA4;
};

inline void Simulation2DDoubleGPURel::setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v)
{
	if (x1 < x0)
		return;
	if (y1 < y0)
		return;

	for (int y = y0 ; y != y1 ; y++)
		for (int x = x0 ; x != x1 ; x++)
			setValue(grid, x, y, v);
}

inline void Simulation2DDoubleGPURel::setValues(std::vector<float> &grid, int x0, int x1, int y0, int y1, double v)
{
	if (x1 < x0)
		return;
	if (y1 < y0)
		return;

	for (int y = y0 ; y != y1 ; y++)
		for (int x = x0 ; x != x1 ; x++)
			setValue(grid, x, y, v);
}

inline void Simulation2DDoubleGPURel::getProperty(const std::vector<double> &grid, std::vector<double> &dst, double multiplier) const
{
	dst.resize(grid.size());
	int num = dst.size();

	for (int i = 0 ; i < num ; i++)
		dst[i] = multiplier*(double)grid[i];
}

inline void Simulation2DDoubleGPURel::getProperty(const std::vector<float> &grid, std::vector<double> &dst, double multiplier) const
{
	dst.resize(grid.size());
	int num = dst.size();

	for (int i = 0 ; i < num ; i++)
		dst[i] = multiplier*(double)grid[i];
}

#endif // SIMULATION2DDOUBLEGPUREL_H

