#ifndef SIMULATION2DDOUBLE_H

#define SIMULATION2DDOUBLE_H

#include "simulation2d.h"

class SimulationState;
class InverseMatrixPoissonSolver2D;

class Simulation2DDouble : public Simulation2D
{
public:
	Simulation2DDouble();
	~Simulation2DDouble();

	// Set grid dimensions
	bool init(int lowestWidth, int lowestHeight, double realWidth, double realHeight, int scaleSteps, 
		  int *pNumX, int *pNumY);
	bool isInitialized() const										{ return m_init; }

	int getNumXPixels() const										{ return m_width; }
	int getNumYPixels() const										{ return m_height; }
	double getPixelWidth() const										{ return m_pixelWidth2; }
	double getPixelHeight() const										{ return m_pixelHeight2; }

	bool save(const std::string &fileName) const;
	bool load(const std::string &fileName);
	bool write(serut::SerializationInterface &s) const;
	bool read(serut::SerializationInterface &s);

	bool setState(const SimulationState &state, std::vector<std::string> &warnings);
	bool storeState(SimulationState &state) const;

	// After 'init' has been called, these functions can be used to specify a specific
	// starting situation
	void setElectronNumberDensity(int x, int y, double dens)						{ setValue(m_n, x, y, dens); }
	void setElectronNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_n, x0, x1, y0, y1, dens); }
	void setElectronNumberDensity(double dens)								{ setValues(m_n, dens); }

	void setHoleNumberDensity(int x, int y, double dens)							{ setValue(m_p, x, y, dens); }
	void setHoleNumberDensity(int x0, int x1, int y0, int y1, double dens)					{ setValues(m_p, x0, x1, y0, y1, dens); }
	void setHoleNumberDensity(double dens)									{ setValues(m_p, dens); }

	void setBackgroundNumberDensity(int x, int y, double dens)						{ setValue(m_background, x, y, dens); }
	void setBackgroundNumberDensity(int x0, int x1, int y0, int y1, double dens)				{ setValues(m_background, x0, x1, y0, y1, dens); }
	void setBackgroundNumberDensity(double dens)								{ setValues(m_background, dens); }

	void setGenerationRate(int x, int y, double rate)							{ setValue(m_generationRate, x, y, rate); }
	void setGenerationRate(int x0, int x1, int y0, int y1, double rate)					{ setValues(m_generationRate, x0, x1, y0, y1, rate); }
	void setGenerationRate(double rate)									{ setValues(m_generationRate, rate); }

	// r = factor * n * p;
	void setRecombinationFactor(int x, int y, double factor)						{ setValue(m_recombinationFactor, x, y, factor); }
	void setRecombinationFactor(int x0, int x1, int y0, int y1, double factor)				{ setValues(m_recombinationFactor, x0, x1, y0, y1, factor); }
	void setRecombinationFactor(double factor)								{ setValues(m_recombinationFactor, factor); }

	// bottom = 0, top = V
	void setPotentialDifference(double V)									{ m_deltaPhi = V; m_phiBackupBetterCounter = 0; }

	void setElectronDiffusionConstant(int x, int y, double D)						{ setValue(m_De, x, y, D); }
	void setElectronDiffusionConstant(int x0, int x1, int y0, int y1, double D)				{ setValues(m_De, x0, x1, y0, y1, D); }
	void setElectronDiffusionConstant(double D)								{ setValues(m_De, D); }

	void setHoleDiffusionConstant(int x, int y, double D)							{ setValue(m_Dh, x, y, D); }
	void setHoleDiffusionConstant(int x0, int x1, int y0, int y1, double D)					{ setValues(m_Dh, x0, x1, y0, y1, D); }
	void setHoleDiffusionConstant(double D)									{ setValues(m_Dh, D); }

	void setElectronMobility(int x, int y, double mu)							{ setValue(m_eMob, x, y, mu); }
	void setElectronMobility(int x0, int x1, int y0, int y1, double mu)					{ setValues(m_eMob, x0, x1, y0, y1, mu); }
	void setElectronMobility(double mu)									{ setValues(m_eMob, mu); }

	void setHoleMobility(int x, int y, double mu)								{ setValue(m_hMob, x, y, mu); }
	void setHoleMobility(int x0, int x1, int y0, int y1, double mu)						{ setValues(m_hMob, x0, x1, y0, y1, mu); }
	void setHoleMobility(double mu)										{ setValues(m_hMob, mu); }

	void setRelativePermittivity(int x, int y, double epsRel)						{ setValue(m_epsRels[0], x, y, epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }
	void setRelativePermittivity(int x0, int x1, int y0, int y1, double epsRel)				{ setValues(m_epsRels[0], x0, x1, y0, y1, epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }
	void setRelativePermittivity(double epsRel)								{ setValues(m_epsRels[0], epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }

	void setPotential(int x, int y, double v)								{ setValue(m_potential[0], x, y, v); setValue(m_backupPotential, x, y, v); m_phiBackupBetterCounter = 0; }
	void setPotential(int x0, int x1, int y0, int y1, double v)						{ setValues(m_potential[0], x0, x1, y0, y1, v); setValues(m_backupPotential, x0, x1, y0, y1, v); m_phiBackupBetterCounter = 0; }
	void setPotential(double v)										{ setValues(m_potential[0], v); setValues(m_backupPotential, v); m_phiBackupBetterCounter = 0; }

	void setExtraElectronPotential(int x, int y, double v)							{ setValue(m_extraPotentialN, x, y, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(int x0, int x1, int y0, int y1, double v)				{ setValues(m_extraPotentialN, x0, x1, y0, y1, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(double v)								{ setValues(m_extraPotentialN, v); m_extraPotentialChanged = true; }

	void setExtraHolePotential(int x, int y, double v)							{ setValue(m_extraPotentialP, x, y, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(int x0, int x1, int y0, int y1, double v)					{ setValues(m_extraPotentialP, x0, x1, y0, y1, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(double v)									{ setValues(m_extraPotentialP, v); m_extraPotentialChanged = true; }

	bool start(int steps, double dt, bool inverseMatrixSolver);
	bool start(int seconds, double dt, int &steps, bool inverseMatrixSolver);
	
	void calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const;
	void calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const;

	void getElectronNumberDensity(std::vector<double> &dst) const						{ getProperty(m_n, dst); }
	void getHoleNumberDensity(std::vector<double> &dst) const						{ getProperty(m_p, dst); }
	void getBackgroundNumberDensity(std::vector<double> &dst) const						{ getProperty(m_background, dst); }
	void getGenerationRate(std::vector<double> &dst) const							{ getProperty(m_generationRate, dst); }
	void getRecombinationFactor(std::vector<double> &dst) const						{ getProperty(m_recombinationFactor, dst); }
	void getPotential(std::vector<double> &dst) const							{ getProperty(m_potential[0], dst); }
	void getElectronDiffusionConstant(std::vector<double> &dst) const					{ getProperty(m_De, dst); }
	void getHoleDiffusionConstant(std::vector<double> &dst) const						{ getProperty(m_Dh, dst); }
	void getElectronMobility(std::vector<double> &dst) const						{ getProperty(m_eMob, dst); }
	void getHoleMobility(std::vector<double> &dst) const							{ getProperty(m_hMob, dst); }
	void getRelativePermittivity(std::vector<double> &dst) const						{ getProperty(m_epsRels[0], dst); }
	void getTotalElectronXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEx, dst); }
	void getTotalElectronYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEy, dst); }
	void getTotalHoleXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHx, dst); }
	void getTotalHoleYCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHy, dst); }
	void getExtraElectronPotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialN, dst); }
	void getExtraHolePotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialP, dst); }

	double getPotentialDifference() const									{ return m_deltaPhi; }

	void writePlotData(FILE *pFile);
private:
	int getIndex(int x, int y) const									{ return x+y*m_width; } 
	void setValue(std::vector<double> &grid, int x, int y, double v)					{ grid[getIndex(x,y)] = v; }
	void setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v);
	void setValues(std::vector<double> &grid, double v)							{ setValues(grid, 0, m_width, 0, m_height, v); }
	void getProperty(const std::vector<double> &grid, std::vector<double> &dst) const			{ dst = grid; }
	double blackRed(double *pSrc, int aIndex, int width, int height, int steps, double w);
	double getSGCurrent(double v, double D, double delta, double n1, double n2);
	void resizeArrays();
	void initializePotentialFinder();
	void potentialFinder();
	void calcNumberCurrents();
	void updateDensities(double dt);

	bool m_init;
	bool m_initPotential;
	bool m_epsChanged;
	int m_width, m_height; // ok
	int m_totalPixels;
	int m_multiScaleSteps; // ok
	
	double m_pixelWidth2, m_pixelHeight2; // ok

	std::vector<double> m_n, m_p, m_background; // ok
	std::vector<double> m_generationRate, m_recombinationFactor; // ok
	std::vector<double> m_De, m_Dh, m_eMob, m_hMob; // ok
	
	// epsRels[0] ok
	// potential[0], potential[m_multiScaleSteps-1] ok
	std::vector<std::vector<double> > m_chargeSums, m_epsRels, m_potential;
	std::vector<std::vector<double> > m_a0s, m_a1s, m_a2s, m_a3s, m_a4s;
	std::vector<double> m_backupPotential; // ok

	std::vector<double> m_electicFieldx, m_electicFieldy; // ok

	std::vector<double> m_numCurTotEx, m_numCurTotEy, m_numCurTotHx, m_numCurTotHy; // ok
	std::vector<double> m_dndt, m_dpdt; // ok

	std::vector<double> m_extraPotentialN; // ok
	std::vector<double> m_extraPotentialP; // ok
	std::vector<double> m_extraElecticFieldNx, m_extraElecticFieldNy; 
	std::vector<double> m_extraElecticFieldPx, m_extraElecticFieldPy; 
	bool m_extraPotentialChanged;

	double m_deltaPhi; // ok

	int m_phiBackupBetterCounter;
	int m_phiMethodThreshold;
	int m_phiMethodAdditionalCheckCount;

	InverseMatrixPoissonSolver2D *m_pPoissonSolver;
	bool m_epsChangedForSolver;
};

inline void Simulation2DDouble::setValues(std::vector<double> &grid, int x0, int x1, int y0, int y1, double v)
{
	if (x1 < x0)
		return;
	if (y1 < y0)
		return;

	for (int y = y0 ; y != y1 ; y++)
		for (int x = x0 ; x != x1 ; x++)
			setValue(grid, x, y, v);
}

#endif // SIMULATION2DDOUBLE_H

