#include "simulation2ddouble.h"
#include "simulationstate.h"
#include "inversematrixpoissonsolver2d.h"
#include <serut/fileserializer.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <iostream>
#include <cmath>

using namespace serut;

// NOTE: All units are SI units. Shouldn't be a problem for 'double' representation of real numbers.

//#define CHECK(value, idx) checkArray(value, idx%m_width, idx/m_width, #value, __LINE__)
#define CHECK(value, idx) 

#ifndef WIN32
inline void checkArray(double val, int x, int y, const char *pName, int lineNumber)
{
	if (std::isnan(val))
	{
		if (x >= 0 && y >= 0)
			std::cerr << "Detected NaN " << pName << " at location (" << x << ", " << y << ") on line " << lineNumber << std::endl;
		else
			std::cerr << "Detected NaN " << pName << " on line " << lineNumber << std::endl;
		exit(-1);
	}
}
#endif // !WIN32

#define LIM 1e-10

inline double Simulation2DDouble::getSGCurrent(double v, double D, double delta, double n1, double n2)
{
	double vDelta = v*delta;
	double x = vDelta/D;
	double j = 0;

	if (x < -LIM)
	{
		double factor = std::exp(x);
		double factorMinusOne = factor - 1.0f;
		double n1factor = n1*factor;
		double n1factorMinusN2 = n1factor - n2;
		double n1factorMinusN2OverFactorMinusOne = n1factorMinusN2/factorMinusOne;

		j = v*n1factorMinusN2OverFactorMinusOne;
	}
	else if (x > LIM)
	{
		double factor = std::exp(-x);
		double n2factor = n2*factor;
		double oneMinusFactor = 1.0f-factor;
		double n1MinusN2Factor = n1-n2factor;
		double n1MinusN2FactorOverOneMinusFactor = n1MinusN2Factor/oneMinusFactor;

		j = v*n1MinusN2FactorOverOneMinusFactor;

	}
	else
	{
		x /= 2.0f;

		double twoD = 2.0f*D;
		double factor = twoD/delta;
		double n1MinusN2 = n1-n2;
		double n1PlusN2 = n1+n2;
		double term1 = 0.5f*n1MinusN2;
		double term2Part = 0.5f*n1PlusN2;
		double term2 = term2Part*x;
		double xSquared = x*x;
		double term3Part = n1MinusN2*xSquared;
		double term3 = term3Part/6.0f;

		double seriesPart = term1 + term2;
		double series = seriesPart + term3;

		j = factor*series;
	}

	return j;
}

Simulation2DDouble::Simulation2DDouble()
{
	m_init = false;

	m_width = 0;
	m_height = 0;
	m_totalPixels = 0;
	m_pixelWidth2 = 0;
	m_pixelHeight2 = 0;
	m_deltaPhi = 0;

	m_phiMethodThreshold = 100;
	m_phiBackupBetterCounter = 0;
	m_phiMethodAdditionalCheckCount = 1000;

	m_pPoissonSolver = 0;
	m_epsChangedForSolver = true;
}

Simulation2DDouble::~Simulation2DDouble()
{
	if (m_pPoissonSolver)
		delete m_pPoissonSolver;
}

// This function initializes the simulation. The actual width will be lowestWidth * 2^(scaleSteps-1),
// the actual height will be lowestHeight * 2^(scaleSteps-1). These two values will be stored in
// pNumX and pNumY.
// 
// The reason for this way of specifying width and height is the routine which calculates the
// potential over the grid. This so called multigrid method will first estimate the potential on
// the coarsest grid. This estimate will the be interpolated to provide a starting point to look
// for the potential on a grid of higher resolution. Note that only for calculating the potential
// such a multigrid method is used. The drift/diffusion equations and continuity equations are
// processed on the highest resolution grid.
//
// 'realHeight' is the actual height of the simulation.

bool Simulation2DDouble::init(int lowestWidth, int lowestHeight, double realWidth, double realHeight, 
		               int scaleSteps, int *pNumX, int *pNumY)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (lowestWidth < 2 || lowestHeight < 3)
	{
		setErrorString("Invalid grid dimensions");
		return false;
	}

	if (scaleSteps < 1)
	{
		setErrorString("Invalid number of multigrid steps");
		return false;
	}

	// Calculate the number of pixels on the highest resolution grid.

	int multiplier = 1 << (scaleSteps-1); // bitshift to accomplish 2^(scaleSteps-1)
	int xPixels = lowestWidth*multiplier;
	int yPixels = lowestHeight*multiplier;

	*pNumX = xPixels;
	*pNumY = yPixels;

	m_multiScaleSteps = scaleSteps;

	m_width = xPixels;
	m_height = yPixels;
	m_totalPixels = m_width*m_height;

	m_pixelWidth2 = realWidth/(double)xPixels;
	m_pixelHeight2 = realHeight/(double)(yPixels-1);

	// Allocate enough memory for various arrays
	resizeArrays();

	m_deltaPhi = 0;

	m_init = true;
	m_initPotential = true;
	m_epsChanged = true;
	m_epsChangedForSolver = true;
	m_extraPotentialChanged = true;

	m_pPoissonSolver = 0;
	return true;
}

#define SIMULATION2DDOUBLE_STARTID	0x494d4953
#define SIMULATION2DDOUBLE_VERSION	1
#define SIMULATION2DDOUBLE_ENDID	0x444e4f43

bool Simulation2DDouble::write(SerializationInterface &s) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninitialized simulation");
		return false;
	}

	s.writeInt32(SIMULATION2DDOUBLE_STARTID);
	s.writeInt32(SIMULATION2DDOUBLE_VERSION);

	s.writeInt32(m_width);
	s.writeInt32(m_height);
	s.writeInt32(m_multiScaleSteps);
	s.writeDouble(m_pixelWidth2);
	s.writeDouble(m_pixelHeight2);

	s.writeDoubles(m_n);
	s.writeDoubles(m_p);
	s.writeDoubles(m_background);

	s.writeDoubles(m_generationRate);
	s.writeDoubles(m_recombinationFactor);
	s.writeDoubles(m_De);
	s.writeDoubles(m_Dh);
	s.writeDoubles(m_eMob);
	s.writeDoubles(m_hMob);

	s.writeDoubles(m_epsRels[0]);
	s.writeDoubles(m_backupPotential);
	s.writeDouble(m_deltaPhi);

	s.writeDoubles(m_potential[0]);
	s.writeDoubles(m_potential[m_multiScaleSteps-1]);
	s.writeDoubles(m_electicFieldx);
	s.writeDoubles(m_electicFieldy);
	s.writeDoubles(m_numCurTotEx);
	s.writeDoubles(m_numCurTotEy);
	s.writeDoubles(m_numCurTotHx);
	s.writeDoubles(m_numCurTotHy);
	s.writeDoubles(m_dndt);
	s.writeDoubles(m_dpdt);

	s.writeDoubles(m_extraPotentialN);
	s.writeDoubles(m_extraPotentialP);

	int32_t flag = (m_initPotential)?1:0;

	s.writeInt32(flag);

	if (!s.writeInt32(SIMULATION2DDOUBLE_ENDID))
	{
		setErrorString(std::string("Error writing to stream: ") + s.getErrorString());
		return false;
	}

	return true;
}

bool Simulation2DDouble::read(SerializationInterface &s)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	int32_t tmp;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file start identifier: ") + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DDOUBLE_STARTID)
	{
		setErrorString("Read an invalid data file start identifier");
		return false;
	}
	if (!s.readInt32(&tmp))
	{
		setErrorString("Unable to read file format version: " + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION2DDOUBLE_VERSION)
	{
		setErrorString("Incompatible file format version");
		return false;
	}

	int32_t w, h, steps;
	double pixelWidth, pixelHeight;

	if (!(s.readInt32(&w) && s.readInt32(&h) && s.readInt32(&steps) && 
	      s.readDouble(&pixelWidth) && s.readDouble(&pixelHeight)))
	{
		setErrorString("Error reading grid dimensions");
		return false;
	}

	m_width = w;
	m_height = h;
	m_multiScaleSteps = steps;
	m_pixelWidth2 = pixelWidth;
	m_pixelHeight2 = pixelHeight;

	int multiplier = 1 << (m_multiScaleSteps-1);
	int lowestWidth = m_width/multiplier;
	int lowestHeight = m_height/multiplier;

	if (!(lowestWidth*multiplier == m_width && lowestHeight*multiplier == m_height))
	{
		setErrorString("Detected inconsistent dimension settings");
		return false;
	}

	m_totalPixels = m_width*m_height;
	
	resizeArrays();

	s.readDoubles(m_n);
	s.readDoubles(m_p);
	s.readDoubles(m_background);

	s.readDoubles(m_generationRate);
	s.readDoubles(m_recombinationFactor);
	s.readDoubles(m_De);
	s.readDoubles(m_Dh);
	s.readDoubles(m_eMob);
	s.readDoubles(m_hMob);

	s.readDoubles(m_epsRels[0]);
	s.readDoubles(m_backupPotential);
	s.readDouble(&m_deltaPhi);

	s.readDoubles(m_potential[0]);
	s.readDoubles(m_potential[m_multiScaleSteps-1]);
	s.readDoubles(m_electicFieldx);
	s.readDoubles(m_electicFieldy);
	s.readDoubles(m_numCurTotEx);
	s.readDoubles(m_numCurTotEy);
	s.readDoubles(m_numCurTotHx);
	s.readDoubles(m_numCurTotHy);
	s.readDoubles(m_dndt);
	s.readDoubles(m_dpdt);

	s.readDoubles(m_extraPotentialN);
	s.readDoubles(m_extraPotentialP);

	s.readInt32(&tmp);
	if (tmp == 0)
		m_initPotential = false;
	else
		m_initPotential = true;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file end identifier: ") + s.getErrorString());
		return false;
	}

	if (tmp != SIMULATION2DDOUBLE_ENDID)
	{
		setErrorString("Read an invalid data file end identifier");
		return false;
	}

	m_init = true;
	m_epsChanged = true;
	m_epsChangedForSolver = true;
	m_extraPotentialChanged = true;
	m_phiBackupBetterCounter = 0;

	m_pPoissonSolver = 0;
	
	return true;
}

bool Simulation2DDouble::save(const std::string &fileName) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninialized simulation");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::WriteOnly))
	{
		setErrorString(std::string("Couldn't save to file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return write(fSer);
}

bool Simulation2DDouble::load(const std::string &fileName)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::ReadOnly))
	{
		setErrorString(std::string("Couldn't load file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return read(fSer);
}

// NOTE: functions for the initialization of electron and hole densities, 
//       mobilities, diffusion constants etc can be found in the
//       simulation.h file

bool Simulation2DDouble::start(int steps, double dt, bool inverseMatrixSolver)
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}
	
	if (!inverseMatrixSolver)
		initializePotentialFinder();
	else
	{
		if (!m_pPoissonSolver)
		{
			m_pPoissonSolver = new InverseMatrixPoissonSolver2D();

			if (!m_pPoissonSolver->init(m_width, m_height, (double)(m_width)*m_pixelWidth2, (double)(m_height-1)*m_pixelHeight2,
						    &(m_potential[0][0])))
			{
				setErrorString("Couldn't initialize inverse matrix potential solver: " + m_pPoissonSolver->getErrorString());
				delete m_pPoissonSolver;
				m_pPoissonSolver = 0;
			}
		}

		if (m_epsChangedForSolver)
		{
			if (!m_pPoissonSolver->setRelativePermittivity(&(m_epsRels[0][0])))
			{
				setErrorString("Couldn't set relative permittivity in solver: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			m_epsChangedForSolver = false;
		}
		if (!m_pPoissonSolver->setPotentialDifference(m_deltaPhi))
		{
			setErrorString("Couldn't set potential difference: " + m_pPoissonSolver->getErrorString());
			return false;
		}
	}

	// The actual simulation consists of a repitition of the following three functions.
	// 1) First, given the current distrubution of electrons, holes and possibly a fixed
	//    background (number of negatively charged acceptors or positively charged donors
	//    for example), and taking into account the specified potential difference between 
	//    top and bottom, the potential is calculated at each point, and the electric field
	//    is calculated as the gradient of this potential.
	// 2) Next, the number currents of electrons and holes are calculated (both diffusion
	//    current and current due to the electric field).
	// 3) When these currents are known, dn/dt and dp/dt can be calculated, and using the
	//    specified timestep 'dt', the densities at each point can be updated.

	for (int N = 0 ; N < steps ; N++)
	{
		if (!inverseMatrixSolver)
			potentialFinder();
		else
		{
			if (!m_pPoissonSolver->findPotential(&(m_p[0]), &(m_n[0]), &(m_background[0])))
			{
				setErrorString("Couldn't solve for potential: " + m_pPoissonSolver->getErrorString());
				return false;
			}

			for (int y = 0 ; y < m_height ; y++)
			{
				for (int x = 0 ; x < m_width ; x++)
				{
					int idx = getIndex(x, y);
					int nextX = (x+1)%m_width;
					int nextIdx = getIndex(nextX, y);

					m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth2;
				}
			}

			for (int x = 0 ; x < m_width ; x++)
			{
				for (int y = 0 ; y < m_height-1 ; y++)
				{	
					int idx = getIndex(x, y);
					int nextIdx = getIndex(x, y+1);

					m_electicFieldy[idx] = -(m_potential[0][nextIdx]-m_potential[0][idx])/m_pixelHeight2;
				}

				// NOTE: the last row is not needed (and cannot be calculated by the way)
			}
		}

		calcNumberCurrents();

		updateDensities(dt);
	}

	return true;
}

bool Simulation2DDouble::start(int seconds, double dt, int &steps, bool inverseMatrixSolver)
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}
	
	if (!inverseMatrixSolver)
		initializePotentialFinder();
	else
	{
		if (!m_pPoissonSolver)
		{
			m_pPoissonSolver = new InverseMatrixPoissonSolver2D();

			if (!m_pPoissonSolver->init(m_width, m_height, (double)(m_width)*m_pixelWidth2, (double)(m_height-1)*m_pixelHeight2,
						    &(m_potential[0][0])))
			{
				setErrorString("Couldn't initialize inverse matrix potential solver: " + m_pPoissonSolver->getErrorString());
				delete m_pPoissonSolver;
				m_pPoissonSolver = 0;
			}
		}

		if (m_epsChangedForSolver)
		{
			if (!m_pPoissonSolver->setRelativePermittivity(&(m_epsRels[0][0])))
			{
				setErrorString("Couldn't set relative permittivity in solver: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			m_epsChangedForSolver = false;
		}
		if (!m_pPoissonSolver->setPotentialDifference(m_deltaPhi))
		{
			setErrorString("Couldn't set potential difference: " + m_pPoissonSolver->getErrorString());
			return false;
		}
	}

	time_t startTime = time(0);
	time_t endTime = startTime;
	int count = 0;
	int maxSteps = steps;

	endTime += seconds;

	do
	{
		if (!inverseMatrixSolver)
			potentialFinder();
		else
		{
			if (!m_pPoissonSolver->findPotential(&(m_p[0]), &(m_n[0]), &(m_background[0])))
			{
				setErrorString("Couldn't solve for potential: " + m_pPoissonSolver->getErrorString());
				return false;
			}

			for (int y = 0 ; y < m_height ; y++)
			{
				for (int x = 0 ; x < m_width ; x++)
				{
					int idx = getIndex(x, y);
					int nextX = (x+1)%m_width;
					int nextIdx = getIndex(nextX, y);

					m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth2;
				}
			}

			for (int x = 0 ; x < m_width ; x++)
			{
				for (int y = 0 ; y < m_height-1 ; y++)
				{	
					int idx = getIndex(x, y);
					int nextIdx = getIndex(x, y+1);

					m_electicFieldy[idx] = -(m_potential[0][nextIdx]-m_potential[0][idx])/m_pixelHeight2;
				}

				// NOTE: the last row is not needed (and cannot be calculated by the way)
			}
		}

		calcNumberCurrents();

		updateDensities(dt);

		count++;

	} while (time(0) <= endTime && count < maxSteps);

	steps = count;

	return true;
}

// NOTE: the functions which together perform the multigrid calculation of the potential
//       are specified at the end of the file. 


// Calculate the number currents of electrons and holes
// Note that (x,y) in the grid is stored at array position x+y*width

void Simulation2DDouble::calcNumberCurrents()
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = getIndex(x, y);
			int nextX = (x+1)%m_width; // for periodic boundary conditions
			int nextIdx = getIndex(nextX, y);

#define SCHARFETTERGUMMEL

#ifdef SCHARFETTERGUMMEL
			double De = (m_De[idx]+m_De[nextIdx])/2.0;
			double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
			double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
			double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

			double ve = -eMob*(m_electicFieldx[idx]+m_extraElecticFieldNx[idx]);
			double vh = +hMob*(m_electicFieldx[idx]+m_extraElecticFieldPx[idx]);

			CHECK(De, idx);
			CHECK(Dh, idx);
			CHECK(eMob, idx);
			CHECK(hMob, idx);
			CHECK(ve, idx);
			CHECK(vh, idx);

			m_numCurTotEx[idx] = getSGCurrent(ve, De, m_pixelWidth2, m_n[idx], m_n[nextIdx]);
			m_numCurTotHx[idx] = getSGCurrent(vh, Dh, m_pixelWidth2, m_p[idx], m_p[nextIdx]);

			CHECK(m_numCurTotEx[idx], idx);
			CHECK(m_numCurTotHx[idx], idx);
#else
			double gradNx = (m_n[nextIdx]-m_n[idx])/m_pixelWidth;
			double gradPx = (m_p[nextIdx]-m_p[idx])/m_pixelWidth;
	
			double m_numCurDiffusionEx = -(m_De[idx]+m_De[nextIdx])/2.0*gradNx;
			double m_numCurDiffusionHx = -(m_Dh[idx]+m_Dh[nextIdx])/2.0*gradPx;

			double m_numCurFieldEx = -(m_n[idx]+m_n[nextIdx])/2.0*(m_eMob[idx]+m_eMob[nextIdx])/2.0*m_electicFieldx[idx];
			double m_numCurFieldHx = +(m_p[idx]+m_p[nextIdx])/2.0*(m_hMob[idx]+m_hMob[nextIdx])/2.0*m_electicFieldx[idx];

			m_numCurTotEx[idx] = m_numCurDiffusionEx + m_numCurFieldEx;
			m_numCurTotHx[idx] = m_numCurDiffusionHx + m_numCurFieldHx;
#endif // SCHARFETTERGUMMEL
		}
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{	
			int idx = getIndex(x, y);
			int nextIdx = getIndex(x, y+1);
#ifdef SCHARFETTERGUMMEL
			double De = (m_De[idx]+m_De[nextIdx])/2.0;
			double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
			double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
			double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

			double ve = -eMob*(m_electicFieldy[idx]+m_extraElecticFieldNy[idx]);
			double vh = +hMob*(m_electicFieldy[idx]+m_extraElecticFieldPy[idx]);

			CHECK(De, idx);
			CHECK(Dh, idx);
			CHECK(eMob, idx);
			CHECK(hMob, idx);
			CHECK(ve, idx);
			CHECK(vh, idx);

			m_numCurTotEy[idx] = getSGCurrent(ve, De, m_pixelHeight2, m_n[idx], m_n[nextIdx]);
			m_numCurTotHy[idx] = getSGCurrent(vh, Dh, m_pixelHeight2, m_p[idx], m_p[nextIdx]);

			CHECK(m_numCurTotEy[idx], idx);
			CHECK(m_numCurTotHy[idx], idx);
#else
			double gradNy = (m_n[nextIdx]-m_n[idx])/m_pixelWidth;
			double gradPy = (m_p[nextIdx]-m_p[idx])/m_pixelWidth;
	
			double m_numCurDiffusionEy = -(m_De[idx]+m_De[nextIdx])/2.0*gradNy;
			double m_numCurDiffusionHy = -(m_Dh[idx]+m_Dh[nextIdx])/2.0*gradPy;

			double m_numCurFieldEy = -(m_n[idx]+m_n[nextIdx])/2.0*(m_eMob[idx]+m_eMob[nextIdx])/2.0*m_electicFieldy[idx];
			double m_numCurFieldHy = +(m_p[idx]+m_p[nextIdx])/2.0*(m_hMob[idx]+m_hMob[nextIdx])/2.0*m_electicFieldy[idx];

			m_numCurTotEy[idx] = m_numCurDiffusionEy + m_numCurFieldEy;
			m_numCurTotHy[idx] = m_numCurDiffusionHy + m_numCurFieldHy;
#endif // SCHARFETTERGUMMEL
		}
	}
}

// Based on the currents, update the densities
// NOTE: these are number currents, not charge currents

void Simulation2DDouble::updateDensities(double dt)
{
	// Calculate dn/dt and dp/dt first

	double maxDt = 1.0;

	for (int x = 0 ; x < m_width ; x++)
	{
		int prevX = (x-1+m_width)%m_width; // For periodic boundary conditions

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			int prevY = y-1;
			int idx = getIndex(x, y);
			int leftIdx = getIndex(prevX, y);
			int belowIdx = getIndex(x, prevY);

			double JexCur = m_numCurTotEx[idx];
			double JexPrev = m_numCurTotEx[leftIdx];
			double JeyCur = m_numCurTotEy[idx];
			double JeyPrev = m_numCurTotEy[belowIdx];

			double JhxCur = m_numCurTotHx[idx];
			double JhxPrev = m_numCurTotHx[leftIdx];
			double JhyCur = m_numCurTotHy[idx];
			double JhyPrev = m_numCurTotHy[belowIdx];

			// Calculate the gradients

			double Jexx = (JexCur - JexPrev)/m_pixelWidth2;
			double Jeyy = (JeyCur - JeyPrev)/m_pixelHeight2;
			double Jhxx = (JhxCur - JhxPrev)/m_pixelWidth2;
			double Jhyy = (JhyCur - JhyPrev)/m_pixelHeight2;
			
			// Calculate the rates according to the continuity equation

			double recombination = m_recombinationFactor[idx]*m_p[idx]*m_n[idx];

			m_dndt[idx] = m_generationRate[idx]-recombination-Jexx-Jeyy;
			m_dpdt[idx] = m_generationRate[idx]-recombination-Jhxx-Jhyy;

		}
	}

	// Then, perform the actual update of the densities

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			int idx = getIndex(x, y);

			m_n[idx] += m_dndt[idx]*dt;
			m_p[idx] += m_dpdt[idx]*dt;
		}
	}
}

// Helper function to calculate the current in the X-direction at the left side, the right side
// and averaged over the entire grid

void Simulation2DDouble::calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int y = 1 ; y < m_height-1 ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;

			double JexCur = m_numCurTotEx[idx];
			double JhxCur = m_numCurTotHx[idx];
			
			sum += -JexCur+JhxCur;
		}

		int idx = 0+y*m_width;

		double JexCur = m_numCurTotEx[idx];
		double JhxCur = m_numCurTotHx[idx];

		sum2 += -JexCur + JhxCur;

		idx = m_width-1+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum3 += -JexCur + JhxCur;

		idx = m_width/2+y*m_width;

		JexCur = m_numCurTotEx[idx];
		JhxCur = m_numCurTotHx[idx];

		sum4 += -JexCur + JhxCur;
	}

	sum /= m_width*(m_height-2);
	sum2 /= m_height-2;
	sum3 /= m_height-2;
	sum4 /= m_height-2;

	leftAvg = sum2;
	rightAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

// Helper function to calculate the current in the Y-direction at the bottom side, the top side
// and averaged over the entire grid

void Simulation2DDouble::calculateYCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{
			int idx = x+y*m_width;

			double JeyCur = m_numCurTotEy[idx];
			double JhyCur = m_numCurTotHy[idx];
			
			sum += -JeyCur+JhyCur;
		}

		int idx = x+0*m_width;

		double JeyCur = m_numCurTotEy[idx];
		double JhyCur = m_numCurTotHy[idx];

		sum2 += -JeyCur + JhyCur;

		idx = x+(m_height-2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum3 += -JeyCur + JhyCur;

		idx = x+(m_height/2)*m_width;

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum4 += -JeyCur + JhyCur;

	}

	sum /= m_width*(m_height-1);
	sum2 /= m_width;
	sum3 /= m_width;
	sum4 /= m_width;

	bottomAvg = sum2;
	topAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

// 
// The rest of the file contains functions for calculating the potential. This is
// probably the most complex part of the calculation, mostly because a multigrid
// method is used.
//
// To calculate the potential, the following inputs are used:
//  - The relative permittivity at each grid point
//  - The electron density at each grid point
//  - The hole density at each grid point
//  - A fixed background charge density
//  - Potential at the top contact (bottom contact is assumed to be at 0 V

// This function initializes some parts of the routine which determines the potential
// (if necessary).

void Simulation2DDouble::initializePotentialFinder()
{
	// Initialize the potential at the coarsest scale

	if (m_initPotential)
	{
		m_initPotential = false;

		int multiplier = 1 << (m_multiScaleSteps-1);
		int lowWidth = m_width/multiplier;
		int lowHeight = m_height/multiplier;

		for (int y = 0 ; y < lowHeight ; y++)
		{
			double V = ((double)y/(double)(lowHeight-1))*m_deltaPhi;

			for (int x = 0 ; x < lowWidth ; x++)
			{
				int idx = x+y*lowWidth;

				m_potential[m_multiScaleSteps-1][idx] = V;
			}
		}
		
		m_phiBackupBetterCounter = 0;
	}
	
	// Build scaled arrays of permittivity at each scale

	if (m_epsChanged)
	{
		m_epsChanged = false;

		int w = m_width;
		int h = m_height;

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				for (int y = 0 ; y < h ; y += 2)
				{
					double epsSum = m_epsRels[i][x+y*w] + m_epsRels[i][x+1+y*w] + m_epsRels[i][x+(y+1)*w] + m_epsRels[i][x+1+(y+1)*w];

					m_epsRels[i+1][x/2+(y/2)*w2] = epsSum/4.0; // average it
				}
			}

			w /= 2;
			h /= 2;
		}


		w = m_width;
		h = m_height;

		double dxdyfrac = m_pixelWidth2/m_pixelHeight2;
		double dxdyfracSquared = dxdyfrac*dxdyfrac;

		for (int i = 0 ; i < m_multiScaleSteps ; i++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				for (int x = 0 ; x < w ; x++)
				{
					// Initialize using periodic boundary conditions in both directions

					int xNext = (x+1)%w;
					int xPrev = (x-1+w)%w;
					int yNext = (y+1)%h;
					int yPrev = (y-1+h)%h;

					int index = x+y*w;
					int leftIndex = xPrev+y*w; 
					int rightIndex = xNext+y*w;
					int upIndex = x+yNext*w;
					int downIndex = x+yPrev*w;

					m_a1s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][rightIndex]);
					m_a2s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][leftIndex]);
					m_a3s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][upIndex])*dxdyfracSquared;
					m_a4s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][downIndex])*dxdyfracSquared;
					m_a0s[i][index] = m_a1s[i][index] + m_a2s[i][index] + m_a3s[i][index] + m_a4s[i][index];
				}
			}

			w /= 2;
			h /= 2;
		}
	
		m_phiBackupBetterCounter = 0;
	}

	if (m_extraPotentialChanged)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				int idx = getIndex(x, y);
				int nextX = (x+1)%m_width;
				int nextIdx = getIndex(nextX, y);

				m_extraElecticFieldNx[idx] = -(m_extraPotentialN[nextIdx] - m_extraPotentialN[idx])/m_pixelWidth2;
				m_extraElecticFieldPx[idx] = -(m_extraPotentialP[nextIdx] - m_extraPotentialP[idx])/m_pixelWidth2;
			}
		}

		for (int x = 0 ; x < m_width ; x++)
		{
			for (int y = 0 ; y < m_height-1 ; y++)
			{	
				int idx = getIndex(x, y);
				int nextIdx = getIndex(x, y+1);

				m_extraElecticFieldNy[idx] = -(m_extraPotentialN[nextIdx]-m_extraPotentialN[idx])/m_pixelHeight2;
				m_extraElecticFieldPy[idx] = -(m_extraPotentialP[nextIdx]-m_extraPotentialP[idx])/m_pixelHeight2;
			}

			// NOTE: the last row is not needed (and cannot be calculated by the way)
		}
		m_extraPotentialChanged = false;
	}
}

void Simulation2DDouble::potentialFinder()
{
	if (m_phiBackupBetterCounter < m_phiMethodThreshold || m_phiBackupBetterCounter%m_phiMethodAdditionalCheckCount == 0)
	{
		// First create scaled grids of the charge

		//std::cout << m_phiBackupBetterCounter << " Trying both " << std::endl;

		int w = m_width;
		int h = m_height;

		double chargeMultiplier = CHARGE_ELECTRON*m_pixelWidth2*m_pixelWidth2/CONST_EPSILON0;

		for (int x = 0 ; x < w ; x++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				int idx = x+y*w;

				m_chargeSums[0][idx] = (m_p[idx]-m_n[idx]+m_background[idx])*chargeMultiplier;
			}
		}

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				for (int y = 0 ; y < h ; y += 2)
				{
					double chargeSum = m_chargeSums[i][x+y*w] + m_chargeSums[i][x+1+y*w] + m_chargeSums[i][x+(y+1)*w] + m_chargeSums[i][x+1+(y+1)*w];

					m_chargeSums[i+1][x/2+(y/2)*w2] = chargeSum; // sum it
				}
			}

			w /= 2;
			h /= 2;
		}

		// Set boundary conditions again on the coarsest scale (just to be safe)
		{
			int yOff = (h-1)*w;
			for (int x = 0 ; x < w ; x++)
			{
				m_potential[m_multiScaleSteps-1][x] = 0;
				m_potential[m_multiScaleSteps-1][x+yOff] = m_deltaPhi;
			}
		}

		// Width and height should now contain the dimensions of the coarsest grid
		
		// Calculate the potential for the lowest resolution grid
		double error = blackRed(&(m_potential[m_multiScaleSteps-1][0]), m_multiScaleSteps-1, w, h, 100, 1.8);

		for (int i = m_multiScaleSteps-2 ; i >= 0 ; i--)
		{
			int w2 = w;
			int h2 = h;

			w *= 2;
			h *= 2;

			// Interpolate the result for a first estimate of the higher resolution result

			for (int y = 0 ; y < h ; y++)
			{
				double yFrac = (double)y/(double)(h-1);
				double y1d = yFrac*(h2-1);
				int y1 = (int)y1d;
				int y2 = y1+1;

				if (y2 == h2)
					y2 = h2-1;

				double t = y1d-(double)y1;

				for (int x = 0 ; x < w ; x++)
				{
					double xFrac = (double)x/(double)(w-1);
					double x1d = xFrac*(w2-1);
					int x1 = (int)x1d;
					int x2 = (x1+1)%w2;

					double s = x1d-(double)x1;

					double v1 = m_potential[i+1][x1+y1*w2];
					double v2 = m_potential[i+1][x2+y1*w2];
					double v3 = m_potential[i+1][x1+y2*w2];
					double v4 = m_potential[i+1][x2+y2*w2];

					// Bilinear interpolation
					m_potential[i][x+y*w] = v1*(1.0-t)*(1.0-s) + v2*(1.0-t)*s + v3*t*(1.0-s) + v4*t*s;
				}
			}

			// Set boundary conditions again

			int yOff = (h-1)*w;
			for (int x = 0 ; x < w ; x++)
			{
				m_potential[i][x] = 0;
				m_potential[i][x+yOff] = m_deltaPhi;
			}

			// Optimize at new scale

			error = blackRed(&(m_potential[i][0]), i, w, h, 10, 1.8);
		}

		// We shall also continuously update another potential field using another
		// method

		// Set boundary conditions again on backup potential
		{
			int yOff = (m_height-1)*m_width;
			for (int x = 0 ; x < m_width ; x++)
			{
				m_backupPotential[x] = 0;
				m_backupPotential[x+yOff] = m_deltaPhi;
			}
		}


		double error2 = blackRed(&(m_backupPotential[0]), 0, m_width, m_height, 4, 1.0);

		//std::cerr << "error = " << error << " error2 = " << error2 << std::endl;

		// We'll use the best result of the two approaches
		if (error2 >= error)
		{
			m_phiBackupBetterCounter = 0;
			memcpy(&(m_backupPotential[0]),&(m_potential[0][0]), sizeof(double)*m_totalPixels);
		}
		else
		{
			m_phiBackupBetterCounter++;
			memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
		}
	}
	else
	{
		// First grid of total charge

		int w = m_width;
		int h = m_height;

		double chargeMultiplier = CHARGE_ELECTRON*m_pixelWidth2*m_pixelWidth2/CONST_EPSILON0;

		for (int x = 0 ; x < w ; x++)
		{
			for (int y = 0 ; y < h ; y++)
			{
				int idx = x+y*w;

				m_chargeSums[0][idx] = (m_p[idx]-m_n[idx]+m_background[idx])*chargeMultiplier;
			}
		}

		// Set boundary conditions again on backup potential
		{
			int yOff = (m_height-1)*m_width;
			for (int x = 0 ; x < m_width ; x++)
			{
				m_backupPotential[x] = 0;
				m_backupPotential[x+yOff] = m_deltaPhi;
			}
		}

		blackRed(&(m_backupPotential[0]), 0, m_width, m_height, 4, 1.0);
		m_phiBackupBetterCounter++;
		memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
	}
	
	// Determine electric field from potential field

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = getIndex(x, y);
			int nextX = (x+1)%m_width;
			int nextIdx = getIndex(nextX, y);

			m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth2;
		}
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{	
			int idx = getIndex(x, y);
			int nextIdx = getIndex(x, y+1);

			m_electicFieldy[idx] = -(m_potential[0][nextIdx]-m_potential[0][idx])/m_pixelHeight2;
		}

		// NOTE: the last row is not needed (and cannot be calculated by the way)
	}

}

double Simulation2DDouble::blackRed(double *pSrc, int aIndex, int width, int height, int steps, double w)
{
	double error = -1;

	for (int i = 0 ; i < steps ; i++)
	{
		error = 0;

		// Alternating updates in the same array seems to be much more stable!

		for (int loop = 0 ; loop < 2 ; loop++)
		{
			int yStart = 1;
			int yStop = height-1;

			for (int y = yStart ; y < yStop ; y++)
			{
				int xOff = (y+loop)%2;

				for (int x = xOff ; x < width ; x += 2)
				{
					int xNext = (x+1)%width;
					int xPrev = (x-1+width)%width;
					int yNext = y+1;
					int yPrev = y-1;

					int index = x + y*width;
					int leftIndex = xPrev + y*width;
					int rightIndex = xNext + y*width;
					int upIndex = x + yNext*width;
					int downIndex = x+ yPrev*width;

					double prediction = (m_a1s[aIndex][index]*pSrc[rightIndex] + m_a2s[aIndex][index]*pSrc[leftIndex]
							  +  m_a3s[aIndex][index]*pSrc[upIndex]  + m_a4s[aIndex][index]*pSrc[downIndex]
							  +  m_chargeSums[aIndex][index])/m_a0s[aIndex][index];

					double curValue = pSrc[index];

					double diff = prediction-curValue;

					pSrc[index] += w*diff;

					// update error

					error += diff*diff;
				}
			}
		}
	}

	return std::sqrt(error/(double)(width*height));
}

void Simulation2DDouble::resizeArrays()
{
	m_n.resize(m_totalPixels);
	m_p.resize(m_totalPixels);
	m_background.resize(m_totalPixels);
	m_generationRate.resize(m_totalPixels);
	m_recombinationFactor.resize(m_totalPixels);
	m_De.resize(m_totalPixels);
	m_Dh.resize(m_totalPixels);
	m_eMob.resize(m_totalPixels);
	m_hMob.resize(m_totalPixels);

	int tmpSize = m_totalPixels;

	// More arrays need to be allocated for the multigrid method

	m_epsRels.resize(m_multiScaleSteps);
	m_chargeSums.resize(m_multiScaleSteps);
	m_potential.resize(m_multiScaleSteps);
	m_a0s.resize(m_multiScaleSteps);
	m_a1s.resize(m_multiScaleSteps);
	m_a2s.resize(m_multiScaleSteps);
	m_a3s.resize(m_multiScaleSteps);
	m_a4s.resize(m_multiScaleSteps);

	m_backupPotential.resize(m_totalPixels);

	for (int i = 0 ; i < m_multiScaleSteps ; i++)
	{
		m_epsRels[i].resize(tmpSize);
		m_chargeSums[i].resize(tmpSize);
		m_potential[i].resize(tmpSize);
		m_a0s[i].resize(tmpSize);
		m_a1s[i].resize(tmpSize);
		m_a2s[i].resize(tmpSize);
		m_a3s[i].resize(tmpSize);
		m_a4s[i].resize(tmpSize);
		tmpSize /= 4;
	}

	m_electicFieldx.resize(m_totalPixels);
	m_electicFieldy.resize(m_totalPixels);

	// Initialize the values of some arrays

	setValues(m_n, 0);
	setValues(m_p, 0);
	setValues(m_background, 0);
	setValues(m_generationRate, 0);
	setValues(m_recombinationFactor, 0);
	setValues(m_De, 0);
	setValues(m_Dh, 0);
	setValues(m_eMob, 0);
	setValues(m_hMob, 0);
	setValues(m_epsRels[0], 1);
	setValues(m_background, 0);

	// Allocate memory for the arrays which will contain the number currents
	// due to diffusion and electric field respectively

	m_numCurTotEx.resize(m_totalPixels);
	m_numCurTotEy.resize(m_totalPixels);
	m_numCurTotHx.resize(m_totalPixels);
	m_numCurTotHy.resize(m_totalPixels);

	m_dndt.resize(m_totalPixels);
	m_dpdt.resize(m_totalPixels);

	m_extraPotentialN.resize(m_totalPixels);
	m_extraPotentialP.resize(m_totalPixels);
	m_extraElecticFieldNx.resize(m_totalPixels);
	m_extraElecticFieldNy.resize(m_totalPixels);
	m_extraElecticFieldPx.resize(m_totalPixels);
	m_extraElecticFieldPy.resize(m_totalPixels);

	setValues(m_extraPotentialN, 0);
	setValues(m_extraPotentialP, 0);
}

void Simulation2DDouble::writePlotData(FILE *pFile)
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x + y*m_width;
			double j1 = 0;
			double j2 = 0;

			if (y == m_height-1)
			{
				j1 = m_numCurTotHx[idx-m_width]-m_numCurTotEx[idx-m_width];
				j2 = m_numCurTotHy[idx-m_width]-m_numCurTotEy[idx-m_width];
			}
			else
			{
				j1 = m_numCurTotHx[idx]-m_numCurTotEx[idx];
				j2 = m_numCurTotHy[idx]-m_numCurTotEy[idx];
			}

			fprintf(pFile, "%d %d %g %g %g %g %g %g  %g %g %g %g %g %g %g %g %g %g\n", x, y, 
					         (double)m_n[idx], (double)m_p[idx], (double)m_potential[0][idx], 
					         (double)(m_recombinationFactor[idx]*m_n[idx]*m_p[idx]),
						 (double)j1, (double)j2,
						 (double)m_generationRate[idx], (double)m_recombinationFactor[idx],
						 (double)m_De[idx], (double)m_Dh[idx], (double)m_eMob[idx], (double)m_hMob[idx],
						 m_numCurTotEx[idx], m_numCurTotHx[idx], m_numCurTotEy[idx], m_numCurTotHy[idx]);
		}
		fprintf(pFile, "\n");
	}
	fprintf(pFile, "\n");
}

bool Simulation2DDouble::setState(const SimulationState &state, std::vector<std::string> &warnings)
{
	if (!m_init)
	{
		setErrorString("Simulation not initialized");
		return false;
	}
	if (!state.isInitialized())
	{
		setErrorString("State to be used is not initialized");
		return false;
	}
	if (state.getDimensions() != 2)
	{
		setErrorString("State is not one-dimensional");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels() || state.getNumberOfYPixels() != getNumYPixels())
	{
		setErrorString("State does not have same dimensions as simulation");
		return false;
	}

	std::vector<bool> m_input(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_output(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_optional(SIMSTATE_GRIDPROP_MAX);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		m_input[i] = false;
		m_output[i] = false;
		m_optional[i] = false;
	}

	m_input[SIMSTATE_GRIDPROP_N] = true;
	m_input[SIMSTATE_GRIDPROP_P] = true;
	m_input[SIMSTATE_GRIDPROP_DN] = true;
	m_input[SIMSTATE_GRIDPROP_DP] = true;
	m_input[SIMSTATE_GRIDPROP_NMOB] = true;
	m_input[SIMSTATE_GRIDPROP_PMOB] = true;
	m_input[SIMSTATE_GRIDPROP_EPSREL] = true;

	m_output[SIMSTATE_GRIDPROP_N] = true;
	m_output[SIMSTATE_GRIDPROP_P] = true;
	m_output[SIMSTATE_GRIDPROP_V] = true;
	m_output[SIMSTATE_GRIDPROP_JNX] = true;
	m_output[SIMSTATE_GRIDPROP_JPX] = true;
	m_output[SIMSTATE_GRIDPROP_JNY] = true;
	m_output[SIMSTATE_GRIDPROP_JPY] = true;
	m_output[SIMSTATE_GRIDPROP_R] = true;

	m_optional[SIMSTATE_GRIDPROP_BG] = true;
	m_optional[SIMSTATE_GRIDPROP_G] = true;
	m_optional[SIMSTATE_GRIDPROP_RF] = true;
	m_optional[SIMSTATE_GRIDPROP_VNEXTRA] = true;
	m_optional[SIMSTATE_GRIDPROP_VPEXTRA] = true;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (m_input[i])
		{
			if (!state.isGridPropertySet(i))
			{
				setErrorString("Grid property '" + state.getGridPropertyName(i) + "' is required, but has not been set");
				return false;
			}
		}
		else
		{
			if (state.isGridPropertySet(i))
			{
				if (!m_output[i] && !m_optional[i])
					warnings.push_back("Property '" + state.getGridPropertyName(i) + "' is set but not used");
			}
		}
	}

	int numX = getNumXPixels();
	int numY = getNumYPixels();

	for (int y = 0 ; y < numY ; y++)
	{
		for (int x = 0 ; x < numX ; x++)
		{
			setElectronNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_N, x, y));
			setHoleNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_P, x, y));

			setHoleMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_P, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_BG))
				setBackgroundNumberDensity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_BG, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_G))
				setGenerationRate(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_G, x, y));
			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_RF))
				setRecombinationFactor(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_RF, x, y));

			setElectronDiffusionConstant(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_DN, x, y));
			setHoleDiffusionConstant(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_DP, x, y));
			setElectronMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_NMOB, x, y));
			setHoleMobility(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_PMOB, x, y));
			setRelativePermittivity(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_EPSREL, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_V))
				setPotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_V, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VNEXTRA))
				setExtraElectronPotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, x, y));

			if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VPEXTRA))
				setExtraHolePotential(x, y, state.getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, x, y));
		}
	}

	double vDiff;

	if (!state.getDoubleProperty(SIMSTATE_PROP_VDIFF, vDiff))
	{
		setErrorString("Potential difference has not been set");
		return false;
	}

	setPotentialDifference(vDiff);

	if (state.getRecombinationModel() == SimulationState::Braun)
	{
		setErrorString("Braun recombination model selected, but not supported in this simulation type");
		return false;
	}
	else
	{
		double a, kf;

		if (state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
			warnings.push_back("'Pair distance' parameter is set, but is only used in the braun model");
		if (state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
			warnings.push_back("'Dissociation rate' parameter 'kf' is set, but is only used in the braun model");
	}

	//FILE *f = fopen("log.txt","wt");
	//writePlotData(f);
	//fclose(f);

	m_epsChanged = true;
	m_epsChangedForSolver = true;

	return true;
}

bool Simulation2DDouble::storeState(SimulationState &state) const
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels() ||
	    state.getNumberOfYPixels() != getNumYPixels() )
	{
		setErrorString("Number of pixels is not same as in simulation state");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		state.clearDoubleProperty(i);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		state.clearGridProperty(i);

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_N, m_n) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_P, m_p) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_V, m_potential[0]))
	{
		setErrorString("Unable to store n,p or V");
		return false;
	}

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_BG, m_background) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DN, m_De) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels[0]) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, m_extraPotentialN) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, m_extraPotentialP) )
	{
		setErrorString("Unable to store fixed simulation items");
		return false;
	}

	for (int x = 0 ; x < m_width ; x++)
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			double jnx = m_numCurTotEx[x+y*m_width];
			double jpx = m_numCurTotHx[x+y*m_width];
			double jny = m_numCurTotEy[x+y*m_width];
			double jpy = m_numCurTotHy[x+y*m_width];

			if (!state.setGridProperty(SIMSTATE_GRIDPROP_JNX, x, y, jnx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPX, x, y, jpx) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JNY, x, y, jny) ||
			    !state.setGridProperty(SIMSTATE_GRIDPROP_JPY, x, y, jpy))
			{
				setErrorString("Unable to store the number current");
				return false;
			}
		}
	}

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int idx = x+y*m_width;
			double R = m_recombinationFactor[idx] * (m_n[idx] * m_p[idx]);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, x, y, R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
		}
	}

	state.setRecombinationModel(SimulationState::Simple);	

	if (!state.setDoubleProperty(SIMSTATE_PROP_VDIFF, m_deltaPhi))
	{
		setErrorString("Couldn't store potential difference");
		return false;
	}

	return true;
}


