#ifndef SIMULATION1DDOUBLE_H

#define SIMULATION1DDOUBLE_H

#include "simulation1d.h"

class SimulationState;
class InverseMatrixPoissonSolver1D;

class Simulation1DDouble : public Simulation1D
{
public:
	Simulation1DDouble();
	~Simulation1DDouble();

	// Set grid dimensions
	bool init(int lowestWidth, double realWidth, int scaleSteps, int *pNumX, double T);
	bool isInitialized() const										{ return m_init; }

	int getNumXPixels() const										{ return m_width; }
	double getPixelWidth() const										{ return m_pixelWidth; }

	bool save(const std::string &fileName) const;
	bool load(const std::string &fileName);
	bool write(serut::SerializationInterface &s) const;
	bool read(serut::SerializationInterface &s);

	bool setState(const SimulationState &state, std::vector<std::string> &warnings);
	bool storeState(SimulationState &state) const;

	// After 'init' has been called, these functions can be used to specify a specific
	// starting situation
	void setElectronNumberDensity(int x,double dens)							{ setValue(m_n, x, dens); }
	void setElectronNumberDensity(int x0, int x1, double dens)						{ setValues(m_n, x0, x1, dens); }
	void setElectronNumberDensity(double dens)								{ setValues(m_n, dens); }
	void setElectronNumberDensity(const double *pDens)							{ for (int i = 0 ; i < m_width ; i++) setValue(m_n, i, pDens[i]); }

	void setIntrinsicElectronNumberDensity(int x,double dens)						{ setValue(m_ni, x, dens); }
	void setIntrinsicElectronNumberDensity(int x0, int x1, double dens)					{ setValues(m_ni, x0, x1, dens); }
	void setIntrinsicElectronNumberDensity(double dens)							{ setValues(m_ni, dens); }

	void setHoleNumberDensity(int x, double dens)								{ setValue(m_p, x, dens); }
	void setHoleNumberDensity(int x0, int x1, double dens)							{ setValues(m_p, x0, x1, dens); }
	void setHoleNumberDensity(double dens)									{ setValues(m_p, dens); }
	void setHoleNumberDensity(const double *pDens)								{ for (int i = 0 ; i < m_width ; i++) setValue(m_p, i, pDens[i]); }

	void setBackgroundNumberDensity(int x, double dens)							{ setValue(m_background, x, dens); }
	void setBackgroundNumberDensity(int x0, int x1, double dens)						{ setValues(m_background, x0, x1, dens); }
	void setBackgroundNumberDensity(double dens)								{ setValues(m_background, dens); }

	void setGenerationRate(int x, double rate)								{ setValue(m_generationRate, x, rate); }
	void setGenerationRate(int x0, int x1, double rate)							{ setValues(m_generationRate, x0, x1, rate); }
	void setGenerationRate(double rate)									{ setValues(m_generationRate, rate); }

	// r = factor * n * p;
	void setRecombinationFactor(int x, double factor)							{ setValue(m_recombinationFactor, x, factor); }
	void setRecombinationFactor(int x0, int x1, double factor)						{ setValues(m_recombinationFactor, x0, x1, factor); }
	void setRecombinationFactor(double factor)								{ setValues(m_recombinationFactor, factor); }

	// bottom = 0, top = V
	void setPotentialDifference(double V)									{ m_deltaPhi = V; m_phiBackupBetterCounter = 0; }

	void setElectronDiffusionConstant(int x, double D)							{ setValue(m_De, x, D); }
	void setElectronDiffusionConstant(int x0, int x1, double D)						{ setValues(m_De, x0, x1, D); }
	void setElectronDiffusionConstant(double D)								{ setValues(m_De, D); }

	void setHoleDiffusionConstant(int x, double D)								{ setValue(m_Dh, x, D); }
	void setHoleDiffusionConstant(int x0, int x1, double D)							{ setValues(m_Dh, x0, x1, D); }
	void setHoleDiffusionConstant(double D)									{ setValues(m_Dh, D); }

	void setElectronMobility(int x, double mu)								{ setValue(m_eMob, x, mu); }
	void setElectronMobility(int x0, int x1, double mu)							{ setValues(m_eMob, x0, x1, mu); }
	void setElectronMobility(double mu)									{ setValues(m_eMob, mu); }

	void setHoleMobility(int x, double mu)									{ setValue(m_hMob, x, mu); }
	void setHoleMobility(int x0, int x1, double mu)								{ setValues(m_hMob, x0, x1, mu); }
	void setHoleMobility(double mu)										{ setValues(m_hMob, mu); }

	void setRelativePermittivity(int x, double epsRel)							{ setValue(m_epsRels[0], x, epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }
	void setRelativePermittivity(int x0, int x1, double epsRel)						{ setValues(m_epsRels[0], x0, x1, epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }
	void setRelativePermittivity(double epsRel)								{ setValues(m_epsRels[0], epsRel); m_epsChanged = true; m_epsChangedForSolver = true; }

	void setPotential(int x, double v)									{ setValue(m_potential[0], x, v); setValue(m_backupPotential, x, v); m_phiBackupBetterCounter = 0; m_initPotential = false; }
	void setPotential(int x0, int x1, double v)								{ setValues(m_potential[0], x0, x1, v); setValues(m_backupPotential, x0, x1, v); m_phiBackupBetterCounter = 0; m_initPotential = false; }
	void setPotential(double v)										{ setValues(m_potential[0], v); setValues(m_backupPotential, v); m_phiBackupBetterCounter = 0; m_initPotential = false; }
	void setPotential(const double *pV)									{ for (int i = 0 ; i < m_width ; i++) { setValue(m_potential[0], i, pV[i]); setValue(m_backupPotential, i, pV[i]); } m_initPotential = false; }

	void setExtraElectronPotential(int x, double v)								{ setValue(m_extraPotentialN, x, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(int x0, int x1, double v)						{ setValues(m_extraPotentialN, x0, x1, v); m_extraPotentialChanged = true; }
	void setExtraElectronPotential(double v)								{ setValues(m_extraPotentialN, v); m_extraPotentialChanged = true; }

	void setExtraHolePotential(int x, double v)								{ setValue(m_extraPotentialP, x, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(int x0, int x1, double v)							{ setValues(m_extraPotentialP, x0, x1, v); m_extraPotentialChanged = true; }
	void setExtraHolePotential(double v)									{ setValues(m_extraPotentialP, v); m_extraPotentialChanged = true; }

	void setSurfaceRecombinationSpeed(double s)								{ m_S = s; }

	bool start(int &steps, double dt, double stopCriterion = -1.0, bool inverseMatrixSolver = true);
	bool start(int seconds, double dt, int &steps, double stopCriterion = -1.0, bool inverseMatrixSolver = true);

	void calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const;
	double calculatePavg() const;

	void getElectronNumberDensity(std::vector<double> &dst) const						{ getProperty(m_n, dst); }
	void getIntrinsicElectronNumberDensity(std::vector<double> &dst) const					{ getProperty(m_ni, dst); }
	void getHoleNumberDensity(std::vector<double> &dst) const						{ getProperty(m_p, dst); }
	void getBackgroundNumberDensity(std::vector<double> &dst) const						{ getProperty(m_background, dst); }
	void getGenerationRate(std::vector<double> &dst) const							{ getProperty(m_generationRate, dst); }
	void getRecombinationFactor(std::vector<double> &dst) const						{ getProperty(m_recombinationFactor, dst); }
	void getPotential(std::vector<double> &dst) const							{ getProperty(m_potential[0], dst); }
	void getElectronDiffusionConstant(std::vector<double> &dst) const					{ getProperty(m_De, dst); }
	void getHoleDiffusionConstant(std::vector<double> &dst) const						{ getProperty(m_Dh, dst); }
	void getElectronMobility(std::vector<double> &dst) const						{ getProperty(m_eMob, dst); }
	void getHoleMobility(std::vector<double> &dst) const							{ getProperty(m_hMob, dst); }
	void getRelativePermittivity(std::vector<double> &dst) const						{ getProperty(m_epsRels[0], dst); }
	void getTotalElectronXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEx, dst); }
	void getTotalHoleXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHx, dst); }
	void getDissociationProbability(std::vector<double> &dst) const						{ getProperty(m_P, dst); }
	void getExtraElectronPotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialN, dst); }
	void getExtraHolePotential(std::vector<double> &dst) const						{ getProperty(m_extraPotentialP, dst); }

	double getPotentialDifference() const									{ return m_deltaPhi; }
	double getTemperature() const 										{ return m_T; }

	void writePlotData(FILE *pFile);

	void setExtendedRecombinationModel(double pairDistance, double kf);
	void setNormalRecombinationModel()									{ m_factorPinRG = false; }
	bool getExtendedRecombinationParameters(double &pairDistance, double &kf) const;

	double getUpdateNAvg() const 										{ return m_updateNAvg; }
	double getUpdatePAvg() const 										{ return m_updatePAvg; }
private:
	void setValue(std::vector<double> &grid, int x, double v)						
	{ 
		grid[x] = v; 
		if (&grid == &m_p)
		{
			if (x == 0)
				m_p00 = v;
			else if (x == m_totalPixels-1)
				m_p0d = v;
		}
		else if (&grid == &m_n)
		{
			if (x == 0)
				m_n00 = v;
			else if (x == m_totalPixels-1)
				m_n0d = v;
		}
	}
	void setValues(std::vector<double> &grid, int x0, int x1, double v);
	void setValues(std::vector<double> &grid, double v)							{ setValues(grid, 0, m_width, v); }
	void getProperty(const std::vector<double> &grid, std::vector<double> &dst) const			{ dst = grid; }
	double blackRed(double *pSrc, int aIndex, int width, int steps, double w);
	double getSGCurrent(double v, double D, double delta, double n1, double n2);
	void resizeArrays();
	void initializePotentialFinder();
	void potentialFinder();
	void calcNumberCurrents();
	void updateDensities(double dt);
	void applyConstraints();
	double calculatePFactor(int idx);
	bool resampleArray(const std::vector<double> &src, std::vector<double> &dst, int srcWidth, int dstWidth); 

	bool m_init;
	bool m_initPotential;
	bool m_epsChanged;
	int m_width;
	int m_totalPixels;
	int m_multiScaleSteps; 
	
	double m_pixelWidth; // ok

	std::vector<double> m_n, m_ni, m_p, m_background; // ok
	std::vector<double> m_generationRate, m_recombinationFactor; // ok
	std::vector<double> m_De, m_Dh, m_eMob, m_hMob, m_gammaE, m_gammaH; // ok
	bool m_useFieldDependentMobility;
	
	// epsRels[0] ok
	// potential[0], potential[m_multiScaleSteps-1] ok
	std::vector<std::vector<double> > m_chargeSums, m_epsRels, m_potential;
	std::vector<std::vector<double> > m_a0s, m_a1s, m_a2s;
	std::vector<double> m_backupPotential; // ok

	std::vector<double> m_electicFieldx; // ok
	std::vector<double> m_numCurTotEx, m_numCurTotHx; // ok
	std::vector<double> m_dndt, m_dpdt; // ok

	std::vector<double> m_P;

	std::vector<double> m_extraPotentialN;
	std::vector<double> m_extraPotentialP;
	std::vector<double> m_extraElecticFieldNx;
	std::vector<double> m_extraElecticFieldPx;
	bool m_extraPotentialChanged;

	double m_deltaPhi; // ok

	int m_phiBackupBetterCounter;
	int m_phiMethodThreshold;
	int m_phiMethodAdditionalCheckCount;

	double m_pairDist;
	double m_T;
	double m_kf;
	bool m_factorPinRG;

	double m_bMax;
	std::vector<double> m_precalcBValues;

	double calcBValue(double b);
	void precalcBValues(double curB);

	double m_n00, m_p00, m_n0d, m_p0d;
	double m_S;

	double m_updateNAvg;
	double m_updatePAvg;

	InverseMatrixPoissonSolver1D *m_pPoissonSolver;
	bool m_epsChangedForSolver;
};

inline void Simulation1DDouble::setValues(std::vector<double> &grid, int x0, int x1, double v)
{
	if (x1 < x0)
		return;

	for (int x = x0 ; x != x1 ; x++)
		setValue(grid, x, v);
}

#endif // SIMULATION1DDOUBLE_H

