#include "simulation1ddouble.h"
#include "simulationstate.h"
#include "inversematrixpoissonsolver1d.h"
#include "constants.h"
#include <serut/fileserializer.h>
//#include <gsl/gsl_sf.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <iostream>
#include <cmath>

#include "dissociationprobability.h"

using namespace serut;

// NOTE: All units are SI units. Shouldn't be a problem for 'double' representation of real numbers.

//#define CHECK(value, idx) checkArray(value, idx%m_width, idx/m_width, #value, __LINE__)
#define CHECK(value, idx) 

#ifndef WIN32
inline void checkArray(double val, int x, int y, const char *pName, int lineNumber)
{
	if (std::isnan(val))
	{
		if (x >= 0 && y >= 0)
			std::cerr << "Detected NaN " << pName << " at location (" << x << ", " << y << ") on line " << lineNumber << std::endl;
		else
			std::cerr << "Detected NaN " << pName << " on line " << lineNumber << std::endl;
		exit(-1);
	}
}
#endif // !WIN32

inline double Simulation1DDouble::getSGCurrent(double v, double D, double delta, double n1, double n2)
{
	double x = v*delta/D;
	double j = 0;

	CHECK(D, -1);
	CHECK(delta, -1);
	CHECK(n1, -1);
	CHECK(n2, -1);
	CHECK(v, -1);
	CHECK(x, -1);

#define LIM 1e-10

	if (x < -LIM)
	{
		double factor = std::exp(x);

		CHECK(factor, -1);

		j = v*(n1*factor-n2)/(factor-1.0);
	}
	else if (x > LIM)
	{
		double factor = std::exp(-x);

		CHECK(factor, -1);

		j = v*(n1-factor*n2)/(1.0-factor);
	}
	else
	{

		x /= 2.0;

		double factor = 2.0*D/delta;

		CHECK(factor, -1);

		double series = 0.5*(n1-n2) + 0.5*(n1+n2)*x + (n1-n2)*x*x/6.0;

		CHECK(series, -1);

		j = factor*series;
	}

	return j;
}


Simulation1DDouble::Simulation1DDouble()
{
	m_init = false;

	m_width = 0;
	m_totalPixels = 0;
	m_pixelWidth = 0;
	m_deltaPhi = 0;

	m_phiMethodThreshold = 100;
	m_phiBackupBetterCounter = 0;
	m_phiMethodAdditionalCheckCount = 1000;

	m_factorPinRG = false;

	m_bMax = -1;

	m_S = -1;
	m_n00 = 0;
	m_p00 = 0;
	m_n0d = 0;
	m_p0d = 0;

	m_epsChangedForSolver = true;
	m_pPoissonSolver = 0;
	m_useFieldDependentMobility = false;
}

Simulation1DDouble::~Simulation1DDouble()
{
	if (m_pPoissonSolver)
		delete m_pPoissonSolver;
}

// This function initializes the simulation. The actual width will be lowestWidth * 2^(scaleSteps-1),
// the actual height will be lowestHeight * 2^(scaleSteps-1). These two values will be stored in
// pNumX and pNumY.
// 
// The reason for this way of specifying width and height is the routine which calculates the
// potential over the grid. This so called multigrid method will first estimate the potential on
// the coarsest grid. This estimate will the be interpolated to provide a starting point to look
// for the potential on a grid of higher resolution. Note that only for calculating the potential
// such a multigrid method is used. The drift/diffusion equations and continuity equations are
// processed on the highest resolution grid.
//
// 'realHeight' is the actual height of the simulation.

bool Simulation1DDouble::init(int lowestWidth, double realWidth, int scaleSteps, int *pNumX, double T)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (lowestWidth < 3)
	{
		setErrorString("Invalid grid dimensions");
		return false;
	}

	if (scaleSteps < 1)
	{
		setErrorString("Invalid number of multigrid steps");
		return false;
	}

	// Calculate the number of pixels on the highest resolution grid.

	int multiplier = 1 << (scaleSteps-1); // bitshift to accomplish 2^(scaleSteps-1)
	int xPixels = lowestWidth*multiplier;

	*pNumX = xPixels;

	m_multiScaleSteps = scaleSteps;

	m_width = xPixels;
	m_totalPixels = m_width;

	m_pixelWidth = realWidth/(double)(xPixels-1);

	// Allocate enough memory for various arrays
	resizeArrays();

	m_deltaPhi = 0;

	m_init = true;
	m_initPotential = true;
	m_epsChanged = true;
	m_epsChangedForSolver = true;
	m_T = T;
	m_extraPotentialChanged = true;

	m_pPoissonSolver = 0;
	m_useFieldDependentMobility = false;

	return true;
}

#define SIMULATION1DDOUBLE_STARTID	0x494d3153
#define SIMULATION1DDOUBLE_VERSION	3
#define SIMULATION1DDOUBLE_ENDID	0x444e4f43

bool Simulation1DDouble::write(SerializationInterface &s) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninitialized simulation");
		return false;
	}

	if (m_S > 0)
	{
		setErrorString("Not all parameters needed can be saved");
		return false;
	}

	s.writeInt32(SIMULATION1DDOUBLE_STARTID);
	s.writeInt32(SIMULATION1DDOUBLE_VERSION);

	s.writeInt32(m_width);
	s.writeInt32(m_multiScaleSteps);
	s.writeDouble(m_pixelWidth);

	s.writeDoubles(m_n);
	s.writeDoubles(m_ni);
	s.writeDoubles(m_p);
	s.writeDoubles(m_background);

	s.writeDoubles(m_generationRate);
	s.writeDoubles(m_recombinationFactor);
	s.writeDoubles(m_De);
	s.writeDoubles(m_Dh);
	s.writeDoubles(m_eMob);
	s.writeDoubles(m_hMob);

	s.writeDoubles(m_epsRels[0]);
	s.writeDoubles(m_backupPotential);
	s.writeDouble(m_deltaPhi);

	s.writeDoubles(m_potential[0]);
	s.writeDoubles(m_potential[m_multiScaleSteps-1]);
	s.writeDoubles(m_electicFieldx);
	s.writeDoubles(m_numCurTotEx);
	s.writeDoubles(m_numCurTotHx);
	s.writeDoubles(m_dndt);
	s.writeDoubles(m_dpdt);

	s.writeDoubles(m_extraPotentialN);
	s.writeDoubles(m_extraPotentialP);

	int32_t flag = (m_initPotential)?1:0;

	s.writeInt32(flag);

	flag = (m_factorPinRG)?1:0;
	s.writeInt32(flag);

	s.writeDouble(m_T);
	s.writeDouble(m_pairDist);
	s.writeDouble(m_kf);
	s.writeDoubles(m_P);

	if (!s.writeInt32(SIMULATION1DDOUBLE_ENDID))
	{
		setErrorString(std::string("Error writing to stream: ") + s.getErrorString());
		return false;
	}

	return true;
}

bool Simulation1DDouble::read(SerializationInterface &s)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	int32_t tmp;

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file start identifier: ") + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION1DDOUBLE_STARTID)
	{
		setErrorString("Read an invalid data file start identifier");
		return false;
	}
	if (!s.readInt32(&tmp))
	{
		setErrorString("Unable to read file format version: " + s.getErrorString());
		return false;
	}
	if (tmp != SIMULATION1DDOUBLE_VERSION)
	{
		setErrorString("Incompatible file format version");
		return false;
	}

	int32_t w, steps;
	double pixelWidth;

	if (!(s.readInt32(&w) && s.readInt32(&steps) && 
	      s.readDouble(&pixelWidth)))
	{
		setErrorString("Error reading grid dimensions");
		return false;
	}

	m_width = w;
	m_multiScaleSteps = steps;
	m_pixelWidth = pixelWidth;

	int multiplier = 1 << (m_multiScaleSteps-1);
	int lowestWidth = m_width/multiplier;

	if (!(lowestWidth*multiplier == m_width))
	{
		setErrorString("Detected inconsistent dimension settings");
		return false;
	}

	m_totalPixels = m_width;
	
	resizeArrays();

	s.readDoubles(m_n);
	s.readDoubles(m_ni);
	s.readDoubles(m_p);
	s.readDoubles(m_background);

	s.readDoubles(m_generationRate);
	s.readDoubles(m_recombinationFactor);
	s.readDoubles(m_De);
	s.readDoubles(m_Dh);
	s.readDoubles(m_eMob);
	s.readDoubles(m_hMob);

	s.readDoubles(m_epsRels[0]);
	s.readDoubles(m_backupPotential);
	s.readDouble(&m_deltaPhi);

	s.readDoubles(m_potential[0]);
	s.readDoubles(m_potential[m_multiScaleSteps-1]);
	s.readDoubles(m_electicFieldx);
	s.readDoubles(m_numCurTotEx);
	s.readDoubles(m_numCurTotHx);
	s.readDoubles(m_dndt);
	s.readDoubles(m_dpdt);

	s.readDoubles(m_extraPotentialN);
	s.readDoubles(m_extraPotentialP);

	s.readInt32(&tmp);
	if (tmp == 0)
		m_initPotential = false;
	else
		m_initPotential = true;

	s.readInt32(&tmp);
	if (tmp == 0)
		m_factorPinRG = false;
	else
		m_factorPinRG = true;

	s.readDouble(&m_T);
	s.readDouble(&m_pairDist);
	s.readDouble(&m_kf);
	s.readDoubles(m_P);

	if (!s.readInt32(&tmp))
	{
		setErrorString(std::string("Error reading data file end identifier: ") + s.getErrorString());
		return false;
	}

	if (tmp != SIMULATION1DDOUBLE_ENDID)
	{
		setErrorString("Read an invalid data file end identifier");
		return false;
	}

	m_init = true;
	m_epsChanged = true;
	m_epsChangedForSolver = true;
	m_phiBackupBetterCounter = 0;
	m_extraPotentialChanged = true;
	
	m_bMax = -1;
	m_S = -1;
	m_n00 = 0;
	m_p00 = 0;
	m_n0d = 0;
	m_p0d = 0;

	return true;
}

bool Simulation1DDouble::save(const std::string &fileName) const
{
	if (!m_init)
	{
		setErrorString("Can't save an uninialized simulation");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::WriteOnly))
	{
		setErrorString(std::string("Couldn't save to file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return write(fSer);
}

bool Simulation1DDouble::load(const std::string &fileName)
{
	if (m_init)
	{
		setErrorString("The simulation has already been initialized, clear it before loading other data in it");
		return false;
	}

	FileSerializer fSer;

	if (!fSer.open(fileName, FileSerializer::ReadOnly))
	{
		setErrorString(std::string("Couldn't load file ") + fileName + std::string(": ") + fSer.getErrorString());
		return false;
	}

	return read(fSer);
}


// NOTE: functions for the initialization of electron and hole densities, 
//       mobilities, diffusion constants etc can be found in the
//       simulation.h file

bool Simulation1DDouble::start(int &steps, double dt, double stopCriterion, bool inverseMatrixSolver)
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}
	
	if (!inverseMatrixSolver)
		initializePotentialFinder();
	else
	{
		if (!m_pPoissonSolver)
		{
			m_pPoissonSolver = new InverseMatrixPoissonSolver1D();

			if (!m_pPoissonSolver->init(m_width, (double)(m_width-1)*m_pixelWidth, &(m_potential[0][0])))
			{
				setErrorString("Couldn't initialize inverse matrix potential solver: " + m_pPoissonSolver->getErrorString());
				delete m_pPoissonSolver;
				m_pPoissonSolver = 0;
			}
		}

		if (m_epsChangedForSolver)
		{
			if (!m_pPoissonSolver->setRelativePermittivity(&(m_epsRels[0][0])))
			{
				setErrorString("Couldn't set relative permittivity in solver: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			m_epsChangedForSolver = false;
		}
		if (!m_pPoissonSolver->setPotentialDifference(m_deltaPhi))
		{
			setErrorString("Couldn't set potential difference: " + m_pPoissonSolver->getErrorString());
			return false;
		}
	}

	// The actual simulation consists of a repitition of the following three functions.
	// 1) First, given the current distrubution of electrons, holes and possibly a fixed
	//    background (number of negatively charged acceptors or positively charged donors
	//    for example), and taking into account the specified potential difference between 
	//    top and bottom, the potential is calculated at each point, and the electric field
	//    is calculated as the gradient of this potential.
	// 2) Next, the number currents of electrons and holes are calculated (both diffusion
	//    current and current due to the electric field).
	// 3) When these currents are known, dn/dt and dp/dt can be calculated, and using the
	//    specified timestep 'dt', the densities at each point can be updated.

	bool done = false;

	for (int N = 0 ; !done && N < steps ; N++)
	{
		if (!inverseMatrixSolver)
			potentialFinder();
		else
		{
			if (!m_pPoissonSolver->findPotential(&(m_p[0]), &(m_n[0]), &(m_background[0])))
			{
				setErrorString("Couldn't solve for potential: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			for (int x = 0 ; x < m_width-1 ; x++)
			{
				int idx = x;
				int nextX = x+1;
				int nextIdx = nextX;

				m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth;
			}
		}

		calcNumberCurrents();

		updateDensities(dt);

		if (stopCriterion > 0 && m_updateNAvg < stopCriterion && m_updatePAvg < stopCriterion)
		{
			done = true;
			steps = N+1;
		}
	}

	return true;
}

bool Simulation1DDouble::start(int seconds, double dt, int &steps, double stopCriterion, bool inverseMatrixSolver)
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}
	
	if (!inverseMatrixSolver)
		initializePotentialFinder();
	else
	{
		if (!m_pPoissonSolver)
		{
			m_pPoissonSolver = new InverseMatrixPoissonSolver1D();

			if (!m_pPoissonSolver->init(m_width, (double)(m_width-1)*m_pixelWidth, &(m_potential[0][0])))
			{
				setErrorString("Couldn't initialize inverse matrix potential solver: " + m_pPoissonSolver->getErrorString());
				delete m_pPoissonSolver;
				m_pPoissonSolver = 0;
			}
		}

		if (m_epsChangedForSolver)
		{
			if (!m_pPoissonSolver->setRelativePermittivity(&(m_epsRels[0][0])))
			{
				setErrorString("Couldn't set relative permittivity in solver: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			m_epsChangedForSolver = false;
		}
		if (!m_pPoissonSolver->setPotentialDifference(m_deltaPhi))
		{
			setErrorString("Couldn't set potential difference: " + m_pPoissonSolver->getErrorString());
			return false;
		}
	}

	time_t startTime = time(0);
	time_t endTime = startTime;
	int count = 0;
	int maxSteps = steps;

	endTime += seconds;

	bool done = false;

	do
	{
		if (!inverseMatrixSolver)
			potentialFinder();
		else
		{
			if (!m_pPoissonSolver->findPotential(&(m_p[0]), &(m_n[0]), &(m_background[0])))
			{
				setErrorString("Couldn't solve for potential: " + m_pPoissonSolver->getErrorString());
				return false;
			}
			for (int x = 0 ; x < m_width-1 ; x++)
			{
				int idx = x;
				int nextX = x+1;
				int nextIdx = nextX;

				m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth;
			}
		}

		calcNumberCurrents();

		updateDensities(dt);

		if (stopCriterion > 0 && m_updateNAvg < stopCriterion && m_updatePAvg < stopCriterion)
			done = true;

		count++;

	} while (!done && time(0) <= endTime && count < maxSteps);

	steps = count;

	return true;
}

// NOTE: the functions which together perform the multigrid calculation of the potential
//       are specified at the end of the file. 


// Calculate the number currents of electrons and holes
// Note that (x,y) in the grid is stored at array position x+y*width

void Simulation1DDouble::calcNumberCurrents()
{
	for (int x = 0 ; x < m_width-1 ; x++)
	{
		int idx = x;
		int nextX = x+1;
		int nextIdx = nextX;

		double De = (m_De[idx]+m_De[nextIdx])/2.0;
		double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
		double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
		double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

		if (m_useFieldDependentMobility)
		{
			double gammaE = (m_gammaE[idx]+m_gammaE[nextIdx])/2.0;
			double gammaH = (m_gammaH[idx]+m_gammaH[nextIdx])/2.0;

			eMob *= std::exp(gammaE * std::sqrt(std::abs(m_electicFieldx[idx])));
			hMob *= std::exp(gammaH * std::sqrt(std::abs(m_electicFieldx[idx])));
		}

		double ve = -eMob*(m_electicFieldx[idx]+m_extraElecticFieldNx[idx]);
		double vh = +hMob*(m_electicFieldx[idx]+m_extraElecticFieldPx[idx]);

		CHECK(De, idx);
		CHECK(Dh, idx);
		CHECK(eMob, idx);
		CHECK(hMob, idx);
		CHECK(ve, idx);
		CHECK(vh, idx);

		m_numCurTotEx[idx] = getSGCurrent(ve, De, m_pixelWidth, m_n[idx], m_n[nextIdx]);
		m_numCurTotHx[idx] = getSGCurrent(vh, Dh, m_pixelWidth, m_p[idx], m_p[nextIdx]);

		CHECK(m_numCurTotEx[idx], idx);
		CHECK(m_numCurTotHx[idx], idx);
	}
}


// Based on the currents, update the densities
// NOTE: these are number currents, not charge currents

void Simulation1DDouble::updateDensities(double dt)
{
	// Calculate dn/dt and dp/dt first

	for (int x = 1 ; x < m_width-1 ; x++)
	{
		int prevX = x-1;
		int idx = x;
		int leftIdx = x-1;

		double JexCur = m_numCurTotEx[idx];
		double JexPrev = m_numCurTotEx[leftIdx];

		double JhxCur = m_numCurTotHx[idx];
		double JhxPrev = m_numCurTotHx[leftIdx];

		// Calculate the gradients

		double Jexx = (JexCur - JexPrev)/m_pixelWidth;
		double Jhxx = (JhxCur - JhxPrev)/m_pixelWidth;
			
		// Calculate the rates according to the continuity equation

		double ni = m_ni[idx];
		double recombination = m_recombinationFactor[idx]*(m_p[idx]*m_n[idx]-ni*ni);

		double netGeneration = 0;

		if (m_factorPinRG)
		{
			double P = calculatePFactor(idx);

			m_P[idx] = P;
			netGeneration = P*m_generationRate[idx]-(1.0-P)*recombination;
		}
		else
			netGeneration = m_generationRate[idx]-recombination;

		m_dndt[idx] = netGeneration-Jexx;
		m_dpdt[idx] = netGeneration-Jhxx;
	}

	// Then, perform the actual update of the densities

	//std::cerr << "maxDt = " << maxDt << std::endl;

	//dt = maxDt;

	double updateNAvg = 0;
	double updatePAvg = 0;
	double nAvg = 0;
	double pAvg = 0;

	for (int x = 1 ; x < m_width-1 ; x++)
	{
		int idx = x;

		m_n[idx] += m_dndt[idx]*dt;
		m_p[idx] += m_dpdt[idx]*dt;

		nAvg += std::abs(m_n[idx]);
		pAvg += std::abs(m_p[idx]);

		updateNAvg += std::abs(m_dndt[idx]*dt);
		updatePAvg += std::abs(m_dpdt[idx]*dt);
	}

	m_updatePAvg = updateNAvg/nAvg;
	m_updateNAvg = updatePAvg/pAvg;

	static int count = 0;

	if (m_S > 0)
	{
		m_n[0] = m_n00 - m_numCurTotEx[0]/m_S;
		m_p[0] = m_p00 - m_numCurTotHx[0]/m_S;
		m_n[m_totalPixels-1] = m_n0d + m_numCurTotEx[m_totalPixels-2]/m_S;
		m_p[m_totalPixels-1] = m_p0d + m_numCurTotHx[m_totalPixels-2]/m_S;

		if (count++ % 10000 == 0)
		{
			std::cout << "n0: " << (m_n[0]/m_n00-1.0) << std::endl;
			std::cout << "p0: " << (m_p[0]/m_p00-1.0) << std::endl;
			std::cout << "nd: " << (m_n[m_totalPixels-1]/m_n0d-1.0) << std::endl;
			std::cout << "pd: " << (m_p[m_totalPixels-1]/m_p0d-1.0) << std::endl;
			std::cout << std::endl;
		}
	}
}

// Helper function to calculate the current in the X-direction at the left side, the right side
// and averaged over the entire grid

void Simulation1DDouble::calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	for (int x = 0 ; x < m_width-1 ; x++)
	{
		int idx = x;

		double JexCur = m_numCurTotEx[idx];
		double JhxCur = m_numCurTotHx[idx];
		
		sum += -JexCur+JhxCur;
	}

	int idx = 0;

	double JexCur = m_numCurTotEx[idx];
	double JhxCur = m_numCurTotHx[idx];

	sum2 += -JexCur + JhxCur;

	idx = m_width-2;

	JexCur = m_numCurTotEx[idx];
	JhxCur = m_numCurTotHx[idx];

	sum3 += -JexCur + JhxCur;

	idx = m_width/2;

	JexCur = m_numCurTotEx[idx];
	JhxCur = m_numCurTotHx[idx];

	sum4 += -JexCur + JhxCur;

	sum /= (m_width-1);

	leftAvg = sum2;
	rightAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

double Simulation1DDouble::calculatePavg() const
{
	double Psum = 0;

	for (int x = 1 ; x < m_width-1 ; x++)	
		Psum += m_P[x];

	Psum /= (double)(m_width-2);

	return Psum;
}

// 
// The rest of the file contains functions for calculating the potential. This is
// probably the most complex part of the calculation, mostly because a multigrid
// method is used.
//
// To calculate the potential, the following inputs are used:
//  - The relative permittivity at each grid point
//  - The electron density at each grid point
//  - The hole density at each grid point
//  - A fixed background charge density
//  - Potential at the top contact (bottom contact is assumed to be at 0 V

// This function initializes some parts of the routine which determines the potential
// (if necessary).

void Simulation1DDouble::initializePotentialFinder()
{
	// Initialize the potential at the coarsest scale

	if (m_initPotential)
	{
		m_initPotential = false;

		int multiplier = 1 << (m_multiScaleSteps-1);
		int lowWidth = m_width/multiplier;

		for (int x = 0 ; x < lowWidth ; x++)
		{
			double V = ((double)x/(double)(lowWidth-1))*m_deltaPhi;
			int idx = x;

			m_potential[m_multiScaleSteps-1][idx] = V;
		}
		
		m_phiBackupBetterCounter = 0;
	}
	
	// Build scaled arrays of permittivity at each scale

	if (m_epsChanged)
	{
		m_epsChanged = false;

		int w = m_width;

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				double epsSum = m_epsRels[i][x] + m_epsRels[i][x+1];

				m_epsRels[i+1][x/2] = epsSum/2.0; // average it
			}

			w /= 2;
		}


		w = m_width;

		for (int i = 0 ; i < m_multiScaleSteps ; i++)
		{
			for (int x = 0 ; x < w ; x++)
			{
				int xNext = (x+1);
				int xPrev = (x-1);

				if (xPrev < 0)
					xPrev = 0;
				if (xNext >= w)
					xNext = w-1;

				int index = x;
				int leftIndex = xPrev; 
				int rightIndex = xNext;

				m_a1s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][rightIndex]);
				m_a2s[i][index] = 0.5*(m_epsRels[i][index]+m_epsRels[i][leftIndex]);
				m_a0s[i][index] = m_a1s[i][index] + m_a2s[i][index];
			}

			w /= 2;
		}
	
		m_phiBackupBetterCounter = 0;
	}

	if (m_extraPotentialChanged)
	{
		m_extraPotentialChanged = false;

		for (int x = 0 ; x < m_width-1 ; x++)
		{	
			int idx = x;
			int nextIdx = x+1;

			m_extraElecticFieldNx[idx] = -(m_extraPotentialN[nextIdx]-m_extraPotentialN[idx])/m_pixelWidth;
			m_extraElecticFieldPx[idx] = -(m_extraPotentialP[nextIdx]-m_extraPotentialP[idx])/m_pixelWidth;
		}

		// NOTE: the last row is not needed (and cannot be calculated by the way)
	}
}

void Simulation1DDouble::potentialFinder()
{
	if (m_phiBackupBetterCounter < m_phiMethodThreshold || m_phiBackupBetterCounter%m_phiMethodAdditionalCheckCount == 0)
	{
		// First create scaled grids of the charge

		//std::cerr << m_phiBackupBetterCounter << " Trying both " << std::endl;

		int w = m_width;

		double chargeMultiplier = CHARGE_ELECTRON*m_pixelWidth*m_pixelWidth/CONST_EPSILON0;

		for (int x = 0 ; x < w ; x++)
		{
			int idx = x;

			m_chargeSums[0][idx] = (m_p[idx]-m_n[idx]+m_background[idx])*chargeMultiplier;
		}

		for (int i = 0 ; i < m_multiScaleSteps-1 ; i++)
		{
			int w2 = w/2;

			for (int x = 0 ; x < w ; x += 2)
			{
				double chargeSum = m_chargeSums[i][x] + m_chargeSums[i][x+1];

				m_chargeSums[i+1][x/2] = chargeSum; // sum it
			}

			w /= 2;
		}

		// Set boundary conditions again on the coarsest scale (just to be safe)
		{
			m_potential[m_multiScaleSteps-1][0] = 0;
			m_potential[m_multiScaleSteps-1][w-1] = m_deltaPhi;
		}

		// Width and height should now contain the dimensions of the coarsest grid
		
		// Calculate the potential for the lowest resolution grid
		double error = blackRed(&(m_potential[m_multiScaleSteps-1][0]), m_multiScaleSteps-1, w, 100, 1.8);

		for (int i = m_multiScaleSteps-2 ; i >= 0 ; i--)
		{
			int w2 = w;

			w *= 2;

			// Interpolate the result for a first estimate of the higher resolution result

			for (int x = 0 ; x < w ; x++)
			{
				double xFrac = (double)x/(double)(w-1);
				double x1d = xFrac*(w2-1);
				int x1 = (int)x1d;
				int x2 = x1+1;

				if (x2 == w2)
					x2 = w2-1;

				double s = x1d-(double)x1;

				double v1 = m_potential[i+1][x1];
				double v2 = m_potential[i+1][x2];

				// linear interpolation
				m_potential[i][x] = v1*(1.0-s) + v2*s;
			}

			// Set boundary conditions again

			m_potential[i][0] = 0;
			m_potential[i][w-1] = m_deltaPhi;

			// Optimize at new scale

			error = blackRed(&(m_potential[i][0]), i, w, 10, 1.8);
		}

		// We shall also continuously update another potential field using another
		// method

		// Set boundary conditions again on backup potential
		{
			m_backupPotential[0] = 0;
			m_backupPotential[m_width-1] = m_deltaPhi;
		}


		double error2 = blackRed(&(m_backupPotential[0]), 0, m_width, 4, 1.0);

		//std::cerr << "error = " << error << " error2 = " << error2 << std::endl;

		// We'll use the best result of the two approaches
		if (error2 >= error)
		{
			m_phiBackupBetterCounter = 0;
			memcpy(&(m_backupPotential[0]),&(m_potential[0][0]), sizeof(double)*m_totalPixels);
		}
		else
		{
			m_phiBackupBetterCounter++;
			memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
		}
	}
	else
	{
		// First grid of total charge

		int w = m_width;

		double chargeMultiplier = CHARGE_ELECTRON*m_pixelWidth*m_pixelWidth/CONST_EPSILON0;

		for (int x = 0 ; x < w ; x++)
		{
			int idx = x;

			m_chargeSums[0][idx] = (m_p[idx]-m_n[idx]+m_background[idx])*chargeMultiplier;
		}

		// Set boundary conditions again on backup potential
		{
			m_backupPotential[0] = 0;
			m_backupPotential[m_width-1] = m_deltaPhi;
		}

		blackRed(&(m_backupPotential[0]), 0, m_width, 4, 1.0);
		m_phiBackupBetterCounter++;
		memcpy(&(m_potential[0][0]), &(m_backupPotential[0]), sizeof(double)*m_totalPixels);
	}
	
	// Determine electric field from potential field

	for (int x = 0 ; x < m_width-1 ; x++)
	{
		int idx = x;
		int nextX = x+1;
		int nextIdx = nextX;

		m_electicFieldx[idx] = -(m_potential[0][nextIdx] - m_potential[0][idx])/m_pixelWidth;
	}

	// NOTE: the last position is not needed (and cannot be calculated by the way)
}

double Simulation1DDouble::blackRed(double *pSrc, int aIndex, int width, int steps, double w)
{
	double error = -1;

	for (int i = 0 ; i < steps ; i++)
	{
		error = 0;

		// Alternating updates in the same array seems to be much more stable!

		for (int loop = 0 ; loop < 2 ; loop++)
		{
			int xOff = loop+1;

			for (int x = xOff ; x < width-1 ; x += 2)
			{
				int xNext = x+1;
				int xPrev = x-1;

				int index = x;
				int leftIndex = xPrev;
				int rightIndex = xNext;

				double prediction = (m_a1s[aIndex][index]*pSrc[rightIndex] + m_a2s[aIndex][index]*pSrc[leftIndex]
						  +  m_chargeSums[aIndex][index])/m_a0s[aIndex][index];

				double curValue = pSrc[index];

				double diff = prediction-curValue;

				pSrc[index] += w*diff;

				// update error

				error += diff*diff;
			}
		}
	}

	return std::sqrt(error/(double)(width));
}

void Simulation1DDouble::resizeArrays()
{
	m_n.resize(m_totalPixels);
	m_ni.resize(m_totalPixels);
	m_p.resize(m_totalPixels);
	m_background.resize(m_totalPixels);
	m_generationRate.resize(m_totalPixels);
	m_recombinationFactor.resize(m_totalPixels);
	m_De.resize(m_totalPixels);
	m_Dh.resize(m_totalPixels);
	m_eMob.resize(m_totalPixels);
	m_hMob.resize(m_totalPixels);
	m_gammaE.resize(m_totalPixels);
	m_gammaH.resize(m_totalPixels);

	int tmpSize = m_totalPixels;

	// More arrays need to be allocated for the multigrid method

	m_epsRels.resize(m_multiScaleSteps);
	m_chargeSums.resize(m_multiScaleSteps);
	m_potential.resize(m_multiScaleSteps);
	m_a0s.resize(m_multiScaleSteps);
	m_a1s.resize(m_multiScaleSteps);
	m_a2s.resize(m_multiScaleSteps);

	m_backupPotential.resize(m_totalPixels);

	for (int i = 0 ; i < m_multiScaleSteps ; i++)
	{
		m_epsRels[i].resize(tmpSize);
		m_chargeSums[i].resize(tmpSize);
		m_potential[i].resize(tmpSize);
		m_a0s[i].resize(tmpSize);
		m_a1s[i].resize(tmpSize);
		m_a2s[i].resize(tmpSize);
		tmpSize /= 2;
	}

	m_electicFieldx.resize(m_totalPixels);

	// Initialize the values of some arrays

	setValues(m_n, 0);
	setValues(m_ni, 0);
	setValues(m_p, 0);
	setValues(m_background, 0);
	setValues(m_generationRate, 0);
	setValues(m_recombinationFactor, 0);
	setValues(m_De, 0);
	setValues(m_Dh, 0);
	setValues(m_eMob, 0);
	setValues(m_hMob, 0);
	setValues(m_gammaE, 0);
	setValues(m_gammaH, 0);
	setValues(m_epsRels[0], 1);
	setValues(m_background, 0);

	// Allocate memory for the arrays which will contain the number currents
	// due to diffusion and electric field respectively

	m_numCurTotEx.resize(m_totalPixels);
	m_numCurTotHx.resize(m_totalPixels);

	m_dndt.resize(m_totalPixels);
	m_dpdt.resize(m_totalPixels);

	m_extraPotentialN.resize(m_totalPixels);
	m_extraPotentialP.resize(m_totalPixels);
	m_extraElecticFieldNx.resize(m_totalPixels);
	m_extraElecticFieldPx.resize(m_totalPixels);

	setValues(m_extraPotentialN, 0);
	setValues(m_extraPotentialP, 0);

	m_P.resize(m_totalPixels);
	setValues(m_P, 0);
}

void Simulation1DDouble::writePlotData(FILE *pFile)
{
	for (int x = 0 ; x < m_width ; x++)
	{
		int idx = x;
		double j1p = 0;
		double j1n = 0;

		if (x == m_width-1)
		{
			j1p = m_numCurTotHx[idx-1]*CHARGE_ELECTRON;
			j1n = -m_numCurTotEx[idx-1]*CHARGE_ELECTRON;
		}
		else
		{
			j1p = m_numCurTotHx[idx]*CHARGE_ELECTRON;
			j1n = -m_numCurTotEx[idx]*CHARGE_ELECTRON;
		}
		double j1 = j1p+j1n;

		double U = 0;

		// TODO
		U = 

		fprintf(pFile, "%d %g %g %g %g %g %g %g %g %g %g %g %g %g %g\n", x,  
					 (double)m_n[idx], (double)m_p[idx], (double)m_potential[0][idx], 
					 (double)(m_recombinationFactor[idx]*m_n[idx]*m_p[idx]),
					 (double)j1n, (double)j1p, (double)j1, (double)U,
					 (double)m_generationRate[idx], (double)m_recombinationFactor[idx],
					 (double)m_De[idx], (double)m_Dh[idx], (double)m_eMob[idx], (double)m_hMob[idx]);
	}
	fprintf(pFile, "\n\n");
}

void Simulation1DDouble::setExtendedRecombinationModel(double a, double kf)
{ 
	m_pairDist = a; 
	m_kf = kf; 
	m_factorPinRG = true; 
}

double Simulation1DDouble::calculatePFactor(int idx)
{
	/*
	double F = 0.5*(m_electicFieldx[idx]+m_electicFieldx[idx-1]);

	F = std::abs(F);

	double kT_ev = m_kT/CHARGE_ELECTRON;
	double b = CHARGE_ELECTRON*F/(8.0*CONST_PI*CONST_EPSILON0*m_epsRels[0][idx]*kT_ev*kT_ev);
	double b2 = b*b;
	double b3 = b2*b;
	double b4 = b2*b2;
	double factor = 1.0+b+b2/3.0+b3/18.0+b4/180.0;

	double kDiss = m_recombinationFactor[idx]*m_expFactor*factor;
	double P = kDiss/(kDiss+m_kf);

//	std::cerr << "b[" << idx << "] = " << b << std::endl;
//	std::cerr << "P = " << P << std::endl;
*/

	double F = 0.5*(m_electicFieldx[idx]+m_electicFieldx[idx-1]);

	F = std::abs(F);

	double kT_ev = m_T*CONST_K/CHARGE_ELECTRON;
	double b = CHARGE_ELECTRON*F/(8.0*CONST_PI*CONST_EPSILON0*m_epsRels[0][idx]*kT_ev*kT_ev);
	
	if (m_bMax < b)
		precalcBValues(b); // TODO: for now we ignore 'idx'

	double bStep = m_bMax/(double)(m_precalcBValues.size()-1);

	int i0 = (int)(b/bStep);
	int i1 = i0+1;

	double frac = ((double)i0)*bStep-b;

	double P = (1.0-frac)*m_precalcBValues[i0]+frac*m_precalcBValues[i1];

	return P;

}

void Simulation1DDouble::precalcBValues(double curB)
{
	m_bMax = curB*2.0;
	int num = 1000;

	m_precalcBValues.resize(num);

	for (int i = 0 ; i < num ; i++)
	{
		double b = (m_bMax/(double)(num-1))*(double)i;

		m_precalcBValues[i] = calcBValue(b);

		std::cout << b << " " << m_precalcBValues[i] << std::endl;
	}

	std::cout << "Precalculating" << std::endl;
}

double Simulation1DDouble::calcBValue(double b)
{
	int idx = 0; // TODO!

	double b2 = b*b;
	double b3 = b2*b;
	double b4 = b2*b2;
	//double factor = 1.0+b+b2/3.0;
	//double factor = 1.0+b+b2/3.0+b3/18.0;
	double factor = 1.0+b+b2/3.0+b3/18.0+b4/180.0;
	//double factor = gsl_sf_bessel_I1(std::sqrt(8.0*b))/std::sqrt(2.0*b);

	double stop = m_pairDist*4.0;
	int num = 200;
	double binWidth = stop/(double)num;
	double x = binWidth/2.0;
	double sum = 0;
	double kT = m_T*CONST_K;

	for (int i = 0 ; i < num ; i++, x += binWidth)
	{
		double Eb = (CHARGE_ELECTRON*CHARGE_ELECTRON/(4.0*CONST_PI*CONST_EPSILON0*m_epsRels[0][idx]*x));
		//std::cout << Eb << " " << m_kT << " " << Eb/CHARGE_ELECTRON << std::endl;
		double kDiss = (3.0*m_recombinationFactor[idx]/(4.0*CONST_PI*x*x*x))*std::exp(-Eb/kT)*factor;

		double xOverA = x/m_pairDist;

		sum += (1.0/(1.0+m_kf/kDiss))*x*x*std::exp(-xOverA*xOverA);
	}

	sum *= 4.0/(CONST_SQRT_PI*m_pairDist*m_pairDist*m_pairDist)*binWidth;

	return sum;
}

bool Simulation1DDouble::getExtendedRecombinationParameters(double &pairDistance, double &kf) const
{
	if (!m_factorPinRG)
	{
		setErrorString("Not using extended recombination model");
		return false;
	}

	pairDistance = m_pairDist;
	kf = m_kf;
	return true;
}

bool Simulation1DDouble::setState(const SimulationState &state, std::vector<std::string> &warnings)
{
	if (!m_init)
	{
		setErrorString("Simulation not initialized");
		return false;
	}
	if (!state.isInitialized())
	{
		setErrorString("State to be used is not initialized");
		return false;
	}
	if (state.getDimensions() != 1)
	{
		setErrorString("State is not one-dimensional");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels())
	{
		setErrorString("State does not have same number of pixels as simulation");
		return false;
	}

	std::vector<bool> m_input(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_output(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_optional(SIMSTATE_GRIDPROP_MAX);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		m_input[i] = false;
		m_output[i] = false;
		m_optional[i] = false;
	}

	m_input[SIMSTATE_GRIDPROP_N] = true;
	m_input[SIMSTATE_GRIDPROP_P] = true;
	m_input[SIMSTATE_GRIDPROP_DN] = true;
	m_input[SIMSTATE_GRIDPROP_DP] = true;
	m_input[SIMSTATE_GRIDPROP_NMOB] = true;
	m_input[SIMSTATE_GRIDPROP_PMOB] = true;
	m_input[SIMSTATE_GRIDPROP_EPSREL] = true;

	m_output[SIMSTATE_GRIDPROP_N] = true;
	m_output[SIMSTATE_GRIDPROP_P] = true;
	m_output[SIMSTATE_GRIDPROP_V] = true;
	m_output[SIMSTATE_GRIDPROP_JNX] = true;
	m_output[SIMSTATE_GRIDPROP_JPX] = true;
	m_output[SIMSTATE_GRIDPROP_R] = true;
	m_output[SIMSTATE_GRIDPROP_DISSPROB] = true;

	m_optional[SIMSTATE_GRIDPROP_BG] = true;
	m_optional[SIMSTATE_GRIDPROP_NI] = true;
	m_optional[SIMSTATE_GRIDPROP_G] = true;
	m_optional[SIMSTATE_GRIDPROP_RF] = true;
	m_optional[SIMSTATE_GRIDPROP_NMOB_GAMMA] = true;
	m_optional[SIMSTATE_GRIDPROP_PMOB_GAMMA] = true;
	m_optional[SIMSTATE_GRIDPROP_VNEXTRA] = true;
	m_optional[SIMSTATE_GRIDPROP_VPEXTRA] = true;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (m_input[i])
		{
			if (!state.isGridPropertySet(i))
			{
				setErrorString("Grid property '" + state.getGridPropertyName(i) + "' is required, but has not been set");
				return false;
			}
		}
		else
		{
			if (state.isGridPropertySet(i))
			{
				if (!m_output[i] && !m_optional[i])
					warnings.push_back("Property '" + state.getGridPropertyName(i) + "' is set but not used");
			}
		}
	}

	int numX = getNumXPixels();

	if (state.isGridPropertySet(SIMSTATE_GRIDPROP_NMOB_GAMMA) || state.isGridPropertySet(SIMSTATE_GRIDPROP_PMOB_GAMMA))
		m_useFieldDependentMobility = true;
	else
		m_useFieldDependentMobility = false;

	for (int x = 0 ; x < numX ; x++)
	{
		setElectronNumberDensity(x, state.getGridProperty(SIMSTATE_GRIDPROP_N, x));
		setHoleNumberDensity(x, state.getGridProperty(SIMSTATE_GRIDPROP_P, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_NI))
			setIntrinsicElectronNumberDensity(x, state.getGridProperty(SIMSTATE_GRIDPROP_NI, x));


		setHoleMobility(x, state.getGridProperty(SIMSTATE_GRIDPROP_P, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_BG))
			setBackgroundNumberDensity(x, state.getGridProperty(SIMSTATE_GRIDPROP_BG, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_G))
			setGenerationRate(x, state.getGridProperty(SIMSTATE_GRIDPROP_G, x));
		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_RF))
			setRecombinationFactor(x, state.getGridProperty(SIMSTATE_GRIDPROP_RF, x));

		setElectronDiffusionConstant(x, state.getGridProperty(SIMSTATE_GRIDPROP_DN, x));
		setHoleDiffusionConstant(x, state.getGridProperty(SIMSTATE_GRIDPROP_DP, x));
		setElectronMobility(x, state.getGridProperty(SIMSTATE_GRIDPROP_NMOB, x));
		setHoleMobility(x, state.getGridProperty(SIMSTATE_GRIDPROP_PMOB, x));
		setRelativePermittivity(x, state.getGridProperty(SIMSTATE_GRIDPROP_EPSREL, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_V))
			setPotential(x, state.getGridProperty(SIMSTATE_GRIDPROP_V, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_NMOB_GAMMA))
			m_gammaE[x] = state.getGridProperty(SIMSTATE_GRIDPROP_NMOB_GAMMA, x);
		else
			m_gammaE[x] = 0;

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_PMOB_GAMMA))
			m_gammaH[x] = state.getGridProperty(SIMSTATE_GRIDPROP_PMOB_GAMMA, x);
		else
			m_gammaH[x] = 0;

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VNEXTRA))
			setExtraElectronPotential(x, state.getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, x));

		if (state.isGridPropertySet(SIMSTATE_GRIDPROP_VPEXTRA))
			setExtraHolePotential(x, state.getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, x));

	}

	double vDiff;

	if (!state.getDoubleProperty(SIMSTATE_PROP_VDIFF, vDiff))
	{
		setErrorString("Potential difference has not been set");
		return false;
	}

	setPotentialDifference(vDiff);

	if (state.getRecombinationModel() == SimulationState::Braun)
	{
		double a, kf;

		if (!state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a) ||
		    !state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
		{
			setErrorString("Braun recombination model selected, but necessary parameters are not set");
			return false;
		}

		setExtendedRecombinationModel(a, kf);	
	}
	else
	{
		double a, kf;

		if (state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
			warnings.push_back("'Pair distance' parameter is set, but is only used in the braun model");
		if (state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
			warnings.push_back("'Dissociation rate' parameter 'kf' is set, but is only used in the braun model");

		setNormalRecombinationModel();
	}

	//FILE *f = fopen("log.txt","wt");
	//writePlotData(f);
	//fclose(f);

	m_epsChanged = true;
	m_epsChangedForSolver = true;

	return true;
}

bool Simulation1DDouble::storeState(SimulationState &state) const
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	if (state.getNumberOfXPixels() != getNumXPixels())
	{
		setErrorString("Number of pixels is not same as in simulation state");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		state.clearDoubleProperty(i);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		state.clearGridProperty(i);

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_N, m_n) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_P, m_p) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_V, m_potential[0]))
	{
		setErrorString("Unable to store n,p or V");
		return false;
	}

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_BG, m_background) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NI, m_ni) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DN, m_De) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels[0]) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, m_extraPotentialN) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, m_extraPotentialP) )
	{
		setErrorString("Unable to store fixed simulation items");
		return false;
	}

	if (m_useFieldDependentMobility)
	{
		if (!state.setGridProperty(SIMSTATE_GRIDPROP_NMOB_GAMMA, m_gammaE) ||
		    !state.setGridProperty(SIMSTATE_GRIDPROP_PMOB_GAMMA, m_gammaH))
		{
			setErrorString("Unable to store the field dependent mobility parameter gamma");
			return false;
		}
	}

	for (int y = 0 ; y < m_width ; y++)
	{
		double jn = m_numCurTotEx[y];
		double jp = m_numCurTotHx[y];

		if (!state.setGridProperty(SIMSTATE_GRIDPROP_JNX, y, jn) ||
		    !state.setGridProperty(SIMSTATE_GRIDPROP_JPX, y, jp))
		{
			setErrorString("Unable to store the number current");
			return false;
		}
	}

	if (!m_factorPinRG)
	{
		for (int idx = 0 ; idx < m_width ; idx++)
		{
			double ni = m_ni[idx];
			double R = m_recombinationFactor[idx] * (m_n[idx] * m_p[idx] - ni*ni);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, idx, R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
		}

		state.setRecombinationModel(SimulationState::Simple);	
	}
	else
	{
		for (int y = 0 ; y < m_width ; y++)
		{
			int idx = y;

			if (idx == 0)
				idx = 1;
			else if (idx == m_width-1)
				idx = m_width-2;

			double ni = m_ni[idx];
			double dissProb = m_P[idx];
			double R = (1.0-dissProb) * m_recombinationFactor[idx] * (m_n[idx] * m_p[idx] - ni*ni);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, y, R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_DISSPROB, y, dissProb))
			{
				setErrorString("Unable to store the dissociation rate");
				return false;
			}
		}

		state.setRecombinationModel(SimulationState::Braun);

		if (!state.setDoubleProperty(SIMSTATE_PROP_KF, m_kf) ||
		    !state.setDoubleProperty(SIMSTATE_PROP_PAIRDIST, m_pairDist))
		{
			setErrorString("Unable to store recombination model properties");
			return false;
		}
	}

	if (!state.setDoubleProperty(SIMSTATE_PROP_VDIFF, m_deltaPhi))
	{
		setErrorString("Couldn't store potential difference");
		return false;
	}

	if (!state.setDoubleProperty(SIMSTATE_PROP_TEMP, m_T))
	{
		setErrorString("Couldn't store temperature");
		return false;
	}

	return true;
}


