#ifndef SIMULATION1DAVE_H

#define SIMULATION1DAVE_H

#include "simulation1d.h"

#ifdef SIMICONDUCTOR_CONFIG_AVE

class DissociationProbability;
class SimulationState;

class Simulation1DAVE : public Simulation1D
{
public:
	Simulation1DAVE();
	~Simulation1DAVE();

	// Set grid dimensions
	bool init(int width, double realWidth, double T, double diffScale, double densScale, bool useLog);
	bool isInitialized() const										{ return m_init; }

	int getNumXPixels() const										{ return m_height; }
	double getPixelWidth() const										{ return m_pixelHeight2; }

	double getTemperature() const										{ return m_T; }
	double getDensScale() const										{ return m_densScale; }
	double getDiffusionScale() const									{ return m_diffScale; }
	bool isUsingLog() const											{ return m_useLog; }

	bool setState(const SimulationState &state, std::vector<std::string> &warnings);
	bool storeState(SimulationState &state) const;

	// After 'init' has been called, these functions can be used to specify a specific
	// starting situation
	void setElectronNumberDensity(int x, double dens)							{ setValue(m_n, x, dens); }
	void setElectronNumberDensity(int x0, int x1, double dens)						{ setValues(m_n, x0, x1, dens); }
	void setElectronNumberDensity(double dens)								{ setValues(m_n, dens); }

	void setIntrinsicElectronNumberDensity(int x,double dens)						{ setValue(m_ni, x, dens); }
	void setIntrinsicElectronNumberDensity(int x0, int x1, double dens)					{ setValues(m_ni, x0, x1, dens); }
	void setIntrinsicElectronNumberDensity(double dens)							{ setValues(m_ni, dens); }

	void setHoleNumberDensity(int x, double dens)								{ setValue(m_p, x, dens); }
	void setHoleNumberDensity(int x0, int x1, double dens)							{ setValues(m_p, x0, x1, dens); }
	void setHoleNumberDensity(double dens)									{ setValues(m_p, dens); }

	void setBackgroundNumberDensity(int x, double dens)							{ setValue(m_background, x, dens); }
	void setBackgroundNumberDensity(int x0, int x1, double dens)						{ setValues(m_background, x0, x1, dens); }
	void setBackgroundNumberDensity(double dens)								{ setValues(m_background, dens); }

	void setGenerationRate(int x, double rate)								{ setValue(m_generationRate, x, rate); }
	void setGenerationRate(int x0, int x1, double rate)							{ setValues(m_generationRate, x0, x1, rate); }
	void setGenerationRate(double rate)									{ setValues(m_generationRate, rate); }

	// r = factor * n * p;
	void setRecombinationFactor(int x, double factor)							{ setValue(m_recombinationFactor, x, factor); }
	void setRecombinationFactor(int x0, int x1, double factor)						{ setValues(m_recombinationFactor, x0, x1, factor); }
	void setRecombinationFactor(double factor)								{ setValues(m_recombinationFactor, factor); }

	// bottom = 0, top = V
	void setPotentialDifference(double V)									{ m_deltaPhi = V; }

	void setElectronDiffusionConstant(int x, double D)							{ setValue(m_De, x, D); }
	void setElectronDiffusionConstant(int x0, int x1, double D)						{ setValues(m_De, x0, x1, D); }
	void setElectronDiffusionConstant(double D)								{ setValues(m_De, D); }

	void setHoleDiffusionConstant(int x, double D)								{ setValue(m_Dh, x, D); }
	void setHoleDiffusionConstant(int x0, int x1, double D)							{ setValues(m_Dh, x0, x1, D); }
	void setHoleDiffusionConstant(double D)									{ setValues(m_Dh, D); }

	void setElectronMobility(int x, double mu)								{ setValue(m_eMob, x, mu); }
	void setElectronMobility(int x0, int x1, double mu)							{ setValues(m_eMob, x0, x1, mu); }
	void setElectronMobility(double mu)									{ setValues(m_eMob, mu); }

	void setHoleMobility(int x, double mu)									{ setValue(m_hMob, x, mu); }
	void setHoleMobility(int x0, int x1, double mu)								{ setValues(m_hMob, x0, x1, mu); }
	void setHoleMobility(double mu)										{ setValues(m_hMob, mu); }

	void setRelativePermittivity(int x, double epsRel)							{ setValue(m_epsRels, x, epsRel); }
	void setRelativePermittivity(int x0, int x1, double epsRel)						{ setValues(m_epsRels, x0, x1, epsRel); }
	void setRelativePermittivity(double epsRel)								{ setValues(m_epsRels, epsRel); }

	void setPotential(int x, double v)									{ setValue(m_potential, x, v); }
	void setPotential(int x0, int x1, double v)								{ setValues(m_potential, x0, x1, v); }
	void setPotential(double v)										{ setValues(m_potential, v); }

	bool start(double &sigma);

	void initPotential();
	
	void calculateXCurrent(double &leftAvg, double &rightAvg, double &overallAvg, double &center) const;

	void getElectronNumberDensity(std::vector<double> &dst) const						{ getProperty(m_n, dst); }
	void getIntrinsicElectronNumberDensity(std::vector<double> &dst) const					{ getProperty(m_ni, dst); }
	void getHoleNumberDensity(std::vector<double> &dst) const						{ getProperty(m_p, dst); }
	void getBackgroundNumberDensity(std::vector<double> &dst) const						{ getProperty(m_background, dst); }
	void getGenerationRate(std::vector<double> &dst) const							{ getProperty(m_generationRate, dst); }
	void getRecombinationFactor(std::vector<double> &dst) const						{ getProperty(m_recombinationFactor, dst); }
	void getPotential(std::vector<double> &dst) const							{ getProperty(m_potential, dst); }
	void getElectronDiffusionConstant(std::vector<double> &dst) const					{ getProperty(m_De, dst); }
	void getHoleDiffusionConstant(std::vector<double> &dst) const						{ getProperty(m_Dh, dst); }
	void getElectronMobility(std::vector<double> &dst) const						{ getProperty(m_eMob, dst); }
	void getHoleMobility(std::vector<double> &dst) const							{ getProperty(m_hMob, dst); }
	void getRelativePermittivity(std::vector<double> &dst) const						{ getProperty(m_epsRels, dst); }
	void getTotalElectronXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotEy, dst, 1.0/m_pixelHeight2); }
	void getTotalHoleXCurrent(std::vector<double> &dst) const						{ getProperty(m_numCurTotHy, dst, 1.0/m_pixelHeight2); }
	void getDissociationProbability(std::vector<double> &dst) const						{ getProperty(m_P, dst); }
	
	double getPotentialDifference() const									{ return m_deltaPhi; }

	void writePlotData(FILE *pFile);

	void setExtendedRecombinationModel(double pairDistance, double kf);
	void setNormalRecombinationModel()									{ m_simpleRecombination = true; }

	bool getExtendedRecombinationParameters(double &pairDistance, double &kf) const;
private:
	void setValue(std::vector<double> &grid, int x, double v)						{ grid[x] = v; }
	void setValues(std::vector<double> &grid, int x0, int x1, double v);
	void setValues(std::vector<double> &grid, double v)							{ setValues(grid, 0, m_height, v); }
	void getProperty(const std::vector<double> &grid, std::vector<double> &dst) const			{ dst = grid; }
	void getProperty(const std::vector<long double> &grid, std::vector<double> &dst, long double factor) const { dst.resize(grid.size()); for (int i = 0 ; i < grid.size() ; i++) dst[i] = grid[i]*factor;  }
	void resizeArrays();

	long double getFValue(int Ftype, int y);
	long double getFV(int y);
	long double getFN(int y);
	long double getFP(int y);
	long double getFJacValue(int Ftype, int y, int varType, int v);
	long double getFJacV(int t, int y, int v);
	long double getFJacN(int t, int y, int v);
	long double getFJacP(int t, int y, int v);
	long double getFJacVV(int y, int v);
	long double getFJacVN(int y, int v);
	long double getFJacVP(int y, int v);
	long double getFJacNV(int y, int v);
	long double getFJacNN(int y, int v);
	long double getFJacNP(int y, int v);
	long double getFJacPV(int y, int v);
	long double getFJacPN(int y, int v);
	long double getFJacPP(int y, int v);
	long double getNetGen(int y);
	long double getNetGenDerivV(int y, int v);
	long double getNetGenDerivN(int y, int v);
	long double getNetGenDerivP(int y, int v);
	void calcFactorsAndCurrents();
	static long double calcW(long double x);
	static long double calcDerivW(long double x);
	static long double calcJ(long double x, long double d1, long double d2);
	static long double calcDerivJx(long double x, long double d1, long double d2);
	static long double calcDerivJd1(long double x, long double d1, long double d2);
	static long double calcDerivJd2(long double x, long double d1, long double d2);
	void calcP(int y);
	long double calcDerivP(int y, int v);

	void calcCurrentVertical(int y);

	bool m_init;
	int m_height; // ok
	int m_totalPixels;
	
	double m_pixelHeight2;

	std::vector<double> m_n, m_p, m_background, m_ni;
	std::vector<double> m_generationRate, m_recombinationFactor; // ok
	std::vector<double> m_De, m_Dh, m_eMob, m_hMob; // ok
	std::vector<double> m_epsRels, m_potential;
	std::vector<long double> m_numCurTotEy, m_numCurTotHy; // ok
	std::vector<double> m_P;

	double m_DL2, m_DLy2;
	int m_numVariablePixels;

	double m_deltaPhi; // ok
	long double m_diffScale, m_kTev, m_densScale;
	double m_T, m_a, m_kf;

	bool m_useLog;

	class AVESearch;

	AVESearch *m_pAVESearch;
	friend class Search;

	bool m_simpleRecombination;
	DissociationProbability *m_pDissProb;
};

inline void Simulation1DAVE::setValues(std::vector<double> &grid, int x0, int x1, double v)
{
	if (x1 < x0)
		return;

	for (int x = x0 ; x != x1 ; x++)
		setValue(grid, x, v);
}

#endif // SIMICONDUCTOR_CONFIG_AVE

#endif // SIMULATION1DAVE_H

