#include "simulation1dave.h"

#ifdef SIMICONDUCTOR_CONFIG_AVE

#include "simulationstate.h"
#include "ave.h"
#include "constants.h"
#include "dissociationprobability.h"
#include <serut/fileserializer.h>
#include <gsl/gsl_sf.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <iostream>
#include <cmath>
#include <math.h>
#include <climits>

using namespace serut;

// NOTE: All units are SI units. Shouldn't be a problem for 'double' representation of real numbers.

class DensScaleStuff1D
{
public:
	DensScaleStuff1D(long double *pScale, std::vector<double> &n, std::vector<double> &p, int w)
	{
		m_scaleBackup = *pScale;
		m_pScale = pScale;

//		std::cout << "Saving scale " << m_scaleBackup << std::endl;

		if (m_scaleBackup < 0)
		{
			double newScale = n[0];

			double N1 = n[0];
			double N2 = n[w-1];
			double P1 = p[0];
			double P2 = p[w-1];

			if (N1 > newScale)
				newScale = N1;
			if (N2 > newScale)
				newScale = N2;
			if (P1 > newScale)
				newScale = P1;
			if (P2 > newScale)
				newScale = P2;

			*pScale = newScale;

	//		std::cout << "Storing new scale " << newScale << std::endl;
		}
	}

	~DensScaleStuff1D()
	{
		*m_pScale = m_scaleBackup;
	//	std::cout << "Restoring old scale " << m_scaleBackup << std::endl;
	}
private:
	long double m_scaleBackup;
	long double *m_pScale;
};

class DiffScaleStuff1D
{
public:
	DiffScaleStuff1D(long double *pScale, std::vector<double> &dn, std::vector<double> &dp, int w)
	{
		m_scaleBackup = *pScale;
		m_pScale = pScale;

		if (m_scaleBackup < 0)
		{
			/*
			double newScale = dn[0];

			for (int y = 0 ; y < h ; y++)
			{
				for (int x = 0 ; x < w ; x++)
				{
					int idx = x+y*w;
					double dN = dn[idx];
					double dP = dp[idx];

					if (dN > newScale)
						newScale = dN;
					if (dP > newScale)
						newScale = dP;
				}
			}
			*/
			double newScale = 0;

			for (int x = 0 ; x < w ; x++)
			{
				double dN = dn[x];
				double dP = dp[x];

				newScale += dN;
				newScale += dP;
			}
			newScale /= (double)(w*2);

			*pScale = newScale;
	
			//std::cout << "Storing new diffusion scale " << newScale << std::endl;
		}
	}

	~DiffScaleStuff1D()
	{
		*m_pScale = m_scaleBackup;
	}
private:
	long double m_scaleBackup;
	long double *m_pScale;
};

// TODO; this is for linear scale only
class Simulation1DAVE::AVESearch : public AdditiveVectorExtrapolation<double>
{
public:
	AVESearch(size_t dim, bool secOrder) : AdditiveVectorExtrapolation(dim, secOrder)
	{
		m_gradient.resize(dim);
		m_epsilon = 1;
	}

	~AVESearch()
	{
	}

	void setEpsilon(double eps)
	{
		m_epsilon = eps;
	}

	double *getGradientBuffer()
	{
		return &(m_gradient[0]);
	}

	void getNextPosition(double *pNextPos, const double *pCurPos)
	{
		size_t dim = m_gradient.size();

		for (size_t i = 0 ; i < dim ; i++)
			pNextPos[i] = pCurPos[i] - m_epsilon * m_gradient[i];
	}

	void onExtrapolatedPosition(double *pPos)
	{
		// TODO??
	}
private:
	std::vector<double> m_gradient;
	double m_epsilon;
};

Simulation1DAVE::Simulation1DAVE()
{
	m_init = false;

	m_height = 0;
	m_totalPixels = 0;
	m_pixelHeight2 = 0;
	m_deltaPhi = 0;

	m_DL2 = 0;
	m_DLy2 = 0;
	m_numVariablePixels = 0;

	m_pAVESearch = 0;

	m_simpleRecombination = true;
	m_pDissProb = 0;
}

Simulation1DAVE::~Simulation1DAVE()
{
	if (m_pAVESearch)
		delete m_pAVESearch;
	if (m_pDissProb)
		delete m_pDissProb;
}

bool Simulation1DAVE::init(int height, double realHeight, double T, double diffScale, double densScale, bool useLog)
{
	if (m_init)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (height < 3)
	{
		setErrorString("Invalid grid dimensions");
		return false;
	}
	// Calculate the number of pixels on the highest resolution grid.

	int yPixels = height;

	m_height = yPixels;
	m_totalPixels = m_height; // TODO: remove these

	m_pixelHeight2 = realHeight/(double)(yPixels-1);

	m_DL2 = m_pixelHeight2 * m_pixelHeight2;
	m_DLy2 = 1.0; // TODO: remove these
	m_numVariablePixels = m_height-2;

	// Allocate enough memory for various arrays
	resizeArrays();

	m_deltaPhi = 0;

	m_pAVESearch = new AVESearch(m_numVariablePixels*3, true);

	m_init = true;
	m_kTev = T*CONST_K/CHARGE_ELECTRON;
	m_T = T;
	m_diffScale = diffScale;
	m_densScale = densScale;
	m_useLog = useLog;

	m_simpleRecombination = true;
	m_a = 0;
	m_kf = 0;

	return true;
}

bool Simulation1DAVE::start(double &sigma)
{
	if (!m_init)
	{
		setErrorString("Simulation hasn't been initialized yet");
		return false;
	}

	DensScaleStuff1D ds(&m_densScale, m_n, m_p, m_height);
	DiffScaleStuff1D diffscale(&m_diffScale, m_De, m_Dh, m_height);

	// force boundary conditions on potential
	
	m_potential[0] = 0;
	m_potential[m_height-1] = m_deltaPhi;

	// calculate factors and currents needed later
	
	calcFactorsAndCurrents();

	double newSigma = 0;
	double sum = 0;
	double factor = 1.0/(double)(m_numVariablePixels*3);

	for (int y = 1 ; y < m_height-1 ; y++)
	{
		double d = 0;

		d = getFV(y);
		sum += d*d;
	}
	for (int y = 1 ; y < m_height-1 ; y++)
	{
		double d = 0;

		d = getFN(y);
		sum += d*d;
	}
	for (int y = 1 ; y < m_height-1 ; y++)
	{
		double d = 0;

		d = getFP(y);
		sum += d*d;
	}

	newSigma = sum*factor;
		
	// TODO: this will just result in a gradient search
	// TODO: this is for linear scale, will still need some modifications for log scale
	
	std::vector<double> curState(m_numVariablePixels*3);

	for (int i = 0 ; i < m_numVariablePixels ; i++)
	{
		curState[i] = m_potential[i+1];
		curState[i+m_numVariablePixels] = m_n[i+1];
		curState[i+m_numVariablePixels*2] = m_p[i+1];
	}

	m_pAVESearch->setStartingPoint(&(curState[0]));
	
	double *pGradientBuffer = m_pAVESearch->getGradientBuffer();

	for (int k = 0 ; k < m_numVariablePixels ; k++)
	{
		int iStart = k-1;
		int iStop = k+1;

		if (iStart < 0)
			iStart = 0;
		if (iStop > m_numVariablePixels-1)
			iStop = m_numVariablePixels-1;

		pGradientBuffer[k] = 0;
		pGradientBuffer[k+m_numVariablePixels] = 0;
		pGradientBuffer[k+m_numVariablePixels*2] = 0;

		for (int i = iStart ; i <= iStop ; i++)
		{
			pGradientBuffer[k] += 2.0*(getFV(i+1)*getFJacVV(i+1,k+1) + getFN(i+1)*getFJacNV(i+1,k+1) + getFP(i+1)*getFJacPV(i+1,k+1) );
			pGradientBuffer[k+m_numVariablePixels] += 2.0*(getFV(i+1)*getFJacVN(i+1,k+1) + getFN(i+1)*getFJacNN(i+1,k+1) + getFP(i+1)*getFJacPN(i+1,k+1) );
			pGradientBuffer[k+m_numVariablePixels*2] += 2.0*(getFV(i+1)*getFJacVP(i+1,k+1) + getFN(i+1)*getFJacNP(i+1,k+1) + getFP(i+1)*getFJacPP(i+1,k+1) );
		}

		pGradientBuffer[k] *= factor;
		pGradientBuffer[k+m_numVariablePixels] *= factor;
		pGradientBuffer[k+m_numVariablePixels*2] *= factor;

//		std::cout << pGradientBuffer[k] << "\t" << pGradientBuffer[k+m_numVariablePixels] << "\t" << pGradientBuffer[k+m_numVariablePixels*2] << std::endl;
	}

#if 0
	double baseSigma = newSigma;

	for (int k = 0 ; k < m_numVariablePixels ; k++)
	{
		double diff = 0.000000001 * m_densScale;
		double tmp;

		//tmp = m_potential[k+1];
		//m_potential[k+1] += diff;

		tmp = m_p[k+1];
		m_p[k+1] += diff;

		calcFactorsAndCurrents();
		

		double newSigma = 0;
		double sum = 0;
		double factor = 1.0/(double)(m_numVariablePixels*3);

		for (int y = 1 ; y < m_height-1 ; y++)
		{
			double d = 0;

			d = getFV(y);
			sum += d*d;
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			double d = 0;

			d = getFN(y);
			sum += d*d;
		}
		for (int y = 1 ; y < m_height-1 ; y++)
		{
			double d = 0;

			d = getFP(y);
			sum += d*d;
		}

		newSigma = sum*factor;

		//m_potential[k+1] = tmp;
		m_p[k+1] = tmp;

		double deriv = (newSigma-baseSigma)/diff;
		std::cout << deriv << std::endl;
	}

	exit(-1);
#endif
	m_pAVESearch->setEpsilon(1e-1);
	m_pAVESearch->step();

	const double *pAccelState = m_pAVESearch->getAcceleratedPosition();

	for (int i = 0 ; i < m_numVariablePixels ; i++)
	{
		m_potential[i+1] = pAccelState[i];
		m_n[i+1] = pAccelState[i+m_numVariablePixels];
		m_p[i+1] = pAccelState[i+m_numVariablePixels*2];
	}

	sigma = newSigma;

	return true;
}

void Simulation1DAVE::initPotential()
{
	for (int y = 0 ; y < m_height ; y++)
	{
		double V = (double)y/(double)(m_height-1) * m_deltaPhi;

		m_potential[y] = V;
	}
	
	for (int y = 0 ; y < m_height ; y++)
	{
		double lN = (double)y/(double)(m_height-1) * (std::log(m_n[(m_height-1)]) 
							    - std::log(m_n[0])) + std::log(m_n[0]);

		m_n[y] = std::exp(lN);

		double lP = (double)y/(double)(m_height-1) * (std::log(m_p[(m_height-1)]) 
							    - std::log(m_p[0])) + std::log(m_p[0]);

		m_p[y] = std::exp(lP);

	}
}

void Simulation1DAVE::calculateXCurrent(double &bottomAvg, double &topAvg, double &overallAvg, double &center) const
{
	double sum = 0;
	double sum2 = 0;
	double sum3 = 0;
	double sum4 = 0;

	{
		for (int y = 0 ; y < m_height-1 ; y++)
		{
			int idx = y;

			double JeyCur = m_numCurTotEy[idx];
			double JhyCur = m_numCurTotHy[idx];
			
			sum += -JeyCur+JhyCur;
		}

		int idx = 0;

		double JeyCur = m_numCurTotEy[idx];
		double JhyCur = m_numCurTotHy[idx];

		sum2 += -JeyCur + JhyCur;

		idx = (m_height-2);

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum3 += -JeyCur + JhyCur;

		idx = (m_height/2);

		JeyCur = m_numCurTotEy[idx];
		JhyCur = m_numCurTotHy[idx];

		sum4 += -JeyCur + JhyCur;

	}

	sum /= (m_height-1);

	sum /= m_pixelHeight2;
	sum2 /= m_pixelHeight2;
	sum3 /= m_pixelHeight2;
	sum4 /= m_pixelHeight2;

	bottomAvg = sum2;
	topAvg = sum3;
	overallAvg = sum;
	center = sum4;
}

void Simulation1DAVE::resizeArrays()
{
	m_n.resize(m_totalPixels);
	m_ni.resize(m_totalPixels);
	m_p.resize(m_totalPixels);
	m_background.resize(m_totalPixels);
	m_generationRate.resize(m_totalPixels);
	m_recombinationFactor.resize(m_totalPixels);
	m_De.resize(m_totalPixels);
	m_Dh.resize(m_totalPixels);
	m_eMob.resize(m_totalPixels);
	m_hMob.resize(m_totalPixels);
	m_epsRels.resize(m_totalPixels);
	m_potential.resize(m_totalPixels);
	m_P.resize(m_totalPixels);

	// Initialize the values of some arrays

	setValues(m_n, 0);
	setValues(m_ni, 0);
	setValues(m_p, 0);
	setValues(m_background, 0);
	setValues(m_generationRate, 0);
	setValues(m_recombinationFactor, 0);
	setValues(m_De, 0);
	setValues(m_Dh, 0);
	setValues(m_eMob, 0);
	setValues(m_hMob, 0);
	setValues(m_epsRels, 1);
	setValues(m_background, 0);
	setValues(m_P, 0);

	// Allocate memory for the arrays which will contain the number currents
	// due to diffusion and electric field respectively

	m_numCurTotEy.resize(m_totalPixels);
	m_numCurTotHy.resize(m_totalPixels);
}

void Simulation1DAVE::writePlotData(FILE *pFile)
{
	for (int y = 0 ; y < m_height ; y++)
	{
		{
			int idx = y;
			double j1n, j1p;
			double j2 = 0;

			if (y == m_height-1)
			{
				j1n = m_numCurTotEy[idx-1];
				j1p = m_numCurTotHy[idx-1];
			}
			else
			{
				j1n = m_numCurTotEy[idx];
				j1p = m_numCurTotHy[idx];
			}
			j2 = j1p-j1n;

			double U = 0;

			if (m_simpleRecombination)
				U = m_generationRate[idx]-m_recombinationFactor[idx]*(m_n[idx]*m_p[idx] - m_ni[idx]*m_ni[idx]);
			else
			{
				double P = 0;

				if (idx == 0)
					P = m_P[1];
				else if (idx == m_height-1)
					P = m_P[m_height-2];
				else
					P = m_P[idx];

				U = P*m_generationRate[idx] - (1.0-P)*m_recombinationFactor[idx]*(m_n[idx]*m_p[idx] - m_ni[idx]*m_ni[idx]);
			}
			fprintf(pFile, "%d %g %g %g %g %g  %g %g %g %g %g %g %g %g %g\n", y, 
					         (double)m_n[idx], (double)m_p[idx], (double)m_potential[idx], 
					         (double)(m_recombinationFactor[idx]*m_n[idx]*m_p[idx]),
						 (double)(j1n/m_pixelHeight2), (double)(j1p/m_pixelHeight2), 
						 (double)(j2/m_pixelHeight2), (double)U, 
						 (double)m_generationRate[idx], (double)m_recombinationFactor[idx],
						 (double)m_De[idx], (double)m_Dh[idx], (double)m_eMob[idx], (double)m_hMob[idx]);
		}
	}
	fprintf(pFile, "\n\n");

	fprintf(pFile, "# %d\n# %d\n# %g\n# %g\n# %g\n# %d\n# %g\n", m_height, m_totalPixels, m_pixelHeight2, m_DL2, m_DLy2, m_numVariablePixels, m_deltaPhi);
	fprintf(pFile, "# %g\n# %g\n# %g\n", (double)m_diffScale, (double)m_kTev, (double)m_densScale);
	fprintf(pFile, "# %g\n# %g\n# %g\n", m_T, m_a, m_kf);
}

long double Simulation1DAVE::getFValue(int Ftype, int yPix)
{
	long double value = -12345;

	switch(Ftype)
	{
	case 0:
		value = getFV(yPix);
		break;
	case 1:
		value = getFN(yPix);
		break;
	case 2:
		value = getFP(yPix);
		break;
	default:
		std::cerr << "Simulation1DAVE::getFValue: invalid type " << Ftype << std::endl;
	}

	return value;
}

long double Simulation1DAVE::getFV(int y)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = -2.0 * m_DLy2 * m_epsRels[idx] * m_potential[idx];

	sum += m_potential[upIndex] * m_DLy2 * 0.5 * (m_epsRels[idx] + m_epsRels[upIndex]);
	sum += m_potential[downIndex] * m_DLy2 * 0.5 * (m_epsRels[idx] + m_epsRels[downIndex]);

	sum += m_DL2*(m_p[idx] - m_n[idx] + m_background[idx])*CHARGE_ELECTRON/CONST_EPSILON0;

	return sum;
}

long double Simulation1DAVE::getFN(int y)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = getNetGen(y)*m_DL2;

	sum -= m_DLy2*(m_numCurTotEy[idx] - m_numCurTotEy[downIndex]);

	if (m_useLog)
		return sum/m_n[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation1DAVE::getFP(int y)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = getNetGen(y)*m_DL2;

	sum -= m_DLy2*(m_numCurTotHy[idx] - m_numCurTotHy[downIndex]);

	if (m_useLog)
		return sum/m_p[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation1DAVE::getFJacValue(int Ftype, int yPix, int varType, int v)
{
	int yPix2 = v;

	long double value = -12345;

	switch(Ftype)
	{
	case 0:
		value = getFJacV(varType, yPix, yPix2);
		break;
	case 1:
		value = getFJacN(varType, yPix, yPix2);
		break;
	case 2:
		value = getFJacP(varType, yPix, yPix2);
		break;
	default:
		std::cerr << "Simulation1DAVE::getFJacValue: invalid type " << Ftype << std::endl;
	}

	return value;
}

long double Simulation1DAVE::getFJacV(int t, int y, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacVV(y, v);
		break;
	case 1:
		value = getFJacVN(y, v);
		break;
	case 2:
		value = getFJacVP(y, v);
		break;
	default:
		std::cerr << "Simulation1DAVE::getFJacV: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation1DAVE::getFJacN(int t, int y, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacNV(y, v);
		break;
	case 1:
		value = getFJacNN(y, v);
		break;
	case 2:
		value = getFJacNP(y, v);
		break;
	default:
		std::cerr << "Simulation1DAVE::getFJacN: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation1DAVE::getFJacP(int t, int y, int v)
{
	long double value = -12345;

	switch(t)
	{
	case 0:
		value = getFJacPV(y, v);
		break;
	case 1:
		value = getFJacPN(y, v);
		break;
	case 2:
		value = getFJacPP(y, v);
		break;
	default:
		std::cerr << "Simulation1DAVE::getFJacP: invalid type " << t << std::endl;
	}

	return value;
}

long double Simulation1DAVE::getFJacVV(int y, int v)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = 0;

	if (y == v)
		sum += -2.0*m_DLy2*m_epsRels[idx];
	if (y+1 == v)
		sum += m_DLy2*0.5*(m_epsRels[idx]+m_epsRels[upIndex]);
	if (y-1 == v)
		sum += m_DLy2*0.5*(m_epsRels[idx]+m_epsRels[downIndex]);

	return sum;
}

long double Simulation1DAVE::getFJacVN(int y, int v)
{
	int idx = y;
	long double sum = 0;
	
	if (y == v)
		sum += -m_DL2*CHARGE_ELECTRON/CONST_EPSILON0;

	if (m_useLog)
		return sum*m_n[idx]/m_kTev;
	return sum;
}

long double Simulation1DAVE::getFJacVP(int y, int v)
{
	int idx = y;
	long double sum = 0;

	if (y == v)
		sum += m_DL2*CHARGE_ELECTRON/CONST_EPSILON0;

	if (m_useLog)
		return sum*m_p[idx]/m_kTev;
	return sum;
}

long double Simulation1DAVE::getFJacNV(int y, int v)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivV(y, v);

	int d1 = (y+1 == v) - (y == v);
	int d2 = (y == v) - (y-1 == v);

	if (d1 || d2)
	{
		long double F1 = (long double)d1;
		long double F2 = (long double)d2;
		long double xUp = ((m_eMob[idx]+m_eMob[upIndex])/(m_De[idx]+m_De[upIndex])) * ( (m_potential[upIndex]) - (m_potential[idx]) );
		long double xDown = ((m_eMob[idx]+m_eMob[downIndex])/(m_De[idx]+m_De[downIndex])) * ( (m_potential[idx]) - (m_potential[downIndex]) );

		sum -= m_DLy2*( 0.5*(m_eMob[idx]+m_eMob[upIndex])*calcDerivJx(xUp, m_n[idx], m_n[upIndex])*F1
			       -0.5*(m_eMob[idx]+m_eMob[downIndex])*calcDerivJx(xDown, m_n[downIndex], m_n[idx])*F2);
	}

	if (m_useLog)
		return sum/m_n[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation1DAVE::getFJacNN(int y, int v)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivN(y, v);

	int d1 = (y == v);
	int d2 = (y+1 == v);
	int d3 = (y-1 == v);

	if (d1 || d2 || d3)
	{
		long double F1 = (long double)d1;
		long double F2 = (long double)d2;
		long double F3 = (long double)d3;

		long double xUp = ((m_eMob[idx]+m_eMob[upIndex])/(m_De[idx]+m_De[upIndex])) * ( (m_potential[upIndex]) - (m_potential[idx]) );
		long double xDown = ((m_eMob[idx]+m_eMob[downIndex])/(m_De[idx]+m_De[downIndex])) * ( (m_potential[idx]) - (m_potential[downIndex]) );

		sum -= m_DLy2*(
			0.5*(m_De[idx]+m_De[upIndex])
			   *(F1*calcDerivJd1(xUp, m_n[idx], m_n[upIndex])
			    +F2*calcDerivJd2(xUp, m_n[idx], m_n[upIndex]) )
		       -0.5*(m_De[idx]+m_De[downIndex])
			   *(F3*calcDerivJd1(xDown, m_n[downIndex], m_n[idx])
			    +F1*calcDerivJd2(xDown, m_n[downIndex], m_n[idx]) )
			);
	}


	if (m_useLog)
	{
		if (sum != 0)
		{
			int idx2 = v;

			sum *= (m_n[idx2]/m_n[idx]);
			sum /= m_diffScale;
		}

		if (y == v)
			sum -= getFN(y)/m_kTev; // getFN has already been divided by diffscale

		return sum;
	}
	else
		sum = sum/m_densScale*m_kTev/m_diffScale;

	return sum;
}

long double Simulation1DAVE::getFJacNP(int y, int v)
{
	long double val = m_DL2*getNetGenDerivP(y, v);
	int idx2 = v;
	int idx = y;

	if (m_useLog)
	{
		if (val != 0)
			val *= (m_p[idx2]/m_n[idx])/m_diffScale;
	}
	else
		val = val/m_densScale*m_kTev/m_diffScale;

	return val;
}

long double Simulation1DAVE::getFJacPV(int y, int v)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivV(y, v);

	int d1 = (y+1 == v) - (y == v);
	int d2 = (y == v) - (y-1 == v);

	if (d1 || d2)
	{
		long double F1 = (long double)d1;
		long double F2 = (long double)d2;
		long double xUp = -((m_hMob[idx]+m_hMob[upIndex])/(m_Dh[idx]+m_Dh[upIndex])) * ( (m_potential[upIndex]) - (m_potential[idx]) );
		long double xDown = -((m_hMob[idx]+m_hMob[downIndex])/(m_Dh[idx]+m_Dh[downIndex])) * ( (m_potential[idx]) - (m_potential[downIndex]) );

		sum -= m_DLy2*(-0.5*(m_hMob[idx]+m_hMob[upIndex])*calcDerivJx(xUp, m_p[idx], m_p[upIndex])*F1
			       +0.5*(m_hMob[idx]+m_hMob[downIndex])*calcDerivJx(xDown, m_p[downIndex], m_p[idx])*F2);
	}

	if (m_useLog)
		return sum/m_p[idx]*m_kTev/m_diffScale;
	return sum/m_densScale*m_kTev/m_diffScale;
}

long double Simulation1DAVE::getFJacPN(int y, int v)
{
	long double val = m_DL2*getNetGenDerivN(y, v);
	int idx2 = v;
	int idx = y;

	if (m_useLog)
	{
		if (val != 0)
			val *= (m_n[idx2]/m_p[idx])/m_diffScale;
	}
	else
		val = val/m_densScale*m_kTev/m_diffScale;

	return val;
}

long double Simulation1DAVE::getFJacPP(int y, int v)
{
	int idx = y;
	int upIndex  = idx + 1;
	int downIndex = idx - 1;

	long double sum = 0;

	sum += m_DL2*getNetGenDerivP(y, v);

	int d1 = (y == v);
	int d2 = (y+1 == v);
	int d3 = (y-1 == v);

	if (d1 || d2 || d3)
	{
		long double F1 = (long double)d1;
		long double F2 = (long double)d2;
		long double F3 = (long double)d3;

		long double xUp = -((m_hMob[idx]+m_hMob[upIndex])/(m_Dh[idx]+m_Dh[upIndex])) * ( (m_potential[upIndex]) - (m_potential[idx]) );
		long double xDown = -((m_hMob[idx]+m_hMob[downIndex])/(m_Dh[idx]+m_Dh[downIndex])) * ( (m_potential[idx]) - (m_potential[downIndex]) );

		sum -= m_DLy2*(
			0.5*(m_Dh[idx]+m_Dh[upIndex])
			   *(F1*calcDerivJd1(xUp, m_p[idx], m_p[upIndex])
			    +F2*calcDerivJd2(xUp, m_p[idx], m_p[upIndex]) )
		       -0.5*(m_Dh[idx]+m_Dh[downIndex])
			   *(F3*calcDerivJd1(xDown, m_p[downIndex], m_p[idx])
			    +F1*calcDerivJd2(xDown, m_p[downIndex], m_p[idx]) )
			);
	}

	if (m_useLog)
	{
		if (sum != 0)
		{
			int idx2 = v;

			sum *= (m_p[idx2]/m_p[idx]);
			sum /= m_diffScale;
		}

		if (y == v)
			sum -= getFP(y)/m_kTev;  // getFP has already been divided by diffScale
	}
	else
		sum = sum/m_densScale*m_kTev/m_diffScale;

	return sum;

}

long double Simulation1DAVE::getNetGen(int y)
{
	long double netGen = 0;
	
	if (m_simpleRecombination)
	{
		int idx = y;
		long double ni = m_ni[idx];

		netGen = m_generationRate[idx] - m_recombinationFactor[idx] * (m_n[idx] * m_p[idx] - ni*ni);
	}
	else
	{
		netGen = m_P[y]*m_generationRate[y] - (1.0-m_P[y]) * m_recombinationFactor[y] * (m_n[y]*m_p[y]-m_ni[y]*m_ni[y]);
	}

	return netGen;
}

long double Simulation1DAVE::getNetGenDerivV(int y, int v)
{
	long double deriv = 0;

	if (m_simpleRecombination)
	{
		deriv = 0;
	}
	else
	{
		long double dPdV = calcDerivP(y, v);
		long double R = m_recombinationFactor[y]*(m_n[y]*m_p[y] - m_ni[y]*m_ni[y]);
		
		deriv = dPdV*(m_generationRate[y]+R);
	}

	return deriv;
}

long double Simulation1DAVE::getNetGenDerivN(int y, int v)
{
	long double deriv = 0;

	if (m_simpleRecombination)
	{
		if (y == v)
		{
			int idx = y;

			deriv = -m_recombinationFactor[idx]*m_p[idx];
		}
	}
	else
	{
		if (y == v)
		{
			deriv = - (1.0-m_P[y])*m_recombinationFactor[y]*m_p[y];
		}
	}
	return deriv;
}

long double Simulation1DAVE::getNetGenDerivP(int y, int v)
{
	long double deriv = 0;

	if (m_simpleRecombination)
	{
		if (y == v)
		{
			int idx = y;

			deriv =  -m_recombinationFactor[idx]*m_n[idx];
		}
	}
	else
	{
		if (y == v)
		{
			deriv = -(1.0-m_P[y])*m_recombinationFactor[y]*m_n[y];
		}
	}
	return deriv;
}

void Simulation1DAVE::calcCurrentVertical(int y)
{
	int idx = y;
	int nextIdx = y+1;

	long double De = (m_De[idx]+m_De[nextIdx])/2.0;
	long double Dh = (m_Dh[idx]+m_Dh[nextIdx])/2.0;
	long double eMob = (m_eMob[idx]+m_eMob[nextIdx])/2.0;
	long double hMob = (m_hMob[idx]+m_hMob[nextIdx])/2.0;

	long double dPhiN = (m_potential[nextIdx]) - (m_potential[idx]);
	long double dPhiP = (m_potential[nextIdx]) - (m_potential[idx]);

	// Note: we won't divide by dx or dy here, we'll do this at another stage
	m_numCurTotEy[idx] = De * calcJ( dPhiN*eMob/De, m_n[idx], m_n[nextIdx]);
	m_numCurTotHy[idx] = Dh * calcJ( -dPhiP*hMob/Dh, m_p[idx], m_p[nextIdx]);
}

void Simulation1DAVE::calcFactorsAndCurrents()
{
	for (int y = 0 ; y < m_height-1 ; y++)
		calcCurrentVertical(y);

	if (!m_simpleRecombination)
	{
		for (int y = 1 ; y < m_height-1 ; y++)
			calcP(y);
	}
}


long double Simulation1DAVE::calcW(long double x)
{
	long double val = 0;

	if (x > 0.01)
	{
		val = x/(1.0-std::exp(-x));
	}
	else if (x < -0.01)
	{
		long double f = std::exp(x);
		
		val = x*f/(f-1.0);
	}
	else
	{
		long double x2 = x*x;
		long double x4 = x2*x2;

		val = 1.0+0.5*x+1.0/12.0*x2-1.0/720.0*x4;
	}
	return val;
}

long double Simulation1DAVE::calcDerivW(long double x)
{
	long double val = 0;

	if (x > 0.01)
	{
		long double f = std::exp(-x);
		long double oneMinusF = 1.0-f;

		val = (1.0-x*f/oneMinusF)/oneMinusF;
	}
	else if (x < -0.01)
	{
		long double f = std::exp(x);
		long double fMinusOne = f-1.0;

		val = f/fMinusOne*(1.0-x/fMinusOne);
	}
	else
	{
		long double x2 = x*x;
		long double x3 = x2*x;

		val = 0.5+1.0/6.0*x-1.0/180.0*x3;
	}

	return val;
}

long double Simulation1DAVE::calcJ(long double x, long double d1, long double d2)
{
	long double Wx = calcW(x);

	return d1*Wx+d2*(x-Wx);
}

long double Simulation1DAVE::calcDerivJx(long double x, long double d1, long double d2)
{
	long double dWdx = calcDerivW(x);

	return d1*dWdx + d2*(1.0-dWdx);
}

long double Simulation1DAVE::calcDerivJd1(long double x, long double d1, long double d2)
{
	return calcW(x);
}

long double Simulation1DAVE::calcDerivJd2(long double x, long double d1, long double d2)
{
	return x - calcW(x);
}

void Simulation1DAVE::calcP(int y)
{
	double rf = m_recombinationFactor[y];
	double epsRel = m_epsRels[y];
	double F = std::abs(m_potential[y+1]-m_potential[y-1])/(2.0*m_pixelHeight2);

	m_P[y] = m_pDissProb->calculate(rf, epsRel, F);
}

long double Simulation1DAVE::calcDerivP(int y, int v)
{
	if (!(y+1 == v || y-1 == v))
		return 0;

	double rf = m_recombinationFactor[y];
	double epsRel = m_epsRels[y];
	double F = std::abs(m_potential[y+1]-m_potential[y-1])/(2.0*m_pixelHeight2);

	double dPdF = m_pDissProb->calculateDerivF(rf, epsRel, F);

	double sign = 1.0;

	if (m_potential[y+1] < m_potential[y-1])
		sign = -1.0;

	int delta1 = (y+1 == v);
	int delta2 = (y-1 == v);
	double d1 = (double)delta1;
	double d2 = (double)delta2;

	return dPdF*sign*(d1-d2)/(2.0*m_pixelHeight2);
}

void Simulation1DAVE::setExtendedRecombinationModel(double pairDistance, double kf)
{
	m_simpleRecombination = false;
	
	if (m_pDissProb)
		delete m_pDissProb;

	m_a = pairDistance;
	m_kf = kf;
	m_pDissProb = new DissociationProbability(pairDistance, m_T, kf);
}

bool Simulation1DAVE::getExtendedRecombinationParameters(double &pairDistance, double &kf) const
{
	if (m_simpleRecombination)
	{
		setErrorString("Not using extended recombination model");
		return false;
	}

	pairDistance = m_a;
	kf = m_kf;
	return true;
}

bool Simulation1DAVE::setState(const SimulationState &state, std::vector<std::string> &warnings)
{
	if (!m_init)
	{
		setErrorString("Simulation not initialized");
		return false;
	}
	if (!state.isInitialized())
	{
		setErrorString("State to be used is not initialized");
		return false;
	}
	if (state.getDimensions() != 1)
	{
		setErrorString("State is not one-dimensional");
		return false;
	}

	if (state.getNumberOfXPixels() != m_height)
	{
		setErrorString("State does not have same number of pixels as simulation");
		return false;
	}

	std::vector<bool> m_input(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_output(SIMSTATE_GRIDPROP_MAX);
	std::vector<bool> m_optional(SIMSTATE_GRIDPROP_MAX);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		m_input[i] = false;
		m_output[i] = false;
		m_optional[i] = false;
	}

	m_input[SIMSTATE_GRIDPROP_N] = true;
	m_input[SIMSTATE_GRIDPROP_P] = true;
	m_input[SIMSTATE_GRIDPROP_V] = true;
	m_input[SIMSTATE_GRIDPROP_DN] = true;
	m_input[SIMSTATE_GRIDPROP_DP] = true;
	m_input[SIMSTATE_GRIDPROP_NMOB] = true;
	m_input[SIMSTATE_GRIDPROP_PMOB] = true;
	m_input[SIMSTATE_GRIDPROP_EPSREL] = true;

	m_output[SIMSTATE_GRIDPROP_N] = true;
	m_output[SIMSTATE_GRIDPROP_P] = true;
	m_output[SIMSTATE_GRIDPROP_JNX] = true;
	m_output[SIMSTATE_GRIDPROP_JPX] = true;
	m_output[SIMSTATE_GRIDPROP_R] = true;
	m_output[SIMSTATE_GRIDPROP_DISSPROB] = true;

	m_optional[SIMSTATE_GRIDPROP_BG] = true;
	m_optional[SIMSTATE_GRIDPROP_NI] = true;
	m_optional[SIMSTATE_GRIDPROP_G] = true;
	m_optional[SIMSTATE_GRIDPROP_RF] = true;

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
	{
		if (m_input[i])
		{
			if (!state.isGridPropertySet(i))
			{
				setErrorString("Grid property '" + state.getGridPropertyName(i) + "' is required, but has not been set");
				return false;
			}
		}
		else
		{
			if (state.isGridPropertySet(i))
			{
				if (!m_output[i] && !m_optional[i])
					warnings.push_back("Property '" + state.getGridPropertyName(i) + "' is set but not used");
			}
		}
	}

	state.getGridProperty(SIMSTATE_GRIDPROP_N, m_n);
	state.getGridProperty(SIMSTATE_GRIDPROP_P, m_p);
	state.getGridProperty(SIMSTATE_GRIDPROP_NI, m_ni);
	state.getGridProperty(SIMSTATE_GRIDPROP_BG, m_background);
	state.getGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate);
	state.getGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor);
	state.getGridProperty(SIMSTATE_GRIDPROP_DN, m_De);
	state.getGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh);
	state.getGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob);
	state.getGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob);
	state.getGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels);
	state.getGridProperty(SIMSTATE_GRIDPROP_V, m_potential);

	double vDiff;

	if (!state.getDoubleProperty(SIMSTATE_PROP_VDIFF, vDiff))
	{
		setErrorString("Potential difference has not been set");
		return false;
	}

	m_deltaPhi = vDiff;

	if (state.getRecombinationModel() == SimulationState::Braun)
	{
		double a, kf;

		if (!state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a) ||
		    !state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
		{
			setErrorString("Braun recombination model selected, but necessary parameters are not set");
			return false;
		}

		setExtendedRecombinationModel(a, kf);	
	}
	else
	{
		double a, kf;

		if (state.getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
			warnings.push_back("'Pair distance' parameter is set, but is only used in the braun model");
		if (state.getDoubleProperty(SIMSTATE_PROP_KF, kf))
			warnings.push_back("'Dissociation rate' parameter 'kf' is set, but is only used in the braun model");

		setNormalRecombinationModel();
	}

	//FILE *f = fopen("log.txt","wt");
	//writePlotData(f);
	//fclose(f);

	return true;
}

bool Simulation1DAVE::storeState(SimulationState &state) const
{
	if (!m_init)
	{
		setErrorString("Not initialized");
		return false;
	}

	if (state.getNumberOfXPixels() != m_height)
	{
		setErrorString("Number of pixels is not same as in simulation state");
		return false;
	}

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		state.clearDoubleProperty(i);

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		state.clearGridProperty(i);

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_N, m_n) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_P, m_p) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_V, m_potential))
	{
		setErrorString("Unable to store n,p or V");
		return false;
	}

	if (!state.setGridProperty(SIMSTATE_GRIDPROP_BG, m_background) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NI, m_ni) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_G, m_generationRate) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_RF, m_recombinationFactor) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DN, m_De) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_DP, m_Dh) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_NMOB, m_eMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_PMOB, m_hMob) ||
	    !state.setGridProperty(SIMSTATE_GRIDPROP_EPSREL, m_epsRels))
	{
		setErrorString("Unable to store fixed simulation items");
		return false;
	}

	for (int y = 0 ; y < m_height ; y++)
	{
		double jn = m_numCurTotEy[y]/m_pixelHeight2;
		double jp = m_numCurTotHy[y]/m_pixelHeight2;

		if (!state.setGridProperty(SIMSTATE_GRIDPROP_JNX, y, jn) ||
		    !state.setGridProperty(SIMSTATE_GRIDPROP_JPX, y, jp))
		{
			setErrorString("Unable to store the number current");
			return false;
		}
	}

	if (m_simpleRecombination)
	{
		for (int idx = 0 ; idx < m_height ; idx++)
		{
			long double ni = m_ni[idx];
			long double R = m_recombinationFactor[idx] * (m_n[idx] * m_p[idx] - ni*ni);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, idx, (double)R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
		}

		state.setRecombinationModel(SimulationState::Simple);	
	}
	else
	{
		for (int y = 0 ; y < m_height ; y++)
		{
			int idx = y;

			if (idx == 0)
				idx = 1;
			else if (idx == m_height-1)
				idx = m_height-2;

			long double ni = m_ni[idx];
			long double dissProb = m_P[idx];
			long double R = (1.0-dissProb) * m_recombinationFactor[idx] * (m_n[idx] * m_p[idx] - ni*ni);
		
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_R, y, (double)R))
			{
				setErrorString("Unable to store the recombination rate");
				return false;
			}
			if (!state.setGridProperty(SIMSTATE_GRIDPROP_DISSPROB, y, (double)dissProb))
			{
				setErrorString("Unable to store the dissociation rate");
				return false;
			}
		}

		state.setRecombinationModel(SimulationState::Braun);

		if (!state.setDoubleProperty(SIMSTATE_PROP_KF, m_kf) ||
		    !state.setDoubleProperty(SIMSTATE_PROP_PAIRDIST, m_a))
		{
			setErrorString("Unable to store recombination model properties");
			return false;
		}
	}

	if (!state.setDoubleProperty(SIMSTATE_PROP_VDIFF, m_deltaPhi))
	{
		setErrorString("Couldn't store potential difference");
		return false;
	}

	if (!state.setDoubleProperty(SIMSTATE_PROP_TEMP, m_T))
	{
		setErrorString("Couldn't store temperature");
		return false;
	}

	return true;
}

#endif // SIMICONDUCTOR_CONFIG_AVE
