#include "simimpimoga.h"
#include "simiconductorinstance.h"
#include "extproggenome.h"
#include <mogal/genome.h>
#include <mogal/gafactory.h>
#include <shellp/fileptriosystem.h>
#include <stdio.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <iostream>

#define MAXLEN 1024

using namespace mogal;
using namespace shellp;

#ifdef WIN32
#define popen _popen
#define pclose _pclose
#endif // WIN32

SimiMPIMOGA::SimiMPIMOGA(const std::string &translateCmd, GAFactory *pFactory)
{
	m_translateError = false;
	if (translateCmd.length() == 0 || translateCmd == std::string("*"))
		m_gotTranslateCmd = false;
	else
	{
		m_translateCmd = translateCmd;
		m_gotTranslateCmd = true;
	}

	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	m_pIOSys = pInst->getIOSystem();
	m_pFactory = pFactory;
}

SimiMPIMOGA::~SimiMPIMOGA()
{
}

void SimiMPIMOGA::onCurrentBest(const std::list<Genome *> &bestGenomes) const
{
	if (bestGenomes.empty())
	{
		m_pIOSys->writeOutputLine("No preferred genome found yet");
	}
	else
	{
		ExtProgGenome *pGenome = (ExtProgGenome *)m_pFactory->selectPreferredGenome(bestGenomes);
		std::string fitnessStr = getFitnessString(pGenome);

		if (!m_gotTranslateCmd || m_translateError)
		{
			m_pIOSys->print("Current preferred (%s): %s", fitnessStr.c_str(), pGenome->getParameterString().c_str());
		}
		else
		{
			std::string cmd = m_translateCmd;
			std::string bestPointString = pGenome->getParameterString();

			cmd += std::string(" ");
			cmd += bestPointString;

			FILE *pFile = popen(cmd.c_str(), "r");
			if (pFile == 0)
			{
				m_translateError = true;
				m_pIOSys->print("Warning: couldn't execute translate command: %s", strerror(errno));
				m_pIOSys->print("Current preferred (%s): %s", fitnessStr.c_str(), pGenome->getFitnessValues()[0], pGenome->getParameterString().c_str());
			}
			else
			{
				m_pIOSys->print("Current preferred (%s):", fitnessStr.c_str());

				FilePtrIOSystem io(pFile, 0);
				std::string nullPrefix, line;
				
				while (io.readInputLine(nullPrefix, line))
					m_pIOSys->writeOutputLine(line);

				int status;

				if ((status = pclose(pFile)) != 0)
				{
					m_translateError = true;
					m_pIOSys->print("Warning: got non-zero return value (%d) from translate command", status);
					m_pIOSys->print("Current preferred (%s): %s", fitnessStr.c_str(), pGenome->getFitnessValues()[0], pGenome->getParameterString().c_str());
				}
			}
		}
	}
}

void SimiMPIMOGA::feedbackStatus(const std::string &str) const
{
	m_pIOSys->writeOutputLine("GA feedback: " + str);
}

void SimiMPIMOGA::writeLog(int level, const char *fmt, ...)
{
/*	va_list ap;
	char str[1024];

	va_start(ap, fmt);
	vsnprintf(str, 1024, fmt, ap);
	va_end(ap);

	std::cerr << "writeLog: " << str << std::endl;*/
}

void SimiMPIMOGA::printSolutions(bool final) const
{
	std::vector<Genome *> bestGenomes;

	getBestGenomes(bestGenomes);
	if (bestGenomes.empty())
		m_pIOSys->writeOutputLine("No preferred genome found");
	else
	{
		if (final)
			m_pIOSys->print("Final non-dominated set (%d genomes): ", bestGenomes.size());
		else
			m_pIOSys->print("Current non-dominated set (%d genomes): ", bestGenomes.size());

		for (int i = 0 ; i < bestGenomes.size() ; i++)
		{
			ExtProgGenome *pGenome = (ExtProgGenome *)bestGenomes[i];
			std::string fitnessStr = getFitnessString(pGenome);

			if (!m_gotTranslateCmd || m_translateError)
			{
				m_pIOSys->print("Genome %d (%s): %s", (i+1), fitnessStr.c_str(), pGenome->getParameterString().c_str());
			}
			else
			{
				std::string cmd = m_translateCmd;
				std::string bestPointString = pGenome->getParameterString();

				cmd += std::string(" ");
				cmd += bestPointString;

				FILE *pFile = popen(cmd.c_str(), "r");
				if (pFile == 0)
				{
					m_translateError = true;
					m_pIOSys->print("Warning: couldn't execute translate command: %s", strerror(errno));
					m_pIOSys->print("Genome %d (%s): %s", (i+1), fitnessStr.c_str(), pGenome->getParameterString().c_str());
				}
				else
				{
					m_pIOSys->print("Genome %d (%s):", (i+1), fitnessStr.c_str());

					FilePtrIOSystem io(pFile, 0);
					std::string nullPrefix, line;
					
					while (io.readInputLine(nullPrefix, line))
						m_pIOSys->writeOutputLine(line);

					int status;

					if ((status = pclose(pFile)) != 0)
					{
						m_translateError = true;
						m_pIOSys->print("Warning: got non-zero return value (%d) from translate command", status);
						m_pIOSys->print("Genome %d (%s): %s", (i+1), fitnessStr.c_str(), pGenome->getParameterString().c_str());
					}
				}
			}
		}
	}
}

std::string SimiMPIMOGA::getFitnessString(ExtProgGenome *pGenome)
{
	std::string fitnessString;
	char str[256];
	const double *pFitnessValues = pGenome->getFitnessValues();
	int numValues = pGenome->getNumberOfFitnessValues();

	sprintf(str, "%.5g", pFitnessValues[0]);
	fitnessString += str;

	for (int i = 1 ; i < numValues ; i++)
	{
		sprintf(str, " %.5g", pFitnessValues[i]);
		fitnessString += str;
	}
	return fitnessString;
}

bool SimiMPIMOGA::onAlgorithmLoop(GAFactory &factory, bool generationInfoChanged)
{
	printSolutions(false);
	return MPIGeneticAlgorithm::onAlgorithmLoop(factory, generationInfoChanged);
}
