#include "simiconductorconfig.h"
#include "simiconductorregion2d.h"
#include <stdio.h>
#ifdef SIMICONDUCTOR_CONFIG_QT4
#include <QImage>
#endif // SIMICONDUCTOR_CONFIG_QT4

SimiConductorRectRegion::SimiConductorRectRegion(int x1, int y1, int x2, int y2)
{
	int tmp;

	if (x2 < x1)
	{
		tmp = x2;
		x2 = x1;
		x1 = tmp;
	}
	if (y2 < y1)
	{
		tmp = y2;
		y2 = y1;
		y1 = tmp;
	}

	m_x1 = x1;
	m_x2 = x2;
	m_y1 = y1;
	m_y2 = y2;
}

SimiConductorRectRegion::~SimiConductorRectRegion()
{
}

SimiConductorRegion2D *SimiConductorRectRegion::createCopy() const					
{ 
	return new SimiConductorRectRegion(m_x1, m_y1, m_x2, m_y2); 
}

std::string SimiConductorRectRegion::getDescription() const
{
	char str[1024];

	sprintf(str, "Rectangular area: (%d, %d) -> (%d, %d)", m_x1, m_y1, m_x2, m_y2);

	return std::string(str);
}

bool SimiConductorRectRegion::fillPixels(std::vector<bool> &pixels, int width, int height) const
{
	int xStart = m_x1-1;
	int yStart = m_y1-1;
	int xEnd = m_x2;
	int yEnd = m_y2;

	if (xStart < 0 || yStart < 0 || xEnd > width || yEnd > height)
	{
		setErrorString("Coordinates lie outside the current grid dimensions");
		return false;
	}

	for (int y = yStart ; y < yEnd ; y++)
		for (int x = xStart ; x < xEnd ; x++)
			pixels[x+y*width] = true;

	return true;
}

SimiConductorMapRegion::SimiConductorMapRegion()
{
	m_width = 0;
	m_height = 0;
	m_xOffset = 0;
	m_yOffset = 0;
}

SimiConductorMapRegion::SimiConductorMapRegion(const std::vector<bool> map, int width, int height, int xOffset, int yOffset)
{
	m_width = width;
	m_height = height;
	m_xOffset = xOffset;
	m_yOffset = yOffset;
	m_map = map;
}

SimiConductorMapRegion::~SimiConductorMapRegion()
{
}

SimiConductorRegion2D *SimiConductorMapRegion::createCopy() const					
{ 
	return new SimiConductorMapRegion(m_map, m_width, m_height, m_xOffset, m_yOffset); 
}

bool SimiConductorMapRegion::fillPixels(std::vector<bool> &pixels, int width, int height) const
{
	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			if (m_map[x+y*m_width])
			{
				int x2 = x+m_xOffset;
				int y2 = y+m_yOffset;

				if (x2 < 0 || y2 < 0 || x2 >= width || y2 >= height)
				{
					setErrorString("Coordinates lie outside the current grid dimensions");
					return false;
				}
				pixels[x2+y2*width] = true;
			}
		}
	}

	return true;
}

std::string SimiConductorMapRegion::getDescription() const
{
	char str[1024];

	// TODO: more information
	sprintf(str, "Pixel map");

	return std::string(str);
}

bool SimiConductorMapRegion::import(const std::string &fileName, double xScale, double yScale, int xOffset, int yOffset)
{
#ifndef SIMICONDUCTOR_CONFIG_QT4
	setErrorString("Image import requires the availability of Qt4 at compile time");
	return false;
#else 
	if (xScale < 0 || yScale < 0)
	{
		setErrorString("Scale values must be positive");
		return false;
	}

	QImage img;

	if (!img.load(fileName.c_str()))
	{
		setErrorString("Unable to load data from specified file");
		return false;
	}


	int width = (int)((double)img.width() * xScale + 0.5);
	int height = (int)((double)img.height() * yScale + 0.5);
	int total = width*height;
	std::vector<bool> tmpMap(total);

	for (int y = 0 ; y < height ; y++)
	{
		for (int x = 0 ; x < width ; x++)
		{
			double xCoord = (double)x + 0.5;
			double yCoord = (double)y + 0.5;

			int xImgCoord = (int)(xCoord/xScale);
			int yImgCoord = (int)(yCoord/yScale);

			bool value = false;

			if (xImgCoord >= 0 && yImgCoord >= 0 && xImgCoord < img.width() && yImgCoord < img.height())
			{
				QRgb pixel = img.pixel(xImgCoord, yImgCoord);

				if (!(qRed(pixel) == 0 && qGreen(pixel) == 0 && qBlue(pixel) == 0))
					value = true;
			}

			// Flip it!
			tmpMap[x+(height-1-y)*width] = value;
		}
	}

	m_width = width;
	m_height = height;
	m_xOffset = xOffset;
	m_yOffset = yOffset;
	m_map = tmpMap;

	return true;
#endif // ! SIMICONDUCTOR_CONFIG_QT4
}

