#include "simiconductorinstance.h"
#include "simulation1d.h"
#include "simulation2d.h"
#include "simulationstate.h"
#include "simiconductorregion1d.h"
#include "simiconductorregion2d.h"
#include <shellp/shellcmdarg.h>
#include <shellp/shellcommand.h>
#include <shellp/iosystem.h>
#include <mogal/geneticalgorithm.h>
#include <stdlib.h>
#include <iostream>
#ifdef WIN32
#include <windows.h>
#endif // WIN32

#include "debugnew.h"

#define SIMI_CONSTNAME_INTWIDTH_1D "NX1"
#define SIMI_CONSTNAME_WIDTH_1D "W1"
#define SIMI_CONSTNAME_INTWIDTH_2D "NX2"
#define SIMI_CONSTNAME_INTHEIGHT_2D "NY2"
#define SIMI_CONSTNAME_WIDTH_2D "W2"
#define SIMI_CONSTNAME_HEIGHT_2D "H2"
#define SIMI_CONSTNAME_PI "PI"
#define SIMI_CONSTNAME_ELECTRONCHARGE "e"
#define SIMI_CONSTNAME_PLACKCONST "h"
#define SIMI_CONSTNAME_HBAR "hbar"
#define SIMI_CONSTNAME_ELECTRONMASS "me"
#define SIMI_CONSTNAME_BOLTZMANN "kB"
#define SIMI_CONSTNAME_EPSILON0 "eps0"
#define SIMI_REGIONNAME_ALL "ALL"

using namespace mogal;

double MinFunction(double a, double b)
{
	if (a < b)
		return a;
	return b;
}

double MaxFunction(double a, double b)
{
	if (a > b)
		return a;
	return b;
}

double ClipFunction(double x, double a, double b)
{
	double minValue, maxValue;

	if (a < b)
	{
		minValue = a;
		maxValue = b;
	}
	else
	{
		minValue = b;
		maxValue = a;
	}

	if (x < minValue)
		return minValue;
	if (x > maxValue)
		return maxValue;
	return x;
}

SimiConductorInstance::SimiConductorInstance()
{
	defineFunction("min", MinFunction);
	defineFunction("max", MaxFunction);
	defineFunction("clip", ClipFunction);

	defineConstant(SIMI_CONSTNAME_INTWIDTH_1D, 0);
	defineConstant(SIMI_CONSTNAME_INTWIDTH_2D, 0);
	defineConstant(SIMI_CONSTNAME_INTHEIGHT_2D, 0);
	defineConstant(SIMI_CONSTNAME_WIDTH_1D, 0);
	defineConstant(SIMI_CONSTNAME_WIDTH_2D, 0);
	defineConstant(SIMI_CONSTNAME_HEIGHT_2D, 0);
	defineConstant(SIMI_CONSTNAME_PI, CONST_PI);
	defineConstant(SIMI_CONSTNAME_ELECTRONCHARGE, CHARGE_ELECTRON);
	defineConstant(SIMI_CONSTNAME_ELECTRONMASS, MASS_ELECTRON);
	defineConstant(SIMI_CONSTNAME_PLACKCONST, CONST_H);
	defineConstant(SIMI_CONSTNAME_HBAR, CONST_HBAR);
	defineConstant(SIMI_CONSTNAME_BOLTZMANN, CONST_K);
	defineConstant(SIMI_CONSTNAME_X, 0);
	defineConstant(SIMI_CONSTNAME_PX, 1);
	defineConstant(SIMI_CONSTNAME_Y, 0);
	defineConstant(SIMI_CONSTNAME_PY, 1);
	defineConstant(SIMI_CONSTNAME_V, 0);
	defineConstant(SIMI_CONSTNAME_EPSILON0, CONST_EPSILON0);

	m_pGAParams = new GeneticAlgorithmParams(1.0, false, false, 0.9);

	m_pSimState1D = 0;
	m_pSimState2D = 0;
}

SimiConductorInstance::~SimiConductorInstance()
{
	setGAParams(0);

	setSimulationState1D(0);
	setSimulationState2D(0);
}

bool SimiConductorInstance::createNew1DRegionName(const std::string &regionName)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved region name was specified");
		return false;
	}

	return internalCreateNew1DRegionName(regionName);
}

bool SimiConductorInstance::internalCreateNew1DRegionName(const std::string &regionName)
{
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	it = m_1DregionMap.find(regionName);

	if (it != m_1DregionMap.end())
	{
		setErrorString("This region name is already in use");
		return false;
	}

	std::list<SimiConductorRegion1D *> newList;

	m_1DregionMap[regionName] = newList;

	return true;
}

bool SimiConductorInstance::append1DRegion(const std::string &regionName, const SimiConductorRegion1D &r)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved region name was specified");
		return false;
	}

	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	it = m_1DregionMap.find(regionName);

	if (it == m_1DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	it->second.push_back(r.createCopy());

	return true;
}

bool SimiConductorInstance::get1DRegions(const std::string &regionName, std::vector<const SimiConductorRegion1D *> &regions)
{
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	it = m_1DregionMap.find(regionName);

	if (it == m_1DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	std::list<SimiConductorRegion1D *>::const_iterator it2;

	regions.clear();
	for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
		regions.push_back(*it2);

	return true;
}

bool SimiConductorInstance::delete1DRegion(const std::string &regionName)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved regionname was specified");
		return false;
	}

	return internalDelete1DRegion(regionName);
}

bool SimiConductorInstance::internalDelete1DRegion(const std::string &regionName)
{
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	it = m_1DregionMap.find(regionName);

	if (it == m_1DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	std::list<SimiConductorRegion1D *>::iterator it2;

	for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
		delete *it2;

	m_1DregionMap.erase(it);

	return true;
}

void SimiConductorInstance::clear1DRegions()
{
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator tmpit;
	std::list<SimiConductorRegion1D *>::iterator it2;

	it = m_1DregionMap.begin();
	while (it != m_1DregionMap.end())
	{
		if (it->first == std::string(SIMI_REGIONNAME_ALL)) // don't delete the reserved region
			it++;
		else
		{
			for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
				delete *it2;

			tmpit = it;
			it++;

			m_1DregionMap.erase(tmpit);
		}
	}
}

void SimiConductorInstance::get1DRegionNames(std::vector<std::string> &names)
{
	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	names.clear();
	
	for (it = m_1DregionMap.begin() ; it != m_1DregionMap.end() ; it++)
	{
		std::string name = it->first;
		names.push_back(name);
	}
}

bool SimiConductorInstance::createNew2DRegionName(const std::string &regionName)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved region name was specified");
		return false;
	}

	return internalCreateNew2DRegionName(regionName);
}

bool SimiConductorInstance::internalCreateNew2DRegionName(const std::string &regionName)
{
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	it = m_2DregionMap.find(regionName);

	if (it != m_2DregionMap.end())
	{
		setErrorString("This region name is already in use");
		return false;
	}

	std::list<SimiConductorRegion2D *> newList;

	m_2DregionMap[regionName] = newList;

	return true;
}

bool SimiConductorInstance::append2DRegion(const std::string &regionName, const SimiConductorRegion2D &r)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved regionname was specified");
		return false;
	}

	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	it = m_2DregionMap.find(regionName);

	if (it == m_2DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	it->second.push_back(r.createCopy());

	return true;
}

bool SimiConductorInstance::get2DRegions(const std::string &regionName, std::vector<const SimiConductorRegion2D *> &regions)
{
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	it = m_2DregionMap.find(regionName);

	if (it == m_2DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	std::list<SimiConductorRegion2D *>::const_iterator it2;

	regions.clear();
	for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
		regions.push_back(*it2);

	return true;
}

bool SimiConductorInstance::delete2DRegion(const std::string &regionName)
{
	if (regionName == std::string(SIMI_REGIONNAME_ALL))
	{
		setErrorString("A reserved region name was specified");
		return false;
	}

	return internalDelete2DRegion(regionName);
}

bool SimiConductorInstance::internalDelete2DRegion(const std::string &regionName)
{
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	it = m_2DregionMap.find(regionName);

	if (it == m_2DregionMap.end())
	{
		setErrorString("This region name has not been created yet");
		return false;
	}

	std::list<SimiConductorRegion2D *>::iterator it2;

	for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
		delete *it2;

	m_2DregionMap.erase(it);

	return true;
}

void SimiConductorInstance::clear2DRegions()
{
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator tmpit;
	std::list<SimiConductorRegion2D *>::iterator it2;

	it = m_2DregionMap.begin();
	while (it != m_2DregionMap.end())
	{
		if (it->first == std::string(SIMI_REGIONNAME_ALL)) // don't delete the reserved region
			it++;
		else
		{
			for (it2 = it->second.begin() ; it2 != it->second.end() ; it2++)
				delete *it2;

			tmpit = it;
			it++;

			m_2DregionMap.erase(tmpit);
		}
	}
}

void SimiConductorInstance::get2DRegionNames(std::vector<std::string> &names)
{
	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	names.clear();
	
	for (it = m_2DregionMap.begin() ; it != m_2DregionMap.end() ; it++)
	{
		std::string name = it->first;
		names.push_back(name);
	}
}

bool SimiConductorInstance::is1DSaved() const
{
	if (m_pSimState1D == 0)
		return true;
	if (m_pSimState1D->isModified())
		return false;

	return true;
}

bool SimiConductorInstance::is2DSaved() const
{
	if (m_pSimState2D == 0)
		return true;
	if (m_pSimState2D->isModified())
		return false;
	return true;
}

bool SimiConductorInstance::isSaved() const
{
	if (is1DSaved() && is2DSaved())
		return true;
	return false;
}

void SimiConductorInstance::setupTotal1DRegion(const std::string &regionName, int numX)
{
	internalDelete1DRegion(SIMI_REGIONNAME_ALL);
	internalCreateNew1DRegionName(SIMI_REGIONNAME_ALL);

	SimiConductorLineRegion line(1, numX);

	std::map<std::string, std::list<SimiConductorRegion1D *> >::iterator it;

	it = m_1DregionMap.find(regionName);

	if (it == m_1DregionMap.end())
	{
		std::cerr << "Internal error: reserved region name not found" << std::endl;
		return;
	}

	it->second.push_back(line.createCopy());
}

void SimiConductorInstance::setupTotal2DRegion(const std::string &regionName, int numX, int numY)
{
	internalDelete2DRegion(SIMI_REGIONNAME_ALL);
	internalCreateNew2DRegionName(SIMI_REGIONNAME_ALL);

	SimiConductorRectRegion rect(1, 1, numX, numY);

	std::map<std::string, std::list<SimiConductorRegion2D *> >::iterator it;

	it = m_2DregionMap.find(regionName);

	if (it == m_2DregionMap.end())
	{
		std::cerr << "Internal error: reserved region name not found" << std::endl;
		return;
	}

	it->second.push_back(rect.createCopy());
}

void SimiConductorInstance::setGAParams(GeneticAlgorithmParams *pParams)
{
	if (m_pGAParams)
		delete m_pGAParams;
	m_pGAParams = pParams;
}

void SimiConductorInstance::setSimulationState1D(SimulationState *pState)
{
	if (m_pSimState1D)
		delete m_pSimState1D;
	m_pSimState1D = pState;

	if (m_pSimState1D)
	{
		if (m_pSimState1D->getDimensions() != 1)
			std::cerr << "INTERNAL ERROR: number of dimensions of the simulation state is not equal to 1" << std::endl;

		int numX = m_pSimState1D->getNumberOfXPixels();
		double w = (double)(numX-1)*m_pSimState1D->getPixelWidth();

		defineConstant(SIMI_CONSTNAME_INTWIDTH_1D, numX);
		defineConstant(SIMI_CONSTNAME_WIDTH_1D, w);
		setupTotal1DRegion(SIMI_REGIONNAME_ALL, numX);
	}
	else
	{
		defineConstant(SIMI_CONSTNAME_INTWIDTH_1D, 0);
		defineConstant(SIMI_CONSTNAME_WIDTH_1D, 0);
		internalDelete1DRegion(SIMI_REGIONNAME_ALL);
	}
}

SimulationState *SimiConductorInstance::getSimulationState1D() const
{
	return m_pSimState1D;
}

void SimiConductorInstance::setSimulationState2D(SimulationState *pState)
{
	if (m_pSimState2D)
		delete m_pSimState2D;
	m_pSimState2D = pState;

	if (m_pSimState2D)
	{
		if (m_pSimState2D->getDimensions() != 2)
			std::cerr << "INTERNAL ERROR: number of dimensions of the simulation state is not equal to 2" << std::endl;

		int numX = m_pSimState2D->getNumberOfXPixels();
		int numY = m_pSimState2D->getNumberOfYPixels();
		double w = (double)numX*m_pSimState2D->getPixelWidth();
		double h = (double)(numY-1)*m_pSimState2D->getPixelHeight();

		defineConstant(SIMI_CONSTNAME_INTWIDTH_2D, numX);
		defineConstant(SIMI_CONSTNAME_INTHEIGHT_2D, numY);
		defineConstant(SIMI_CONSTNAME_WIDTH_2D, w);
		defineConstant(SIMI_CONSTNAME_HEIGHT_2D, h);
		setupTotal2DRegion(SIMI_REGIONNAME_ALL, numX, numY);
	}
	else
	{
		defineConstant(SIMI_CONSTNAME_INTWIDTH_2D, 0);
		defineConstant(SIMI_CONSTNAME_INTHEIGHT_2D, 0);
		defineConstant(SIMI_CONSTNAME_WIDTH_2D, 0);
		defineConstant(SIMI_CONSTNAME_HEIGHT_2D, 0);
		internalDelete2DRegion(SIMI_REGIONNAME_ALL);
	}
}

SimulationState *SimiConductorInstance::getSimulationState2D() const
{
	return m_pSimState2D;
}

