#include "openclkernel.h"
#include <stdio.h>
#include <iostream>

OpenCLKernel::OpenCLKernel()
{
	m_init = false;
	m_context = 0;
	m_queue = 0;
	m_program = 0;
	m_pDevices = 0;
}

OpenCLKernel::~OpenCLKernel()
{
	destroy();
}

bool OpenCLKernel::init(int numKernels)
{
	if (numKernels < 1)
	{
		setErrorString("Number of kernels should be at least one");
		return false;
	}

	if (m_init)
	{
		setErrorString("OpenCL kernel loader is already initialized");
		return false;
	}

	m_currentProgram = "";
	m_currentKernelNames.resize(numKernels);
	m_kernel.resize(numKernels);

	for (int i = 0 ; i < numKernels ; i++)
	{
		m_currentKernelNames[i] = "";
		m_kernel[i] = 0;
	}

	cl_platform_id platforms[16];
	cl_uint numPlatforms, numDevices;
	cl_int err = clGetPlatformIDs(16, platforms, &numPlatforms);

	if (err != CL_SUCCESS)
	{
		setErrorString(getCLErrorString(err));
		releaseAll();
		return false;
	}

	if (numPlatforms < 1)
	{
		setErrorString("No OpenCL platforms available");
		releaseAll();
		return false;
	}

	cl_platform_id platform = platforms[0]; // TODO: make this configurable?

	err = clGetDeviceIDs(platform, CL_DEVICE_TYPE_GPU, 0, NULL, &numDevices);
	if (err != CL_SUCCESS)
	{
		setErrorString("Can't find any GPU devices");
		releaseAll();
		return false;
	}

	m_pDevices = new cl_device_id [numDevices];

	clGetDeviceIDs(platform, CL_DEVICE_TYPE_GPU, numDevices, m_pDevices, NULL);

	m_context = clCreateContext(0, 1, m_pDevices, NULL, NULL, &err);
	if (err != CL_SUCCESS)
	{
		setErrorString(std::string("Can't create OpenCL context: ") + getCLErrorString(err));
		releaseAll();
		return false;
	}

	m_deviceIndex = 0; // TODO: make this configurable?

	m_queue = clCreateCommandQueue(m_context, m_pDevices[m_deviceIndex], 0, &err);
	if (err != CL_SUCCESS)
	{
		setErrorString(std::string("Can't create OpenCL command queue: ") + getCLErrorString(err));
		return false;
	}

	m_init = true;

	return true;
}

bool OpenCLKernel::loadProgram(const std::string &programString, std::string &failLog)
{
	if (!m_init)
	{
		setErrorString("OpenCL kernel loader was not initialized");
		return false;
	}

	if (programString.length() == 0)
	{
		setErrorString("No program specified");
		return false;
	}

	if (m_program == 0 || (m_program != 0 && programString != m_currentProgram))
	{
		// compile
		
		if (m_program)
			clReleaseProgram(m_program);
		
		for (int i = 0 ; i < m_kernel.size() ; i++)
		{
			if (m_kernel[i])
			{
				clReleaseKernel(m_kernel[i]);
				m_kernel[i] = 0;
			}
			m_currentKernelNames[i] = "";
		}

		m_program = 0;
		m_currentProgram = "";

		const char *pProgStr = programString.c_str();
		cl_int err;

		m_program = clCreateProgramWithSource(m_context, 1, &pProgStr, 0, &err);
		if (err != CL_SUCCESS)
		{
			setErrorString(std::string("Unable to create OpenCL program from specified source: ") + getCLErrorString(err));
			return false;
		}

		err = clBuildProgram(m_program, 1, &(m_pDevices[m_deviceIndex]), "", 0, 0);
		if (err != CL_SUCCESS)
		{
			size_t logLength;

			clGetProgramBuildInfo(m_program, m_pDevices[m_deviceIndex], CL_PROGRAM_BUILD_LOG, 0, 0, &logLength);

			char *pLog = new char[logLength+1];

			clGetProgramBuildInfo(m_program, m_pDevices[m_deviceIndex], CL_PROGRAM_BUILD_LOG, logLength, pLog, 0);
			pLog[logLength] = 0;

			failLog = std::string(pLog);
			delete [] pLog;

			clReleaseProgram(m_program);
			m_program = 0;

			setErrorString(std::string("Unable to build OpenCL program: ") + getCLErrorString(err));

			return false;
		}

		m_currentProgram = programString;
	}

	return true;
}

bool OpenCLKernel::loadKernel(int idx, const std::string &kernelName)
{
	if (!m_init)
	{
		setErrorString("OpenCL kernel loader was not initialized");
		return false;
	}
	
	if (idx < 0 || idx >= m_kernel.size())
	{
		setErrorString("An invalid index was specified");
		return false;
	}

	if (kernelName.length() == 0)
	{
		setErrorString("No program or no kernel name specified");
		return false;
	}

	if (m_kernel[idx] == 0 || (m_kernel[idx] != 0 && m_currentKernelNames[idx] != kernelName))
	{
		cl_int err;

		if (m_kernel[idx])
			clReleaseKernel(m_kernel[idx]);

		m_currentKernelNames[idx] = "";

		m_kernel[idx] = clCreateKernel(m_program, kernelName.c_str(), &err);
		if (err != CL_SUCCESS)
		{
			setErrorString(std::string("Unable to get OpenCL kernel from program: ") + getCLErrorString(err));
			return false;
		}

		m_currentKernelNames[idx] = kernelName;
	}

	return true;
}

bool OpenCLKernel::destroy()
{
	if (!m_init)
	{
		setErrorString("OpenCL kernel loader was not initialized");
		return false;
	}

	m_init = false;

	m_currentProgram = "";

	releaseAll();

	return true;
}

void OpenCLKernel::releaseAll()
{
	for (int i = 0 ; i < m_kernel.size() ; i++)
	{
		if (m_kernel[i])
			clReleaseKernel(m_kernel[i]);
	}
	m_kernel.resize(0);
	m_currentKernelNames.resize(0);

	if (m_program)
		clReleaseProgram(m_program);
	if (m_queue)
		clReleaseCommandQueue(m_queue);
	if (m_context)
		clReleaseContext(m_context);
	if (m_pDevices)
		delete [] m_pDevices;

	m_context = 0;
	m_queue = 0;
	m_program = 0;
	m_pDevices = 0;
}

std::string OpenCLKernel::getCLErrorString(int errNum)
{
	char str[256];

	sprintf(str, "OpenCL error code %d", errNum);

	return std::string(str);
}
