#include "newtonraphsonsearch.h"
#include "klu.h"

#ifdef SIMICONDUCTOR_HAVE_GSL
	#include <gsl/gsl_matrix.h>
	#include <gsl/gsl_vector.h>
	#include <gsl/gsl_permutation.h>
	#include <gsl/gsl_linalg.h>
#endif // SIMICONDUCTOR_HAVE_GSL

#include <stdio.h>
#include <string.h>
#include <cmath>

#include <iostream>

NewtonRaphsonSearch::NewtonRaphsonSearch()
{
	m_dimension = -1;
}

NewtonRaphsonSearch::~NewtonRaphsonSearch()
{
#ifdef SIMICONDUCTOR_HAVE_GSL
	if (m_dimension > 0 && !m_useSparseSolver)
		gsl_permutation_free((gsl_permutation *)m_pPerm);
#endif // SIMICONDUCTOR_HAVE_GSL
}

bool NewtonRaphsonSearch::init(int dimension, bool useSparseSolver, bool useSparseElementAccess)
{
	if (m_dimension > 0)
	{
		setErrorString("Already initialized");
		return false;
	}

	if (dimension < 1)
	{
		setErrorString("Dimension must be at least one");
		return false;
	}

#ifndef SIMICONDUCTOR_HAVE_GSL
	if (!useSparseSolver)
	{
		setErrorString("Requested use of non-sparse solver, but GSL was not available at compile time");
		return false;
	}
#endif // SIMICONDUCTOR_HAVE_GSL

	if (useSparseSolver)
	{
		int maxCount = 0;
		
		for (int col = 0 ; col < dimension ; col++)
		{
			int nzRows = getMaxNonZeroRows(col);

			if (nzRows < 0)
			{
				setErrorString("Got a negative value for the maximum non zero rows of a column");
				return false;
			}
			maxCount += nzRows;
		}

		m_jacobianElements.resize(maxCount);
		m_nonzeroRows.resize(maxCount);
		m_colPointers.resize(dimension+1);
	}
	else
	{
		m_jacobianElements.resize(dimension*dimension);
#ifdef SIMICONDUCTOR_HAVE_GSL
		m_pPerm = gsl_permutation_alloc(dimension);
#endif // SIMICONDUCTOR_HAVE_GSL
	}

	m_functionValues.resize(dimension);
	m_differences.resize(dimension);

	m_dimension = dimension;
	m_useSparseAccess = useSparseElementAccess;
	m_useSparseSolver = useSparseSolver;

	return true;
}

bool NewtonRaphsonSearch::step(double &sigma)
{
	if (m_dimension < 1)
	{
		setErrorString("Not initialized yet");
		return false;
	}

	double sum = 0;

	for (int i = 0 ; i < m_dimension ; i++)
	{
		double val = - getFunctionValue(i); // negative!
		m_functionValues[i] = val;

		sum += val*val;
	}

	sum /= (double)m_dimension;
	sum = std::sqrt(sum);
	sigma = sum;

	if (m_useSparseSolver)
	{
		if (m_useSparseAccess)
		{
			int count = 0;
			int col;

			for (col = 0 ; col < m_dimension ; col++)
			{
				m_colPointers[col] = count;

				startColumn(col);

				int row = getNextNonZeroRow();

				while (row >= 0)
				{
					double value = getDerivative(row, col);

					if (value != 0)
					{
						m_jacobianElements[count] = value;
						m_nonzeroRows[count] = row;

						count++;
					}

					int newRow = getNextNonZeroRow();

					if (newRow >= 0 && newRow < row)
					{
						setErrorString("A new row index is smaller than the current one");
						return false;
					}
					row = newRow;
				}
			}
			m_colPointers[col] = count;
		}
		else // don't have sparse access, need to check all elements
		{
			int count = 0;
			int col;

			for (col = 0 ; col < m_dimension ; col++)
			{
				m_colPointers[col] = count;

				for (int row = 0 ; row < m_dimension ; row++)
				{
					double value = getDerivative(row, col);

					if (value != 0)
					{
						m_jacobianElements[count] = value;
						m_nonzeroRows[count] = row;

						count++;
					}
				}
			}
			m_colPointers[col] = count;
		}

		//printMatrix();

		m_differences = m_functionValues;

		klu_symbolic *Symbolic;
		klu_numeric *Numeric;
		klu_common Common;

		klu_defaults (&Common);

		Symbolic = klu_analyze(m_dimension, &(m_colPointers[0]), &(m_nonzeroRows[0]), &Common);
		Numeric = klu_factor(&(m_colPointers[0]), &(m_nonzeroRows[0]), &(m_jacobianElements[0]), Symbolic, &Common);
		klu_solve(Symbolic, Numeric, m_dimension, 1, &(m_differences[0]), &Common);
		klu_free_symbolic(&Symbolic, &Common) ;
		klu_free_numeric(&Numeric, &Common) ;
	}
	else // not using the sparse solver, so we have a square matrix representing the jacobian
	{
		if (m_useSparseAccess)
		{
			memset(&(m_jacobianElements[0]), 0, m_jacobianElements.size()*sizeof(double));

			for (int col = 0 ; col < m_dimension ; col++)
			{
				startColumn(col);

				int row = getNextNonZeroRow();
				while (row >= 0)
				{
					double value = getDerivative(row, col);

					m_jacobianElements[col + row*m_dimension] = value;
					row = getNextNonZeroRow();
				}
			}
		}
		else // not using sparse access
		{
			int offset = 0;

			for (int row = 0 ; row < m_dimension ; row++)
				for (int col = 0 ; col < m_dimension ; col++, offset++)
					m_jacobianElements[offset] = getDerivative(row, col);

		}

		//printMatrix();

		// Ok, have the jacobian matrix
#ifdef SIMICONDUCTOR_HAVE_GSL
		gsl_matrix_view jacView = gsl_matrix_view_array(&(m_jacobianElements[0]), m_dimension, m_dimension);
		gsl_vector_view FView = gsl_vector_view_array(&(m_functionValues[0]), m_dimension);
		gsl_vector_view diffView = gsl_vector_view_array(&(m_differences[0]), m_dimension);

		int permSign = 0;
		int status;
		char str[1024];

		status = gsl_linalg_LU_decomp(&(jacView.matrix), (gsl_permutation *)m_pPerm, &permSign);
		if (status != GSL_SUCCESS)
		{
			sprintf(str, "Error in LU decomposition (GSL error %d)", status);
			setErrorString(str);
			return false;
		}
	
		status = gsl_linalg_LU_solve(&(jacView.matrix), (gsl_permutation *)m_pPerm, &(FView.vector), &(diffView.vector));
		if (status != GSL_SUCCESS)
		{
			sprintf(str, "Error solving LU system (GSL error %d)", status);
			setErrorString(str);
			return false;
		}
#else
		setErrorString("GSL was not available at compile time");
		return false;
#endif // SIMICONDUCTOR_HAVE_GSL
	}

	return true;
}

int NewtonRaphsonSearch::getMaxNonZeroRows(int col)
{
	return -1;
}

void NewtonRaphsonSearch::printMatrix() const
{
	if (m_useSparseSolver)
	{
		std::vector<double> tmp(m_dimension*m_dimension);

		memset(&(tmp[0]), 0, sizeof(double)*m_dimension*m_dimension);

		for (int j = 0 ; j < m_dimension ; j++)
		{
			int startPos = m_colPointers[j];
			int endPos = m_colPointers[j+1];

			for (int p = startPos ; p < endPos ; p++)
			{
				int i = m_nonzeroRows[p];
				double val = m_jacobianElements[p];

				int idx = i*m_dimension+j;

				tmp[idx] = val;
			}
		}

		int offset = 0;

		for (int i = 0 ; i < m_dimension ; i++)
		{
			for (int j = 0 ; j < m_dimension ; j++, offset++)
				std::cout << tmp[offset] << "\t";
			std::cout << std::endl;
		}
		std::cout << std::endl;

	}
	else
	{
		int offset = 0;

		for (int i = 0 ; i < m_dimension ; i++)
		{
			for (int j = 0 ; j < m_dimension ; j++, offset++)
				std::cout << m_jacobianElements[offset] << "\t";
			std::cout << std::endl;
		}
		std::cout << std::endl;
	}
}

void NewtonRaphsonSearch::getMatrix(std::vector<double> &m) const
{
	m.resize(m_dimension*m_dimension);

	if (m_useSparseSolver)
	{
		memset(&(m[0]), 0, sizeof(double)*m_dimension*m_dimension);

		for (int j = 0 ; j < m_dimension ; j++)
		{
			int startPos = m_colPointers[j];
			int endPos = m_colPointers[j+1];

			for (int p = startPos ; p < endPos ; p++)
			{
				int i = m_nonzeroRows[p];
				double val = m_jacobianElements[p];

				int idx = i*m_dimension+j;

				m[idx] = val;
			}
		}
	}
	else
	{
		int offset = 0;

		for (int i = 0 ; i < m_dimension ; i++)
			for (int j = 0 ; j < m_dimension ; j++, offset++)
				m[offset] = m_jacobianElements[offset];
	}
}

