#include <mpi.h>
#include "simiconductorconfig.h"
#include "simiconductorinstance.h"
#include "extproggafactory.h"
#include "extprogmogafactory.h"
#include "mainmpi.h"
#include "cmdoptimizempi.h"
#include <serut/memoryserializer.h>
#include <mogal/mpigeneticalgorithm.h>
#include <iostream>

#include "debugnew.h"

std::list<shellp::ShellCommand *> registeredMPICommands;

void RegisterMPICommands()
{
	registeredMPICommands.push_back(new CmdOptimizeMPIGASingle("opt/mpi/ga"));
	registeredMPICommands.push_back(new CmdOptimizeMPIGAMulti("opt/mpi/moga"));
}

void UnregisterMPICommands()
{
	while (!registeredMPICommands.empty())
	{
		delete *(registeredMPICommands.begin());
		registeredMPICommands.pop_front();
	}
}

void RegisterCommands();
void UnregisterCommands();

int main(int argc, char **argv)
{
	int worldSize, myRank;

	MPI_Init(&argc, &argv);
	MPI_Comm_size( MPI_COMM_WORLD, &worldSize);

	if (worldSize <= 1)
	{
		std::cerr << "Need more than one process to be able to work" << std::endl;
		MPI_Abort(MPI_COMM_WORLD, -1);
	}

	MPI_Comm_rank(MPI_COMM_WORLD, &myRank);

	int returnStatus = 0;

	if (myRank == 0)
	{
		std::cout << "Starting MPI session with " << worldSize << " processes" << std::endl;

		SimiConductorInstance instance;

		RegisterCommands();
		RegisterMPICommands();

		returnStatus = instance.run("SimiConductor", "simiconductor> ", argc, argv);

		UnregisterMPICommands();
		UnregisterCommands();

		int command = SIMICONDUCTOR_MPICOMMAND_EXIT;
		MPI_Bcast(&command, 1, MPI_INT, 0, MPI_COMM_WORLD);
	}
	else
	{
		bool done = false;

		while (!done)
		{
			int command = 0;
			MPI_Bcast(&command, 1, MPI_INT, 0, MPI_COMM_WORLD);

			if (command == SIMICONDUCTOR_MPICOMMAND_EXIT)
				done = true;
			else
			{
				if (command == SIMICONDUCTOR_MPICOMMAND_SINGLEOBJECTIVE)
				{
					int paramsSize = 0;
					MPI_Bcast(&paramsSize, 1, MPI_INT, 0, MPI_COMM_WORLD);

					std::vector<uint8_t> paramBuffer(paramsSize);
					MPI_Bcast(&(paramBuffer[0]), paramsSize, MPI_BYTE, 0, MPI_COMM_WORLD);
					serut::MemorySerializer mSer(&(paramBuffer[0]), paramsSize, 0, 0);

					ExtProgGAFactory factory;
					ExtProgGAFactoryParams factoryParams;
					
					if (!factoryParams.read(mSer))
					{
						std::cerr << "Internal error: can't read factory parameters in rank " << myRank << " from MPI rank 0" << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}

					if (!factory.init(&factoryParams))
					{
						std::cerr << "Internal error: unable to initialize factory in rank " << myRank << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}

					mogal::MPIGeneticAlgorithm gaHelper;

					if (!gaHelper.runMPIHelper(factory))
					{
						std::cerr << "Internal error: error running MPI GA helper: " << gaHelper.getErrorString() << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}
				}
				else if (command == SIMICONDUCTOR_MPICOMMAND_MULTIOBJECTIVE)
				{
					int paramsSize = 0;
					MPI_Bcast(&paramsSize, 1, MPI_INT, 0, MPI_COMM_WORLD);
					
					std::vector<uint8_t> paramBuffer(paramsSize);
					MPI_Bcast(&(paramBuffer[0]), paramsSize, MPI_BYTE, 0, MPI_COMM_WORLD);
					serut::MemorySerializer mSer(&(paramBuffer[0]), paramsSize, 0, 0);

					ExtProgMOGAFactory factory;
					ExtProgMOGAFactoryParams factoryParams;
					
					if (!factoryParams.read(mSer))
					{
						std::cerr << "Internal error: can't read factory parameters in rank " << myRank << " from MPI rank 0" << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}

					if (!factory.init(&factoryParams))
					{
						std::cerr << "Internal error: unable to initialize factory in rank " << myRank << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}

					mogal::MPIGeneticAlgorithm gaHelper;

					if (!gaHelper.runMPIHelper(factory))
					{
						std::cerr << "Internal error: error running MPI GA helper: " << gaHelper.getErrorString() << std::endl;
						MPI_Abort(MPI_COMM_WORLD, -1);
					}
				}
				else
				{
					std::cerr << "Received illegal MPI helper command number " << command << std::endl;
					MPI_Abort(MPI_COMM_WORLD, -1);
				}
			}
		}
	}
	
	MPI_Finalize();

	return returnStatus;
}


