#include "dissociationprobability.h"
#include "constants.h"
#include <gsl/gsl_sf.h>

#define DISSOCIATIONPROBABILITY_WORKSPACESIZE 1000

DissociationProbability::DissociationProbability(double electronHoleDistance, double T, double kf, gsl_integration_workspace *pWorkspace)
{
	m_a = electronHoleDistance;
	m_a3 = m_a*m_a*m_a;
	m_kTev = T*CONST_K/CHARGE_ELECTRON;
	m_kTev2 = m_kTev * m_kTev;
	m_kf = kf;

	if (pWorkspace == 0)
	{
		m_pWorkSpace = gsl_integration_workspace_alloc(DISSOCIATIONPROBABILITY_WORKSPACESIZE); // TODO: what's a good value here?
		m_deleteWorkSpace = true;
	}
	else
	{
		m_pWorkSpace = pWorkspace;
		m_deleteWorkSpace = false;
	}
}


DissociationProbability::~DissociationProbability()
{
	if (m_deleteWorkSpace)
		gsl_integration_workspace_free(m_pWorkSpace);
}

double DissociationProbability::calculate(double rf, double epsilonRel, double F)
{
	double b = (CHARGE_ELECTRON/(8.0*CONST_PI*CONST_EPSILON0))*(F/(epsilonRel*m_kTev2));
	double dissociationFactor = 0;
	
	if (b > 1e-4)
		dissociationFactor = gsl_sf_bessel_I1(std::sqrt(8.0*b))/std::sqrt(2.0*b);
	else
	{
		double b2 = b*b;
		double b3 = b2*b;
		double b4 = b3*b;

		dissociationFactor = 1.0+b+b2/3.0+b3/18.0+b4/180.0;
	}

	m_expFactor = (CHARGE_ELECTRON/(4.0*CONST_PI*CONST_EPSILON0))*(1.0/(m_kTev*epsilonRel*m_a));
	m_expPrefactor = (3.0/(4.0*CONST_PI))*(rf/m_a3)*dissociationFactor;
	
	gsl_function function;

	function.function = staticIntegrationFunction;
	function.params = this;

	double result = 0;
	double err = 0;

	gsl_integration_qags(&function, 0, 5, 0, 1e-7, m_pWorkSpace->limit, m_pWorkSpace, &result, &err);

        return result*(4.0/CONST_SQRT_PI);
}

double DissociationProbability::calculateDerivF(double rf, double epsilonRel, double F)
{
	double b = (CHARGE_ELECTRON/(8.0*CONST_PI*CONST_EPSILON0))*(F/(epsilonRel*m_kTev2));
	double dissociationFactor = 0;
	double dissociationFactorDeriv = 0;
	
	if (b > 1e-4)
	{
		double arg = std::sqrt(8.0*b);
		double sqrtb = std::sqrt(2.0*b);

		dissociationFactor = gsl_sf_bessel_I1(arg)/sqrtb;
		dissociationFactorDeriv = 1.0/b*(gsl_sf_bessel_I0(arg)-dissociationFactor);
	}
	else
	{ 
		double b2 = b*b;
		double b3 = b2*b;
		double b4 = b3*b;

		dissociationFactor = 1.0 +b + b2/3.0 + b3/18.0 + b4/180.0;
		dissociationFactorDeriv = 1.0 + 2.0/3.0*b + 1.0/6.0*b2 + 1.0/45.0*b3 + 1.0/540.0*b4;
	}

	m_expFactor = (CHARGE_ELECTRON/(4.0*CONST_PI*CONST_EPSILON0))*(1.0/(m_kTev*epsilonRel*m_a));
	m_expPrefactor = (3.0/(4.0*CONST_PI))*(rf/m_a3)*dissociationFactor;
	
	gsl_function function;

	function.function = staticDerivativeIntegrationFunction;
	function.params = this;

	double result = 0;
	double err = 0;

	gsl_integration_qags(&function, 0, 5, 0, 1e-7, m_pWorkSpace->limit, m_pWorkSpace, &result, &err);

	double factor = (3.0*CHARGE_ELECTRON)/(CONST_PI*CONST_PI*CONST_SQRT_PI*8.0*CONST_EPSILON0)
		      * rf*dissociationFactorDeriv/(m_kf*m_a3*epsilonRel*m_kTev2);

        return result*factor;
}

