#include "cmdsimulation2dnew.h"
#include "cmdsimulation1dnew.h"
#include "simiconductorconfig.h"
#include "simiconductorinstance.h"
#include "simulation2ddouble.h"
#include "simulation2ddoublerel.h"
#ifdef SIMICONDUCTOR_CONFIG_OPENCL
	#include "simulation2ddoublegpurel.h"
#endif // SIMICONDUCTOR_CONFIG_OPENCL
#include "simulation2dnr2.h"
#include "simulationstate.h"
#include "binimage.h"
#include "timeutil.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/shellcmdboolarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdchoicearg.h>
#include <shellp/iosystem.h>
#include <serut/fileserializer.h>
#include <stdio.h>
#include <cmath>
#include <algorithm>

#include <iostream>

using namespace shellp;

#ifdef WIN32
#define ISFINITE _finite
#define MIN(a,b) (((a)<(b))?(a):(b))
#else
#define ISFINITE std::isfinite
#define MIN(a,b) std::min(a,b)
#endif // WIN32

#define PLOTWINID_N					0
#define PLOTWINID_P					1
#define PLOTWINID_BG					2
#define PLOTWINID_G					3
#define PLOTWINID_RF					4
#define PLOTWINID_DN					5
#define PLOTWINID_DP					6
#define PLOTWINID_NMOB					7
#define PLOTWINID_PMOB					8
#define PLOTWINID_EPS					9
#define PLOTWINID_V					10
#define PLOTWINID_R					11
#define PLOTWINID_JX					12
#define PLOTWINID_JY					13
#define PLOTWINID_JNX					14
#define PLOTWINID_JNY					15
#define PLOTWINID_JPX					16
#define PLOTWINID_JPY					17
#define PLOTWINID_VN					18
#define PLOTWINID_VP					19
#define PLOTWINID_MAX					20

#define PLOTWINNAME_N					"n"
#define PLOTWINNAME_P					"p"
#define PLOTWINNAME_BG					"bg"
#define PLOTWINNAME_G					"g"
#define PLOTWINNAME_RF					"rf"
#define PLOTWINNAME_DN					"dn"
#define PLOTWINNAME_DP					"dp"
#define PLOTWINNAME_NMOB				"nmob"
#define PLOTWINNAME_PMOB				"pmob"
#define PLOTWINNAME_EPS					"eps"
#define PLOTWINNAME_V					"V"
#define PLOTWINNAME_R					"r"
#define PLOTWINNAME_JX					"jx"
#define PLOTWINNAME_JY					"jy"
#define PLOTWINNAME_JNX					"jnx"
#define PLOTWINNAME_JNY					"jny"
#define PLOTWINNAME_JPX					"jpx"
#define PLOTWINNAME_JPY					"jpy"
#define PLOTWINNAME_VN					"Vn"
#define PLOTWINNAME_VP					"Vp"

bool getPlotInfo(int plotID, SimulationState *pSim, std::vector<double> &dst, int &numX, int &numY,
		                       double &xOffset, double &yOffset, std::string &title, std::string &xLabel,
				       std::string &yLabel, std::string &zLabel)
{
	std::vector<double> tmp2, tmp3;
	int numTotal = pSim->getNumberOfXPixels()*pSim->getNumberOfYPixels();
	int subY = 0;
	bool status = true;

	switch(plotID)
	{
	case PLOTWINID_N:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_N, dst);	
		title = PLOTWINNAME_N;
		break;
	case PLOTWINID_P:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_P, dst);	
		title = PLOTWINNAME_P;
		break;
	case PLOTWINID_BG:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_BG, dst);	
		title = PLOTWINNAME_BG;
		break;
	case PLOTWINID_G:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_G, dst);	
		title = PLOTWINNAME_G;
		break;
	case PLOTWINID_RF:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_RF, dst);	
		title = PLOTWINNAME_RF;
		break;
	case PLOTWINID_DN:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_DN, dst);	
		title = PLOTWINNAME_DN;
		break;
	case PLOTWINID_DP:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_DP, dst);	
		title = PLOTWINNAME_DP;
		break;
	case PLOTWINID_NMOB:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_NMOB, dst);
		title = PLOTWINNAME_NMOB;
		break;
	case PLOTWINID_PMOB:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_PMOB, dst);
		title = PLOTWINNAME_PMOB;
		break;
	case PLOTWINID_EPS:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_EPSREL, dst);
		title = PLOTWINNAME_EPS;
		break;
	case PLOTWINID_V:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_V, dst);
		title = PLOTWINNAME_V;
		break;
	case PLOTWINID_R:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_R, dst);
		title = PLOTWINNAME_R;
		break;
	case PLOTWINID_JX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPX, dst);
			status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_JNX, tmp2);

			for (int i = 0 ; i < numTotal ; i++)
			{
				dst[i] -= tmp2[i];
				dst[i] *= CHARGE_ELECTRON;
			}

			subY = 1;
			title = PLOTWINNAME_JX;
		}
		break;
	case PLOTWINID_JY:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPY, dst);
			status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_JNY, tmp2);

			for (int i = 0 ; i < numTotal ; i++)
			{
				dst[i] -= tmp2[i];
				dst[i] *= CHARGE_ELECTRON;
			}

			subY = 1;
			title = PLOTWINNAME_JY;
		}
		break;
	case PLOTWINID_JNX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JNX, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= -CHARGE_ELECTRON;

			subY = 1;
			title = PLOTWINNAME_JNX;
		}
		break;
	case PLOTWINID_JNY:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JNY, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= -CHARGE_ELECTRON;

			subY = 1;
			title = PLOTWINNAME_JNY;
		}
		break;
	case PLOTWINID_JPX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPX, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= CHARGE_ELECTRON;

			subY = 1;
			title = PLOTWINNAME_JPX;
		}
		break;
	case PLOTWINID_JPY:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPY, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= CHARGE_ELECTRON;

			subY = 1;
			title = PLOTWINNAME_JPY;
		}
		break;
	case PLOTWINID_VN:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, dst);
		title = PLOTWINNAME_VN;
		break;
	case PLOTWINID_VP:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, dst);
		title = PLOTWINNAME_VP;
		break;
	default:
		return false;
	}

	if (!status)
		return false;

	double pixelWidth = pSim->getPixelWidth();

	xOffset = 0;
	yOffset = pixelWidth/2.0*(double)subY;

	numX = pSim->getNumberOfXPixels();
	numY = pSim->getNumberOfYPixels() - subY;

	xLabel = "X";
	yLabel = "Y";
	zLabel = "Z";

	return true;
}

CmdSimulation2DNEWClear::CmdSimulation2DNEWClear(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag force clearing the simulation even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Clear the current simulation entirely.");
}

CmdSimulation2DNEWClear::~CmdSimulation2DNEWClear()
{
	delete m_pForceArg;
}

bool CmdSimulation2DNEWClear::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is2DSaved())) )
		{
			setErrorString("Simulation not saved, specify force flag to override");
			return false;
		}
	}

	pInst->setSimulationState2D(0);
	//pInst->set2DSavedFlag(true);

	return true;
}

CmdSimulation2DNEWNew::CmdSimulation2DNEWNew(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNumXArg = new ShellCmdIntArg("numx", "32");
	m_pNumXArg->setMin(3);
	m_pNumXArg->setDescription("Number of pixels in X direction.");
	addArgument(m_pNumXArg);
	
	m_pNumYArg = new ShellCmdIntArg("numy", "32");
	m_pNumYArg->setMin(3);
	m_pNumYArg->setDescription("Number of pixels in Y direction.");
	addArgument(m_pNumYArg);

	m_pWidthArg = new ShellCmdRealArg("width");
	m_pWidthArg->setMin(0); // can't be negative
	m_pWidthArg->setDescription("Actual physical size in X direction.");
	addArgument(m_pWidthArg);

	m_pHeightArg = new ShellCmdRealArg("height");
	m_pHeightArg->setMin(0); // can't be negative
	m_pHeightArg->setDescription("Actual physical size in Y direction.");
	addArgument(m_pHeightArg);

	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag forces a new grid even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Create a new 2D simulation state.");
}

CmdSimulation2DNEWNew::~CmdSimulation2DNEWNew()
{
	delete m_pNumXArg;
	delete m_pNumYArg;
	delete m_pWidthArg;
	delete m_pHeightArg;
	delete m_pForceArg;
}

bool CmdSimulation2DNEWNew::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is2DSaved())) )
		{
			setErrorString("Simulation state not saved, specify force flag to override");
			return false;
		}
	}

	int numX = m_pNumXArg->getValue();
	int numY = m_pNumYArg->getValue();
	double width = m_pWidthArg->getValue();
	double height = m_pHeightArg->getValue();
	double pixelWidth = width/(double)(numX); // periodic boundary conditions!
	double pixelHeight = height/(double)(numY-1);

	SimulationState *pNewState = new SimulationState();

	if (!pNewState->init(numX, numY, pixelWidth, pixelHeight))
	{
		setErrorString("Unable to create new simulation state: " + pNewState->getErrorString());
		delete pNewState;
		return false;
	}

	pInst->setSimulationState2D(pNewState);
	//pInst->set2DSavedFlag(true);

	pIOSys->print("Created a new grid with %dx%d pixels", numX, numY);
	return true;
}

CmdSimulation2DNEWSave::CmdSimulation2DNEWSave(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to write the simulation to.");
	addArgument(m_pFileNameArg);

	setDescription("Save current simulation settings.");
}

CmdSimulation2DNEWSave::~CmdSimulation2DNEWSave()
{
	delete m_pFileNameArg;
}

bool CmdSimulation2DNEWSave::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::string fileName = m_pFileNameArg->getValue();

	if (!pSim->save(fileName))
	{
		setErrorString(pSim->getErrorString());
		return false;
	}

	pSim->setModified(false);

	pIOSys->writeOutputLine("Simulation saved.");

	//pInst->set2DSavedFlag(true);

	return true;
}

CmdSimulation2DNEWLoad::CmdSimulation2DNEWLoad(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to read the simulation from.");
	addArgument(m_pFileNameArg);

	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag forces loading a grid even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Load 2D simulation state.");
}

CmdSimulation2DNEWLoad::~CmdSimulation2DNEWLoad()
{
	delete m_pFileNameArg;
	delete m_pForceArg;
}

bool CmdSimulation2DNEWLoad::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is2DSaved())) )
		{
			setErrorString("Simulation not saved, specify force flag to override");
			return false;
		}
	}

	std::string fileName = m_pFileNameArg->getValue();
	std::string errStr, simType;

	SimulationState *pSim = new SimulationState();
	if (!pSim->load(fileName))
	{
		setErrorString("Couldn't load specified simulation state: " + pSim->getErrorString());
		delete pSim;
		return false;
	}

	if (pSim->getDimensions() != 2)
	{
		setErrorString("Specified simulation state file is not a two-dimensional one");
		delete pSim;
		return false;
	}

	pInst->setSimulationState2D(pSim);
	//pInst->set2DSavedFlag(true);
	pSim->setModified(false);

	int numX = pSim->getNumberOfXPixels();
	int numY = pSim->getNumberOfYPixels();

	pIOSys->print("Loaded simulation state on a new grid with dimensions %dx%d", numX, numY);

	return true;
}

CmdSimulation2DNEWImport::CmdSimulation2DNEWImport(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file from which simulation data should be imported into the current simulation.");
	addArgument(m_pFileNameArg);

	m_pVarOnlyArg = new ShellCmdBoolArg("varonly");
	m_pVarOnlyArg->setDefault("false");
	m_pVarOnlyArg->setDescription("If set, only the properties that can change during the simulation will be imported: electron density, hole density and potential.");
	addArgument(m_pVarOnlyArg);

	m_pExcludeBorderArg = new ShellCmdBoolArg("excludeborder");
	m_pExcludeBorderArg->setDefault("false");
	m_pExcludeBorderArg->setDescription("If set, the top and bottom pixels in the simulation will not be replaced.");
	addArgument(m_pExcludeBorderArg);

	setDescription("Import data from a specified simulation save file into the current simulation. Grid sizes do not need to match, values will be interpolated if needed.");
}

CmdSimulation2DNEWImport::~CmdSimulation2DNEWImport()
{
	delete m_pFileNameArg;
	delete m_pVarOnlyArg;
	delete m_pExcludeBorderArg;
}

bool CmdSimulation2DNEWImport::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::string errStr, simType;
	std::string fileName = m_pFileNameArg->getValue();
	bool varOnly = m_pVarOnlyArg->getValue();
	bool excludeBorder = m_pExcludeBorderArg->getValue();

	SimulationState importState;

	if (!importState.load(fileName))
	{
		setErrorString("Couldn't load state to be imported");
		return false;
	}

	int numX = pSim->getNumberOfXPixels();
	int numY = pSim->getNumberOfYPixels();
	std::vector<bool> excludeMap;
	bool varFlags[SIMSTATE_GRIDPROP_MAX];
	bool doubleFlags[SIMSTATE_PROP_MAX];

	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		doubleFlags[i] = true;

	if (!varOnly)
	{
		for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
			varFlags[i] = true;
	}
	else
	{
		for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
			varFlags[i] = false;

		varFlags[SIMSTATE_GRIDPROP_N] = true;
		varFlags[SIMSTATE_GRIDPROP_P] = true;
		varFlags[SIMSTATE_GRIDPROP_V] = true;
	}


	if (excludeBorder)
	{
		excludeMap.resize(numX*numY);

		for (int i = 0 ; i < excludeMap.size() ; i++)
			excludeMap[i] = false;

		for (int x = 0 ; x < numX ; x++)
		{
			excludeMap[x] = true;
			excludeMap[x+(numY-1)*numX] = true;
		}
	}

	if (!pSim->import(importState, &(varFlags[0]), &(doubleFlags[0]), excludeMap))
	{
		setErrorString("Could not import: " + pSim->getErrorString());
		return false;
	}

	return true;
}


CmdSimulation2DNEWSetGridProperty::CmdSimulation2DNEWSetGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pX0 = new ShellCmdIntArg("x1");
	m_pX0->setMin(1);
	m_pX0->setDescription("Left pixel coordinate of the region.");
	addArgument(m_pX0);
	m_pY0 = new ShellCmdIntArg("y1");
	m_pY0->setMin(1);
	m_pY0->setDescription("Bottom pixel coordinate of the region.");
	addArgument(m_pY0);
	m_pX1 = new ShellCmdIntArg("x2");
	m_pX1->setMin(1);
	m_pX1->setDescription("Right pixel coordinate of the region.");
	addArgument(m_pX1);
	m_pY1 = new ShellCmdIntArg("y2");
	m_pY1->setMin(1);
	m_pY1->setDescription("Top pixel coordinate of the region.");
	addArgument(m_pY1);

	m_pValueArg = new ShellCmdRealArg("value");
	m_pValueArg->setDescription("Value to be set in the specified region.");
	addArgument(m_pValueArg);

	setDescription("Set the value of a property of the simulation state in a specific region.");
}

CmdSimulation2DNEWSetGridProperty::~CmdSimulation2DNEWSetGridProperty()
{
	delete m_pPropertyArg;
	delete m_pX0;
	delete m_pY0;
	delete m_pX1;
	delete m_pY1;
	delete m_pValueArg;
}

bool CmdSimulation2DNEWSetGridProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	int x0 = m_pX0->getValue();
	int y0 = m_pY0->getValue();
	int x1 = m_pX1->getValue();
	int y1 = m_pY1->getValue();
	int width = pSim->getNumberOfXPixels();
	int height = pSim->getNumberOfYPixels();

	if (x0 > x1 || x1 > width || y0 > y1 || y1 > height)
	{
		setErrorString("Invalid region specified");
		return false;
	}

	x0--;
	y0--;

	int propID = m_pPropertyArg->getValue();
	double value = m_pValueArg->getValue();

	for (int y = y0 ; y < y1 ; y++)
	{
		for (int x = x0 ; x < x1 ; x++)
		{
			if (!pSim->setGridProperty(propID, x, y, value))
			{
				setErrorString("Unable to set the specified grid property: " + pSim->getErrorString());
				return false;
			}
		}
	}

	return true;
}

CmdSimulation2DNEWFilePlotGridProperty::CmdSimulation2DNEWFilePlotGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);
	m_pAppendArg = new ShellCmdBoolArg("append");
	m_pAppendArg->setDescription("Flag indicating if the plot data should be appended to an existing file.");
	addArgument(m_pAppendArg);

	setDescription("Write all data of the current simulation to a file which can be plotted by gnuplot. The columns are: " + SimulationState::getPlotFileColumns(2));
}

CmdSimulation2DNEWFilePlotGridProperty::~CmdSimulation2DNEWFilePlotGridProperty()
{
	delete m_pFileNameArg;
	delete m_pAppendArg;
}

bool CmdSimulation2DNEWFilePlotGridProperty::execute()
{	
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	std::string fileName = m_pFileNameArg->getValue();
	bool append = m_pAppendArg->getValue();
	FILE *pFile = 0;

	if (append)
		pFile = fopen(fileName.c_str(), "at");
	else
		pFile = fopen(fileName.c_str(), "wt");

	if (pFile == 0)
	{
		setErrorString("Unable to write to the specified file");
		return false;
	}

	pSim->writePlotData(pFile);

	fclose(pFile);
	return true;
}

CmdSimulation2DNEWPlotGridProperty::CmdSimulation2DNEWPlotGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to plot.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", PLOTWINID_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", PLOTWINID_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", PLOTWINID_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", PLOTWINID_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", PLOTWINID_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", PLOTWINID_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", PLOTWINID_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", PLOTWINID_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", PLOTWINID_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", PLOTWINID_EPS);
	m_pPropertyArg->registerChoice(PLOTWINNAME_R, "recombination rate", PLOTWINID_R);
	m_pPropertyArg->registerChoice(PLOTWINNAME_V, "potential", PLOTWINID_V);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JX, "current in x-direction", PLOTWINID_JX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JY, "current in y-direction", PLOTWINID_JY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNX, "electron current in x-direction", PLOTWINID_JNX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNY, "electron current in y-direction", PLOTWINID_JNY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPX, "hole current in x-direction", PLOTWINID_JPX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPY, "hole current in y-direction", PLOTWINID_JPY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", PLOTWINID_VN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", PLOTWINID_VP);
	addArgument(m_pPropertyArg);

	setDescription("Plots a property of the simulation.");
}

CmdSimulation2DNEWPlotGridProperty::~CmdSimulation2DNEWPlotGridProperty()
{
	delete m_pPropertyArg;
}

bool CmdSimulation2DNEWPlotGridProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (!pIOSys->supports3DPlot())
	{
		setErrorString("Interactive 3D plotting is not supported by the current IO system");
		return false;
	}

	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int id = m_pPropertyArg->getValue();
	int width;
	int height;
	double pixelWidth = pSim->getPixelWidth();
	double pixelHeight = pSim->getPixelHeight();

	double xOffset;
	double yOffset;
	std::string title, xLabel, yLabel, zLabel;

	std::vector<double> dst;

	if (!getPlotInfo(id, pSim, dst, width, height, xOffset, yOffset, title, xLabel, yLabel, zLabel))
	{
		setErrorString("Couldn't get plot data, perhaps the specified property has not been set yet");
		return false;
	}

	if (!pIOSys->plot3D(id, title, xLabel, yLabel, zLabel, xOffset, yOffset, pixelWidth, pixelHeight, 1, 1, 1, dst, width, height))
	{
		setErrorString(std::string("Unable to plot desired property: ") + pIOSys->getErrorString());
		return false;
	}

	return true;
}

CmdSimulation2DNEWSetRegionProperty::CmdSimulation2DNEWSetRegionProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pRegionArg = new ShellCmdStringArg("regionname");
	m_pRegionArg->setDescription("Name of the region in which this value should be set.");
	addArgument(m_pRegionArg);

	m_pValueArg = new ShellCmdRealArg("value");
	m_pValueArg->setDescription("Value to be set in the specified region.");
	addArgument(m_pValueArg);

	setDescription("Set the value of a property of the simulation in a specific named region.");
}

CmdSimulation2DNEWSetRegionProperty::~CmdSimulation2DNEWSetRegionProperty()
{
	delete m_pPropertyArg;
	delete m_pRegionArg;
	delete m_pValueArg;
}

bool CmdSimulation2DNEWSetRegionProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	std::string regionName = m_pRegionArg->getValue();
	std::vector<const SimiConductorRegion2D *> regionParts;

	if (!pInst->get2DRegions(regionName, regionParts))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	double value = m_pValueArg->getValue();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int numX = pSim->getNumberOfXPixels();
	int numY = pSim->getNumberOfYPixels();
	int numTotal = numX*numY;

	std::vector<bool> regionPixels(numTotal);

	for (int i = 0 ; i < numTotal ; i++)
		regionPixels[i] = false;

	for (int i = 0 ; i < regionParts.size() ; i++)
	{
		if (!regionParts[i]->fillPixels(regionPixels, numX, numY))
		{
			setErrorString(std::string("One of the areas of the region produced an error: ") + regionParts[i]->getErrorString());
			return false;
		}
	}

	int propID = m_pPropertyArg->getValue();

	for (int y = 0 ; y < numY ; y++)
	{
		for (int x = 0 ; x < numX ; x++)
		{
			if (regionPixels[x+y*numX])
			{
				if (!pSim->setGridProperty(propID, x, y, value))
				{
					setErrorString("Unable to set the specified grid property: " + pSim->getErrorString());
					return false;
				}
			}
		}
	}
	
	return true;
}

CmdSimulation2DNEWSetRegionPropertyFormula::CmdSimulation2DNEWSetRegionPropertyFormula(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pRegionArg = new ShellCmdStringArg("regionname");
	m_pRegionArg->setDescription("Name of the region in which this value should be set.");
	addArgument(m_pRegionArg);

	m_pFormulaArg = new ShellCmdStringArg("formula");
	m_pFormulaArg->setDescription("The formula to apply to the pixels in the region. You can use 'X' to specify the physical coordinate of a pixel (which begins at 0), and "
			              "'PX' to specify a pixel value (which begins at 1)");
	addArgument(m_pFormulaArg);
	
	setDescription("Set the value of a property of the simulation in a specific named region according to a specific formula.");
}

CmdSimulation2DNEWSetRegionPropertyFormula::~CmdSimulation2DNEWSetRegionPropertyFormula()
{
	delete m_pPropertyArg;
	delete m_pRegionArg;
	delete m_pFormulaArg;
}

bool CmdSimulation2DNEWSetRegionPropertyFormula::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	std::string regionName = m_pRegionArg->getValue();
	std::string formula = m_pFormulaArg->getValue();
	std::vector<const SimiConductorRegion2D *> regionParts;

	if (!pInst->get2DRegions(regionName, regionParts))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int numX = pSim->getNumberOfXPixels();
	int numY = pSim->getNumberOfYPixels();
	double pixelWidth = pSim->getPixelWidth();
	double pixelHeight = pSim->getPixelHeight();
	int numTotal = numX*numY;

	std::vector<bool> regionPixels(numTotal);

	for (int i = 0 ; i < numTotal ; i++)
		regionPixels[i] = false;

	for (int i = 0 ; i < regionParts.size() ; i++)
	{
		if (!regionParts[i]->fillPixels(regionPixels, numX, numY))
		{
			setErrorString(std::string("One of the areas of the region produced an error: ") + regionParts[i]->getErrorString());
			return false;
		}
	}

	int propID = m_pPropertyArg->getValue();

	for (int y = 0 ; y < numY ; y++)
	{
		for (int x = 0 ; x < numX ; x++)
		{
			if (regionPixels[x+y*numX])
			{
				double X = (double)x*pixelWidth;
				int PX = x + 1;
				double Y = (double)y*pixelHeight;
				int PY = y + 1;
				double value = 0;
				
				pInst->defineConstant(SIMI_CONSTNAME_X, X);
				pInst->defineConstant(SIMI_CONSTNAME_PX, PX);
				pInst->defineConstant(SIMI_CONSTNAME_Y, Y);
				pInst->defineConstant(SIMI_CONSTNAME_PY, PY);
				
				if (!pInst->evaluateExpression(formula, value))
				{
					setErrorString("Could not evaluate the specified formula: " + pInst->getErrorString());
					return false;
				}
				if (!pSim->setGridProperty(propID, x, y, value))
				{
					setErrorString("Unable to set the specified grid property: " + pSim->getErrorString());
					return false;
				}
			}
		}
	}

	return true;
}

CmdSimulation2DNEWSetRegionPropertyBIN::CmdSimulation2DNEWSetRegionPropertyBIN(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pRegionArg = new ShellCmdStringArg("regionname");
	m_pRegionArg->setDescription("Name of the region in which this value should be set.");
	addArgument(m_pRegionArg);

	m_pBINFileArg = new ShellCmdStringArg("binfile");
	m_pBINFileArg->setDescription("File name of the BIN file (or image file to import from) containing the values to be used.");
	addArgument(m_pBINFileArg);

	m_pFormulaArg = new ShellCmdStringArg("formula");
	m_pFormulaArg->setDescription("The formula to apply to the pixels in the region. You can use 'X' to specify the physical coordinate of a pixel (which begins at 0), and "
			              "'PX' to specify a pixel value (which begins at 1). The raw values in the BIN file can be referred to by 'V', the values rescaled to an interval [0,1] can "
				      "be specified using 'NV'.");
	m_pFormulaArg->setDefault(SIMI_CONSTNAME_V);
	addArgument(m_pFormulaArg);
	
	setDescription("Set the value of a property of the simulation in a specific named region according to a specific formula, using the values from a BIN file.");
}

CmdSimulation2DNEWSetRegionPropertyBIN::~CmdSimulation2DNEWSetRegionPropertyBIN()
{
	delete m_pPropertyArg;
	delete m_pRegionArg;
	delete m_pBINFileArg;
	delete m_pFormulaArg;
}

bool CmdSimulation2DNEWSetRegionPropertyBIN::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();
	std::string regionName = m_pRegionArg->getValue();
	std::string formula = m_pFormulaArg->getValue();
	std::vector<const SimiConductorRegion2D *> regionParts;

	if (!pInst->get2DRegions(regionName, regionParts))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	BinImage binImg;
	std::string fileName = m_pBINFileArg->getValue();
	uint16_t minValue, maxValue;

	if (!binImg.load(fileName))
	{
		if (!binImg.importImage(fileName))
		{
			setErrorString("Unable to load the specified BIN file or image file '" + fileName + "'");
			return false;
		}
	}

	binImg.getMinMax(minValue, maxValue);
	if (minValue == maxValue)
	{
		if (maxValue < 65535)
			maxValue++;
		else
			minValue--;
	}

	int numX = pSim->getNumberOfXPixels();
	int numY = pSim->getNumberOfYPixels();
	double pixelWidth = pSim->getPixelWidth();
	double pixelHeight = pSim->getPixelHeight();
	int numTotal = numX*numY;

	if (!(numX == binImg.getWidth() && numY == binImg.getHeight()))
	{
		pIOSys->writeOutputLine("Dimensions of the BIN file do not match the dimensions of the simulation, rescaling");
		if (!binImg.resize(numX, numY))
		{
			setErrorString("Error rescaling BIN image: " + binImg.getErrorString());
			return false;
		}
	}

	std::vector<bool> regionPixels(numTotal);

	for (int i = 0 ; i < numTotal ; i++)
		regionPixels[i] = false;

	for (int i = 0 ; i < regionParts.size() ; i++)
	{
		if (!regionParts[i]->fillPixels(regionPixels, numX, numY))
		{
			setErrorString(std::string("One of the areas of the region produced an error: ") + regionParts[i]->getErrorString());
			return false;
		}
	}

	int propID = m_pPropertyArg->getValue();

	for (int y = 0 ; y < numY ; y++) 
	{ 
		for (int x = 0 ; x < numX ; x++) 
		{
			if (regionPixels[x+y*numX]) 
			{ 
				double X = (double)x*pixelWidth; 
				int PX = x + 1; 
				double Y = (double)y*pixelHeight; 
				int PY = y + 1; 
				double rawValue = binImg.getPixel(x, y); 
				double normValue = (rawValue-(double)minValue)/(double)(maxValue-minValue); 
				double value = 0; 
				
				pInst->defineConstant(SIMI_CONSTNAME_X, X); 
				pInst->defineConstant(SIMI_CONSTNAME_PX, PX); 
				pInst->defineConstant(SIMI_CONSTNAME_Y, Y); 
				pInst->defineConstant(SIMI_CONSTNAME_PY, PY); 
				pInst->defineConstant(SIMI_CONSTNAME_V, rawValue); 
				pInst->defineConstant(SIMI_CONSTNAME_NV, normValue); 
				
				if (!pInst->evaluateExpression(formula, value)) 
				{
					setErrorString("Could not evaluate the specified formula: " + pInst->getErrorString()); 
					return false; 
				} 

				if (!pSim->setGridProperty(propID, x, y, value))
				{
					setErrorString("Unable to set the specified grid property: " + pSim->getErrorString());
					return false;
				}
			} 
		} 
	}

	return true;
}


CmdSimulation2DNEWSetPotentialDifference::CmdSimulation2DNEWSetPotentialDifference(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPhiArg = new ShellCmdRealArg("potentialdifference");
	m_pPhiArg->setDescription("Difference in potential between the top row and bottom row in the simulation.");
	addArgument(m_pPhiArg);

	m_pInitGridArg = new ShellCmdBoolArg("initgrid");
	m_pInitGridArg->setDefault("no");
	m_pInitGridArg->setDescription("If this flag is set, not only the potential at the edges will be set, but also the potential on other grid points, using a linear interpolation.");
	addArgument(m_pInitGridArg);

	setDescription("Sets the potential difference to be used in the simulation. Note that this includes the possible internal potential difference of the simulated device.");
}

CmdSimulation2DNEWSetPotentialDifference::~CmdSimulation2DNEWSetPotentialDifference()
{
	delete m_pPhiArg;
	delete m_pInitGridArg;
}

bool CmdSimulation2DNEWSetPotentialDifference::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double V = m_pPhiArg->getValue();
	
	if (!pSim->setDoubleProperty(SIMSTATE_PROP_VDIFF, V))
	{
		setErrorString("Unable to set the potential difference: " + pSim->getErrorString());
		return false;
	}

	int w = pSim->getNumberOfXPixels();
	int h = pSim->getNumberOfYPixels();

	if (m_pInitGridArg->getValue())
	{
		for (int y = 0 ; y < h ; y++)
		{
			double v = ((double)y/(double)(h-1))*V;
			
			for (int x = 0 ; x < w ; x++)
			{
				if (!pSim->setGridProperty(SIMSTATE_GRIDPROP_V, x, y, v))
				{
					setErrorString("Unable to initialize all electrostatic potential values: " + pSim->getErrorString());
					return false;
				}
			}
		}
	}

	for (int x = 0 ; x < w ; x++)
	{
		if (!pSim->setGridProperty(SIMSTATE_GRIDPROP_V, x, 0, 0) ||
		    !pSim->setGridProperty(SIMSTATE_GRIDPROP_V, x, h-1, V) )
		{
			setErrorString("Unable to initialize potential at border: " + pSim->getErrorString());
			return false;
		}
	}

	return true;
}

CmdSimulation2DNEWShowPotentialDifference::CmdSimulation2DNEWShowPotentialDifference(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Shows the potential difference set in the current simulation (Vtop-Vbottom).");
}

CmdSimulation2DNEWShowPotentialDifference::~CmdSimulation2DNEWShowPotentialDifference()
{
}

bool CmdSimulation2DNEWShowPotentialDifference::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	double Vdiff = 0;
	
	if (!pSim->getDoubleProperty(SIMSTATE_PROP_VDIFF, Vdiff))
	{
		setErrorString("Potential difference has not been set yet");
		return false;
	}

	pIOSys->print("%g V", Vdiff);

	return true;
}

CmdSimulation2DNEWGetPixelProperty::CmdSimulation2DNEWGetPixelProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to retrieve.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pXArg = new ShellCmdIntArg("x");
	m_pXArg->setDescription("X coordinate of the simulation pixel.");
	m_pXArg->setMin(1);
	addArgument(m_pXArg);

	m_pYArg = new ShellCmdIntArg("y");
	m_pYArg->setDescription("Y coordinate of the simulation pixel.");
	m_pYArg->setMin(1);
	addArgument(m_pYArg);

	m_pVarName = new ShellCmdStringArg("variable");
	m_pVarName->setDefault("*");
	m_pVarName->setDescription("If specified, the resulting pixel value will be stored in this variable.");
	addArgument(m_pVarName);

	setDescription("Retrieves the value of a simulation property at a specific pixel coordinate, and optionally stores the result in a variable.");
}

CmdSimulation2DNEWGetPixelProperty::~CmdSimulation2DNEWGetPixelProperty()
{
	delete m_pPropertyArg;
	delete m_pXArg;
	delete m_pYArg;
	delete m_pVarName;
}

bool CmdSimulation2DNEWGetPixelProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int xPos = m_pXArg->getValue()-1;
	int yPos = m_pYArg->getValue()-1;
	int width = pSim->getNumberOfXPixels();
	int height = pSim->getNumberOfYPixels();

	if (xPos < 0 || xPos >= width || yPos < 0 || yPos >= height)
	{
		setErrorString("Invalid pixel coordinate");
		return false;
	}

	int propID = m_pPropertyArg->getValue();

	if (!pSim->isGridPropertySet(propID))
	{
		setErrorString("Specified grid property has not been set");
		return false;
	}

	double propValue = pSim->getGridProperty(propID, xPos, yPos);
	std::string varName = m_pVarName->getValue();

	if (varName != std::string("*"))
	{
		if (!pInst->storeVariable(varName, propValue))
		{
			setErrorString(std::string("Unable to store result in variable: ") + pInst->getErrorString());
			return false;
		}
		
		pIOSys->print("Result: %s = %g", m_pVarName->getValue().c_str(), propValue);
	}
	else		
		pIOSys->print("Result: %g", propValue);

	return true;
}

CmdSimulation2DNEWInitDens::CmdSimulation2DNEWInitDens(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->setDescription("Specifies the property to initialize based on the boundary conditions.");
	addArgument(m_pPropertyArg);

	m_pLogArg = new ShellCmdBoolArg("logarithm");
	m_pLogArg->setDefault("yes");
	m_pLogArg->setDescription("If not set, the number densities themselves will be used, otherwise the logarithm of the number densities will be used in the linear interpolation.");
	addArgument(m_pLogArg);

	m_pUseVExtraArg = new ShellCmdBoolArg("useVextra");
	m_pUseVExtraArg->setDefault("yes");
	m_pUseVExtraArg->setDescription("Use the Vextra potentials to account for concentration differences in N and P. Only works if the logarithm is being used.");
	addArgument(m_pUseVExtraArg);

	setDescription("Initialize one of the densities based on the values set at the boundaries. A linear interpolation of either the densities themselves or their logarithms is used.");
}

CmdSimulation2DNEWInitDens::~CmdSimulation2DNEWInitDens()
{
	delete m_pLogArg;
	delete m_pPropertyArg;
	delete m_pUseVExtraArg;
}

bool CmdSimulation2DNEWInitDens::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	int w = pSim->getNumberOfXPixels();
	int h = pSim->getNumberOfYPixels();
	int propID = m_pPropertyArg->getValue();
	bool useLog = m_pLogArg->getValue();
	bool useVextra = m_pLogArg->getValue();
	std::vector<double> dens(w*h);

	if (useVextra && !useLog)
	{
		setErrorString("The 'useVextra' argument can only be true if the logarithm of the densities is being used");
		return false;
	}

	if (!pSim->isGridPropertySet(propID))
	{
		setErrorString("Specified grid property has not been set");
		return false;
	}

	if (!useVextra)
	{
		for (int x = 0 ; x < w ; x++)
		{
			double v0 = pSim->getGridProperty(propID, x, 0);
			double v1 = pSim->getGridProperty(propID, x, h-1);

			if (useLog)
			{
				v0 = std::log(v0);
				v1 = std::log(v1);
			}

			for (int y = 0 ; y < h ; y++)
			{
				double frac = ((double)y/(double)(h-1));
				double v = (v1-v0)*frac + v0;			

				if (useLog)
					v = std::exp(v);

				if (!pSim->setGridProperty(propID, x, y, v))
				{	
					setErrorString("Unable to set the grid value: " + pSim->getErrorString());
					return false;
				}
			}
		}
	}
	else // implies useLog
	{
		int propID2 = -1;

		if (propID == SIMSTATE_GRIDPROP_N)
			propID2 = SIMSTATE_GRIDPROP_VNEXTRA;
		else
			propID2 = SIMSTATE_GRIDPROP_VPEXTRA;

		if (!pSim->isDoublePropertySet(SIMSTATE_PROP_TEMP))
		{
			setErrorString("Temperature must be set for this");
			return false;
		}

		double T = pSim->getDoubleProperty(SIMSTATE_PROP_TEMP);
		double kTev = T*CONST_K/CHARGE_ELECTRON;

		if (propID2 == SIMSTATE_GRIDPROP_VPEXTRA)
			kTev *= -1;

		for (int x = 0 ; x < w ; x++)
		{
			double v0 = pSim->getGridProperty(propID, x, 0);
			double v1 = pSim->getGridProperty(propID, x, h-1);
			double vExtraBottom = pSim->getGridProperty(propID2, x, 0);
			double vExtraTop = pSim->getGridProperty(propID2, x, h-1);
			
			v0 = std::log(v0);
			v1 = std::log(v1);

			double Vdiff = (v1-v0)*kTev - (vExtraTop-vExtraBottom);;

			//std::cout << "Vdiff = " << Vdiff << std::endl;


			for (int y = 0 ; y < h ; y++)
			{
				double frac = ((double)y/(double)(h-1));
				double VextraCur = pSim->getGridProperty(propID2, x, y);
				double v = Vdiff*frac/kTev + v0 + (VextraCur-vExtraBottom)/kTev;

				v = std::exp(v);

				if (!pSim->setGridProperty(propID, x, y, v))
				{	
					setErrorString("Unable to set the grid value: " + pSim->getErrorString());
					return false;
				}
			}
		}
	}
		
	return true;
}

CmdSimulation2DNEWShowCurrent::CmdSimulation2DNEWShowCurrent(bool yCurrent, const std::string &cmdName) : ShellCommand(cmdName)
{
	m_yCurrent = yCurrent;

	setDescription("Calculate and display the net current in the simulation at certain positions.");
}

CmdSimulation2DNEWShowCurrent::~CmdSimulation2DNEWShowCurrent()
{
}

bool CmdSimulation2DNEWShowCurrent::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	if (m_yCurrent)
	{
		double bottom, top, overall, center;

		if (!pSim->calculateCurrent2DY(bottom, center, top, overall))
		{
			setErrorString("Unable to calculate current averages: " + pSim->getErrorString());
			return false;
		}
		
		bottom *= CHARGE_ELECTRON;
		top *= CHARGE_ELECTRON;
		overall *= CHARGE_ELECTRON;
		center *= CHARGE_ELECTRON;

		pIOSys->print("Top average:     %g A/m^2", top);
		pIOSys->print("Center average:  %g A/m^2", center);
		pIOSys->print("Bottom average:  %g A/m^2", bottom);
		pIOSys->print("Overall average: %g A/m^2", overall);
	}
	else
	{
		double left, right, overall, center;

		if (!pSim->calculateCurrent2DX(left, center, right, overall))
		{
			setErrorString("Unable to calculate current averages: " + pSim->getErrorString());
			return false;
		}
		
		left *= CHARGE_ELECTRON;
		right *= CHARGE_ELECTRON;
		overall *= CHARGE_ELECTRON;
		center *= CHARGE_ELECTRON;

		pIOSys->print("Left average:    %g A/m^2", left);
		pIOSys->print("Center average:  %g A/m^2", center);
		pIOSys->print("Right average:   %g A/m^2", right);
		pIOSys->print("Overall average: %g A/m^2", overall);
	}

	return true;
}

CmdSimulation2DNEWAvgPlot::CmdSimulation2DNEWAvgPlot(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to plot.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", PLOTWINID_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", PLOTWINID_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", PLOTWINID_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", PLOTWINID_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", PLOTWINID_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", PLOTWINID_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", PLOTWINID_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", PLOTWINID_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", PLOTWINID_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", PLOTWINID_EPS);
	m_pPropertyArg->registerChoice(PLOTWINNAME_R, "recombination rate", PLOTWINID_R);
	m_pPropertyArg->registerChoice(PLOTWINNAME_V, "potential", PLOTWINID_V);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JX, "current in x-direction", PLOTWINID_JX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JY, "current in y-direction", PLOTWINID_JY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNX, "electron current in x-direction", PLOTWINID_JNX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNY, "electron current in y-direction", PLOTWINID_JNY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPX, "hole current in x-direction", PLOTWINID_JPX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPY, "hole current in y-direction", PLOTWINID_JPY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", PLOTWINID_VN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", PLOTWINID_VP);
	addArgument(m_pPropertyArg);
	
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);

	m_pXYArg = new ShellCmdChoiceArg("xy");
	m_pXYArg->setDescription("Specifies the axis which should remain.");
	m_pXYArg->registerChoice("x", "X axis", 0);
	m_pXYArg->registerChoice("y", "Y axis", 1);
	addArgument(m_pXYArg);

	setDescription("Write 1D plot data of a quantity to a file. The values along the other axis are averaged out.");
}

CmdSimulation2DNEWAvgPlot::~CmdSimulation2DNEWAvgPlot()
{
	delete m_pPropertyArg;
	delete m_pFileNameArg;
	delete m_pXYArg;
}

bool CmdSimulation2DNEWAvgPlot::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	int width, height;
	double xOffset, yOffset;
	std::string title, xLabel, yLabel, zLabel;
	std::vector<double> dst;
	std::vector<double> result;

	int id = m_pPropertyArg->getValue();
	if (!getPlotInfo(id, pSim, dst, width, height, xOffset, yOffset, title, xLabel, yLabel, zLabel))
	{
		setErrorString("Couldn't get plot data, perhaps the specified property has not been set yet");
		return false;
	}

	if (m_pXYArg->getValue() == 0) // keep X-axis
	{
		result.resize(width);

		for (int x = 0 ; x < width ; x++)
		{
			double sum = 0;

			for (int y = 0 ; y < height ; y++)
				sum += dst[x+y*width];

			sum /= (double)height;

			result[x] = sum;
		}
	}
	else // keep Y-axis
	{
		result.resize(height);

		for (int y = 0 ; y < height ; y++)
		{
			double sum = 0;

			for (int x = 0 ; x < width ; x++)
				sum += dst[x+y*width];

			sum /= (double)width;

			result[y] = sum;
		}
	}

	std::string fileName = m_pFileNameArg->getValue();
	FILE *pFile = fopen(fileName.c_str(), "wt");

	if (pFile == 0)
	{
		setErrorString("Unable to open the specified file");
		return false;
	}

	for (int i = 0 ; i < result.size() ; i++)
		fprintf(pFile, "%d %g\n", i, result[i]);

	fclose(pFile);
	return true;
}

CmdSimulation2DNEWLinePlot::CmdSimulation2DNEWLinePlot(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to plot.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", PLOTWINID_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", PLOTWINID_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", PLOTWINID_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", PLOTWINID_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*n*p", PLOTWINID_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", PLOTWINID_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", PLOTWINID_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "electron mobility", PLOTWINID_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "hole mobility", PLOTWINID_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", PLOTWINID_EPS);
	m_pPropertyArg->registerChoice(PLOTWINNAME_R, "recombination rate", PLOTWINID_R);
	m_pPropertyArg->registerChoice(PLOTWINNAME_V, "potential", PLOTWINID_V);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JX, "current in x-direction", PLOTWINID_JX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JY, "current in y-direction", PLOTWINID_JY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNX, "electron current in x-direction", PLOTWINID_JNX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNY, "electron current in y-direction", PLOTWINID_JNY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPX, "hole current in x-direction", PLOTWINID_JPX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPY, "hole current in y-direction", PLOTWINID_JPY);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", PLOTWINID_VN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", PLOTWINID_VP);
	addArgument(m_pPropertyArg);
	
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);
	
	m_pCoordArg = new ShellCmdIntArg("coordinate");
	m_pCoordArg->setDescription("The X or Y coordinate of the line that should be plotted.");
	addArgument(m_pCoordArg);

	m_pXYArg = new ShellCmdChoiceArg("xy");
	m_pXYArg->setDescription("Specifies the axis which should remain.");
	m_pXYArg->registerChoice("x", "X axis", 0);
	m_pXYArg->registerChoice("y", "Y axis", 1);
	addArgument(m_pXYArg);

	setDescription("Write 1D plot data of a quantity to a file. The values along a specific line are used.");
}

CmdSimulation2DNEWLinePlot::~CmdSimulation2DNEWLinePlot()
{
	delete m_pPropertyArg;
	delete m_pFileNameArg;
	delete m_pCoordArg;
	delete m_pXYArg;
}

bool CmdSimulation2DNEWLinePlot::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	int width, height;
	double xOffset, yOffset;
	std::string title, xLabel, yLabel, zLabel;
	std::vector<double> dst;
	std::vector<double> result;

	int id = m_pPropertyArg->getValue();
	if (!getPlotInfo(id, pSim, dst, width, height, xOffset, yOffset, title, xLabel, yLabel, zLabel))
	{
		setErrorString("Couldn't get plot data, perhaps the specified property has not been set yet");
		return false;
	}

	int coord = m_pCoordArg->getValue()-1;

	bool isX = (m_pXYArg->getValue() == 0);
	if (coord < 0 || (isX && coord >= height) || (!isX && coord >= width))
	{
		setErrorString("An invalid coordinate was specified");
		return false;
	}

	if (isX) // keep X-axis
	{
		result.resize(width);

		for (int x = 0 ; x < width ; x++)
			result[x] = dst[x+coord*width];
	}
	else // keep Y-axis
	{
		result.resize(height);

		for (int y = 0 ; y < height ; y++)
			result[y] = dst[coord+y*width];
	}

	std::string fileName = m_pFileNameArg->getValue();
	FILE *pFile = fopen(fileName.c_str(), "wt");

	if (pFile == 0)
	{
		setErrorString("Unable to open the specified file");
		return false;
	}

	for (int i = 0 ; i < result.size() ; i++)
		fprintf(pFile, "%d %g\n", i, result[i]);

	fclose(pFile);
	return true;
}

CmdSimulation2DNEWRunDirect::CmdSimulation2DNEWRunDirect(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("1000");
	m_pMaxIterationsArg->setDescription("Maximum number of iterations to perform.");
	addArgument(m_pMaxIterationsArg);

	m_pShowCurrentsArg = new ShellCmdBoolArg("showcurrents");
	m_pShowCurrentsArg->setDefault("yes");
	m_pShowCurrentsArg->setDescription("If enabled, after each iteration the current in the y-direction will be shown.");
	addArgument(m_pShowCurrentsArg);

	m_pErrorArg = new ShellCmdRealArg("errlimit");
	m_pErrorArg->setDefault("-1");
	m_pErrorArg->setDescription("If the error after convergence exceeds this value, an error is generated. A negative value disables this feature.");
	addArgument(m_pErrorArg);

	m_pTimeArg = new ShellCmdRealArg("timeout");
	m_pTimeArg->setDefault("-1");
	m_pTimeArg->setDescription("The maximum amount of time the search for the equilibrium situation can take. Set to a negative value to disable.");
	addArgument(m_pTimeArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale, is only used in NR based simulation and potential based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale, is only used in NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	m_pNPOnlyArg = new ShellCmdBoolArg("nponly");
	m_pNPOnlyArg->setDescription("Do not change the electrostatic potential V, only look for N and P solutions.");
	m_pNPOnlyArg->setDefault("no");
	addArgument(m_pNPOnlyArg);

	setDescription("Instead of using a time step based simulation, use a direct search for the steady state solution.");
}

CmdSimulation2DNEWRunDirect::~CmdSimulation2DNEWRunDirect()
{
	delete m_pMaxIterationsArg;
	delete m_pShowCurrentsArg;
	delete m_pErrorArg;
	delete m_pTimeArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
	delete m_pNPOnlyArg;
}

bool CmdSimulation2DNEWRunDirect::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pState = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	int numY = pState->getNumberOfYPixels();
	double pixW = pState->getPixelWidth();
	double pixH = pState->getPixelHeight();
	double simWidth = (double)(numX)*pixW;
	double simHeight = (double)(numY-1)*pixH;
	bool useLog = m_pLogArg->getValue();
	bool npOnly = m_pNPOnlyArg->getValue();

	Simulation2DNR2 sim;
	Simulation2DNR2 *pSim = &sim;

	if (!pSim->init(numX, numY, simWidth, simHeight, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	int maxIterations = m_pMaxIterationsArg->getValue();
	bool showCurrent = m_pShowCurrentsArg->getValue();
	int iterations = 0;
	double errLimit = m_pErrorArg->getValue();
	double timeOut = m_pTimeArg->getValue();

	if (timeOut < 0)
		timeOut = 1e200; // should be more than enough

	double stopTime = getTimeOfDay() + timeOut;

	bool status;
	
	if (!npOnly) // default behaviour, look for everything
		status = searchEquilibrium(pSim, maxIterations, showCurrent, false, errLimit, iterations, stopTime);
	else
		status = searchEquilibrium(pSim, maxIterations, showCurrent, false, errLimit, iterations, stopTime, true, false, true);

	if (!status)
	{
		setErrorString(errStr);
		return false;
	}

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state: " + pSim->getErrorString());
		return false;
	}

	return true;
}

bool CmdSimulation2DNEWRunDirect::searchEquilibrium(Simulation2D *pSim0, int maxIterations, bool showCurrent, 
		bool errorOnMaxCount, double errorTolerance, int &iterations, std::string &errStr,
		double stopTime, bool runNPfirst, bool runVfirst, bool stopAfterNP)
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	int count = 0;
	Simulation2DNR2 *pSim = dynamic_cast<Simulation2DNR2 *>(pSim0);
	if (pSim == 0)
	{
		errStr = std::string("Not a Simulation2DNR2 instance");
		return false;
	}
	
	// TODO
	//if (0) 
	if (runVfirst)
	{
#if 0
		double sigmaPrev = 1e100;
		double sigma = sigmaPrev; 
		bool done = false;

		while (!done && count++ < maxIterations && getTimeOfDay() < stopTime)
		{
	//		std::cout << "stopTime    = " << stopTime << std::endl;
	//		std::cout << "currentTime = " << getTimeOfDay() << std::endl;

			if (pInst->isInterrupted())
			{
				if (showCurrent)
					pIOSys->writeStatusLine("");

				errStr = "Simulation interrupted by user";
				return false;
			}

	//		std::cout << "Iteration " << count << std::endl;

			if (!pSim->startV(sigma))
			{
				errStr = std::string("Error looking for steady state solution: ") + pSim->getErrorString();
				return false;
			}

			if (showCurrent)
			{
				char str[1024];
				double top, bottom, overall, center;

				pSim->calculateYCurrent(bottom, top, overall, center);

				bottom *= CHARGE_ELECTRON;
				top *= CHARGE_ELECTRON;
				overall *= CHARGE_ELECTRON;
				center *= CHARGE_ELECTRON;

				sprintf(str, "Error: %g | Bottom avg: %g | Center avg: %g | Top avg: %g | Overall avg: %g", sigma, bottom, center, top, overall);
				pIOSys->writeStatusLine(str);
			}

			if (!ISFINITE(sigma))
				break;

			if (sigma >= sigmaPrev)
				done = true;

			sigmaPrev = sigma;
		}

		iterations = count;

		if (!done || !ISFINITE(sigma))
		{
			if (!ISFINITE(sigma))
			{
				errStr = "Got NaN or inf value";
				return false;
			}

			if (count < maxIterations) // must have gotten a timeout then
			{
				pIOSys->print("Exceeded allowed run time");
				return false;
			}
			else
			{
				pIOSys->print("Maximum count reached, current error is %g", sigma);
				if (errorOnMaxCount)
				{
					errStr = "Maximum count reached";
					return false;
				}
			}
		}
		else
		{
			pIOSys->print("V Convergence reached after %d iterations, current error is %g", count, sigma);
			if (errorTolerance > 0 && (sigma > errorTolerance))
			{
				errStr = "Current error exceeds the specified error tolerance";
				return false;
			}
		}
#else
		if (!pSim->runV())
		{
			errStr = "Couldn't solve for V alone: " + pSim->getErrorString();
			return false;
		}
		pIOSys->print("Solved for V separately in 1 step");
#endif

	}

	// TODO
	//if (0) //
	if (runNPfirst)
	{
		double sigmaPrev = 1e100;
		double sigma = sigmaPrev; 
		bool done = false;

		while (!done && count++ < maxIterations && getTimeOfDay() < stopTime)
		{
	//		std::cout << "stopTime    = " << stopTime << std::endl;
	//		std::cout << "currentTime = " << getTimeOfDay() << std::endl;

			if (pInst->isInterrupted())
			{
				if (showCurrent)
					pIOSys->writeStatusLine("");

				errStr = "Simulation interrupted by user";
				return false;
			}

	//		std::cout << "Iteration " << count << std::endl;

			if (!pSim->startNP(sigma))
			{
				errStr = std::string("Error looking for steady state solution: ") + pSim->getErrorString();
				return false;
			}

			if (showCurrent)
			{
				char str[1024];
				double top, bottom, overall, center;

				pSim->calculateYCurrent(bottom, top, overall, center);

				bottom *= CHARGE_ELECTRON;
				top *= CHARGE_ELECTRON;
				overall *= CHARGE_ELECTRON;
				center *= CHARGE_ELECTRON;

				sprintf(str, "Error: %g | Bottom avg: %g | Center avg: %g | Top avg: %g | Overall avg: %g", sigma, bottom, center, top, overall);
				pIOSys->writeStatusLine(str);
			}

			if (!ISFINITE(sigma))
				break;

			if (sigma >= sigmaPrev)
				done = true;

			sigmaPrev = sigma;
		}

		iterations = count;

		if (!done || !ISFINITE(sigma))
		{
			if (!ISFINITE(sigma))
			{
				errStr = "Got NaN or inf value";
				return false;
			}

			if (count < maxIterations) // must have gotten a timeout then
			{
				pIOSys->print("Exceeded allowed run time");
				return false;
			}
			else
			{
				pIOSys->print("Maximum count reached, current error is %g", sigma);
				if (errorOnMaxCount)
				{
					errStr = "Maximum count reached";
					return false;
				}
			}
		}
		else
		{
			pIOSys->print("NP Convergence reached after %d iterations, current error is %g", count, sigma);
			if (errorTolerance > 0 && (sigma > errorTolerance))
			{
				errStr = "Current error exceeds the specified error tolerance";
				return false;
			}
		}

	}

	if (stopAfterNP)
		return true;
	else
	{
		double sigmaPrev = 1e100;
		double sigma = sigmaPrev; 
		bool done = false;

		while (!done && count++ < maxIterations && getTimeOfDay() < stopTime)
		{
	//		std::cout << "stopTime    = " << stopTime << std::endl;
	//		std::cout << "currentTime = " << getTimeOfDay() << std::endl;

			if (pInst->isInterrupted())
			{
				if (showCurrent)
					pIOSys->writeStatusLine("");

				errStr = "Simulation interrupted by user";
				return false;
			}

	//		std::cout << "Iteration " << count << std::endl;

			if (!pSim->start(sigma))
			{
				errStr = std::string("Error looking for steady state solution: ") + pSim->getErrorString();
				return false;
			}

			if (showCurrent)
			{
				char str[1024];
				double top, bottom, overall, center;

				pSim->calculateYCurrent(bottom, top, overall, center);

				bottom *= CHARGE_ELECTRON;
				top *= CHARGE_ELECTRON;
				overall *= CHARGE_ELECTRON;
				center *= CHARGE_ELECTRON;

				sprintf(str, "Error: %g | Bottom avg: %g | Center avg: %g | Top avg: %g | Overall avg: %g", sigma, bottom, center, top, overall);
				pIOSys->writeStatusLine(str);
			}

			if (!ISFINITE(sigma))
				break;

			if (sigma >= sigmaPrev)
				done = true;

			sigmaPrev = sigma;
		}

		iterations = count;

		if (!done || !ISFINITE(sigma))
		{
			if (!ISFINITE(sigma))
			{
				errStr = "Got NaN or inf value";
				return false;
			}

			if (count < maxIterations) // must have gotten a timeout then
			{
				pIOSys->print("Exceeded allowed run time");
				return false;
			}
			else
			{
				pIOSys->print("Maximum count reached, current error is %g", sigma);
				if (errorOnMaxCount)
				{
					errStr = "Maximum count reached";
					return false;
				}
			}
		}
		else
		{
			pIOSys->print("Total convergence reached after %d iterations, current error is %g", count, sigma);
			if (errorTolerance > 0 && (sigma > errorTolerance))
			{
				errStr = "Current error exceeds the specified error tolerance";
				return false;
			}
		}
	}

	return true;
}

CmdSimulation2DNEWSetTemperature::CmdSimulation2DNEWSetTemperature(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pTemp = new ShellCmdRealArg("T");
	m_pTemp->setDescription("Temperature for the simulation.");
	m_pTemp->setMin(0);
	m_pTemp->setMax(10000);
	addArgument(m_pTemp);

	setDescription("Sets the temperature parameter of the simulation.");
}

CmdSimulation2DNEWSetTemperature::~CmdSimulation2DNEWSetTemperature()
{
	delete m_pTemp;
}

bool CmdSimulation2DNEWSetTemperature::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double T = m_pTemp->getValue();

	if (!pSim->setDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("Couldn't set property: " + pSim->getErrorString());
		return false;
	}

	return true;
}

CmdSimulation2DNEWShowTemperature::CmdSimulation2DNEWShowTemperature(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Shows the potential difference set in the current simulation (Vright-Vleft).");
}

CmdSimulation2DNEWShowTemperature::~CmdSimulation2DNEWShowTemperature()
{
}

bool CmdSimulation2DNEWShowTemperature::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	if (!pSim->isDoublePropertySet(SIMSTATE_PROP_TEMP))
	{
		setErrorString("No temperature has been set yet");
		return false;
	}

	double T = pSim->getDoubleProperty(SIMSTATE_PROP_TEMP);

	pIOSys->print("%g K", T);

	return true;
}

CmdSimulation2DNEWIVCurve::CmdSimulation2DNEWIVCurve(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pVIntArg = new ShellCmdRealArg("vint");
	m_pVIntArg->setDescription("The built-in potential.");
	addArgument(m_pVIntArg);

	m_pVStartArg = new ShellCmdRealArg("vstart");
	m_pVStartArg->setDescription("Start value of the applied potential.");
	addArgument(m_pVStartArg);

	m_pVStopArg = new ShellCmdRealArg("vstop");
	m_pVStopArg->setDescription("End value of the applied potential.");
	addArgument(m_pVStopArg);

	m_pNumStepsArg = new ShellCmdIntArg("steps");
	m_pNumStepsArg->setMin(2);
	m_pNumStepsArg->setDefault("100");
	m_pNumStepsArg->setDescription("The number of points in the IV curve.");
	addArgument(m_pNumStepsArg);

	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDefault("*");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);

	m_pAppendArg = new ShellCmdBoolArg("append");
	m_pAppendArg->setDescription("Flag indicating if the plot data should be appended to an existing file.");
	m_pAppendArg->setDefault("yes");
	addArgument(m_pAppendArg);

	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("1000");
	m_pMaxIterationsArg->setDescription("Maximum number of solution iterations to perform, for each potential difference.");
	addArgument(m_pMaxIterationsArg);

	m_pErrorArg = new ShellCmdRealArg("errlimit");
	m_pErrorArg->setDefault("-1");
	m_pErrorArg->setDescription("If the error after convergence exceeds this value, the multiresolution search will stop. A negative value disables this feature.");
	addArgument(m_pErrorArg);

	m_pTimeArg = new ShellCmdRealArg("timeout");
	m_pTimeArg->setDefault("-1");
	m_pTimeArg->setDescription("The maximum amount of time the search for the IV curve can take. Set to a negative value to disable.");
	addArgument(m_pTimeArg);

	m_pDoubleSearchArg = new ShellCmdBoolArg("doublesearch");
	m_pDoubleSearchArg->setDefault("false");
	m_pDoubleSearchArg->setDescription("If set, a routine to search for the equilibrium at each voltage will be run twice. This takes a bit longer but gives better results for more difficult parameter combinations.");
	addArgument(m_pDoubleSearchArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	m_pWarnError = new ShellCmdBoolArg("warningsaserrors");
	m_pWarnError->setDescription("Treat warnings as errors");
	m_pWarnError->setDefault("yes");
	addArgument(m_pWarnError);

	setDescription("Calculate an IV curve using the direct solution method. The voltage between right and left "
		       "contacts is set to 'vint-vapplied'.");
}

CmdSimulation2DNEWIVCurve::~CmdSimulation2DNEWIVCurve()
{
	delete m_pVIntArg;
	delete m_pVStartArg;
	delete m_pVStopArg;
	delete m_pNumStepsArg;
	delete m_pFileNameArg;
	delete m_pAppendArg;
	delete m_pMaxIterationsArg;
	delete m_pErrorArg;
	delete m_pTimeArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
	delete m_pWarnError;
}

bool CmdSimulation2DNEWIVCurve::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pState = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	int numY = pState->getNumberOfYPixels();
	double pixW = pState->getPixelWidth();
	double pixH = pState->getPixelHeight();
	double simWidth = (double)(numX)*pixW;
	double simHeight = (double)(numY-1)*pixH;
	bool useLog = m_pLogArg->getValue();

	Simulation2DNR2 sim;
	Simulation2DNR2 *pSim = &sim;

	if (!pSim->init(numX, numY, simWidth, simHeight, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());

		if (m_pWarnError->getValue())
		{
			setErrorString("Treating warnings as errors!");
			return false;
		}
	}

	bool doubleSearch = m_pDoubleSearchArg->getValue();
	double timeOut = m_pTimeArg->getValue();

	if (timeOut < 0)
		timeOut = 1e200; // should be more than enough

	double stopTime = getTimeOfDay() + timeOut;

	std::string fileName = m_pFileNameArg->getValue();
	bool append = m_pAppendArg->getValue();
	FILE *pFile = 0;
	
	if (fileName != std::string("*"))
	{
		if (append)
			pFile = fopen(fileName.c_str(), "at");
		else
			pFile = fopen(fileName.c_str(), "wt");

		if (pFile == 0)
		{
			setErrorString("Unable to write to the specified file");
			return false;
		}
	}

	double Vint = m_pVIntArg->getValue();
	double V0 = m_pVStartArg->getValue();
	double V1 = m_pVStopArg->getValue();
	double errLimit = m_pErrorArg->getValue();
	int maxIterations = m_pMaxIterationsArg->getValue();
	int steps = m_pNumStepsArg->getValue();
	double dV = (V1-V0)/(double)(steps-1);

	double V = V0;

	std::vector<double> Vvalues(steps);
	std::vector<double> Jvalues;

	for (int i = 0 ; i < steps ; i++, V += dV)
		Vvalues[i] = Vint-V;

	double curVdiff = pSim->getPotentialDifference();
	double minDiff = std::abs(curVdiff-Vvalues[0]);

	int minPos = 0;

	for (int i = 1 ; i < steps ; i++)
	{
		double d = std::abs(curVdiff - Vvalues[i]);

		if (d < minDiff)
		{
			minDiff = d;
			minPos = i;
		}
	}

	std::vector<double> Vdiffs[2];
	int targetIteration = -1;
	int targetIndex = -1;
	double targetVDiff = Vvalues[steps-1];

	for (int i = minPos ; i >= 0 ; i--)
		Vdiffs[0].push_back(Vvalues[i]);
	
	for (int i = minPos+1 ; i < steps ; i++)
		Vdiffs[1].push_back(Vvalues[i]);

	if (minPos+1 < steps)
	{
		targetIteration = 1;
		targetIndex = (steps-1)-(minPos+1);
	}
	else // should mean that minPos+1 = steps <=> minPos = steps-1
	{
		targetIteration = 0;
		targetIndex = 0;
	}

	Vvalues.clear();

	std::vector<double> n, p, potential;
	std::vector<double> targetN, targetP, targetV;

	pSim->getElectronNumberDensity(n);
	pSim->getHoleNumberDensity(p);
	pSim->getPotential(potential);

	int totalCount = 0;

	pIOSys->setPercentageFeedback(0);

	for (int j = 0 ; j < 2 ; j++)
	{
		for (int i = 0 ; i < Vdiffs[j].size() ; i++, totalCount++)
		{
			double Vdiff = Vdiffs[j][i]; // Vdiff = Vint - Vapp
			double Vapp = Vint - Vdiff;

			pSim->setPotentialDifference(Vdiff);

			bool showCurrent = true;
			std::string errStr;
			int iterations = 0;

			if (doubleSearch)
			{
				if (!CmdSimulation2DNEWRunDirect::searchEquilibrium(pSim, maxIterations, showCurrent, true, errLimit, iterations, errStr, stopTime, true, true))
				{
					setErrorString(errStr);
					CmdSimulation1DNEWIVCurve::finishFile(pFile, Vvalues, Jvalues);
					return false;
				}
				if (!CmdSimulation2DNEWRunDirect::searchEquilibrium(pSim, maxIterations, showCurrent, true, errLimit, iterations, errStr, stopTime, true, true))
				{
					setErrorString(errStr);
					CmdSimulation1DNEWIVCurve::finishFile(pFile, Vvalues, Jvalues);
					return false;
				}
			}
			else // old-style search
			{
				if (!CmdSimulation2DNEWRunDirect::searchEquilibrium(pSim, maxIterations, showCurrent, true, errLimit, iterations, errStr, stopTime))
				{
					setErrorString(errStr);
					CmdSimulation1DNEWIVCurve::finishFile(pFile, Vvalues, Jvalues);
					return false;
				}
			}

			double top, bottom, overall, center;

			pSim->calculateYCurrent(bottom, top, overall, center);

			//double J = overall;
			double J = center;

			J *= CHARGE_ELECTRON;

			Vvalues.push_back(Vapp);
			Jvalues.push_back(J);

			pIOSys->plot2D(PLOTWINID_2DIV, PLOTWINIDNAME_2DIV, "V", "J", Vvalues, Jvalues, true);

			if (j == targetIteration && i == targetIndex)
			{
				pSim->getElectronNumberDensity(targetN);
				pSim->getHoleNumberDensity(targetP);
				pSim->getPotential(targetV);
			}

			pIOSys->setPercentageFeedback(MIN((double)(totalCount+1)/(double)steps*100.0+0.5,100.0));
		}

		if (j == 0)
		{
			if (n.size() > 0)
			{
				int numX = pSim->getNumXPixels();
				int numY = pSim->getNumYPixels();

				pSim->setPotentialDifference(curVdiff);

				for (int x = 0 ; x < numX ; x++)
				{
					for (int y = 0 ; y < numY ; y++)
					{
						int pos = x+y*numX;

						pSim->setElectronNumberDensity(x, y, n[pos]);
						pSim->setHoleNumberDensity(x, y, p[pos]);
						pSim->setPotential(x, y, potential[pos]);
					}
				}
			}

			// Swap order

			int num = Vvalues.size();
			int num2 = num/2;

			for (int i = 0 ; i < num2 ; i++)
			{
				double tmp = Vvalues[i];
				Vvalues[i] = Vvalues[num-1-i];
				Vvalues[num-1-i] = tmp;

				tmp = Jvalues[i];
				Jvalues[i] = Jvalues[num-1-i];
				Jvalues[num-1-i] = tmp;
			}
		}
	}

	//int numX = pSim->getNumXPixels();
	//int numY = pSim->getNumYPixels();

	pSim->setPotentialDifference(targetVDiff);

	for (int x = 0 ; x < numX ; x++)
	{
		for (int y = 0 ; y < numY ; y++)
		{
			int pos = x+y*numX;

			pSim->setElectronNumberDensity(x, y, targetN[pos]);
			pSim->setHoleNumberDensity(x, y, targetP[pos]);
			pSim->setPotential(x, y, targetV[pos]);
		}
	}

	CmdSimulation1DNEWIVCurve::finishFile(pFile, Vvalues, Jvalues);
	
	double Voc = -1;
	double Jsc = -1;

	if (CmdSimulation1DNEWIVCurve::findVoc(Vvalues, Jvalues, Voc))
		pIOSys->print("Estimated Voc = %g", Voc);
	else
		pIOSys->writeOutputLine("Unable to estimate Voc from curve");

	if (CmdSimulation1DNEWIVCurve::findVoc(Jvalues, Vvalues, Jsc)) // Abusing the Voc routine to find Jsc
		pIOSys->print("Estimated Jsc = %g", Jsc);
	else
		pIOSys->writeOutputLine("Unable to estimate Jsc from curve");

	pIOSys->setPercentageFeedback(100);

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

CmdSimulation2DNEWRunDirectMultiRes::CmdSimulation2DNEWRunDirectMultiRes(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("100000");
	m_pMaxIterationsArg->setDescription("Maximum number of iterations to perform.");
	addArgument(m_pMaxIterationsArg);

	m_pShowCurrentsArg = new ShellCmdBoolArg("showcurrents");
	m_pShowCurrentsArg->setDefault("yes");
	m_pShowCurrentsArg->setDescription("If enabled, after each iteration the current in the y-direction will be shown.");
	addArgument(m_pShowCurrentsArg);

	m_pMinXArg = new ShellCmdIntArg("minx");
	m_pMinXArg->setMin(2);
	m_pMinXArg->setDefault("8");
	m_pMinXArg->setDescription("Minimal grid size in the x-direction");
	addArgument(m_pMinXArg);

	m_pMinYArg = new ShellCmdIntArg("miny");
	m_pMinYArg->setMin(3);
	m_pMinYArg->setDefault("8");
	m_pMinYArg->setDescription("Minimal grid size in the y-direction");
	addArgument(m_pMinYArg);

	m_pStepsArg = new ShellCmdIntArg("steps");
	m_pStepsArg->setMin(2);
	m_pStepsArg->setDefault("5");
	m_pStepsArg->setDescription("Number of multiresolution steps.");
	addArgument(m_pStepsArg);

	m_pErrorArg = new ShellCmdRealArg("errlimit");
	m_pErrorArg->setDefault("-1");
	m_pErrorArg->setDescription("If the error after convergence exceeds this value, the multiresolution search will stop. A negative value disables this feature.");
	addArgument(m_pErrorArg);

//	m_pFileNameArg = new ShellCmdStringArg("filename");
//	m_pFileNameArg->setDefault("*");
//	m_pFileNameArg->setDescription("If specified, the grids obtained during the search will be written to this file. First in the file will be "
//			               "the initial situation of the target grid, then, for each resolution the begin and end situation is written.");
//	addArgument(m_pFileNameArg);

	m_pTimeArg = new ShellCmdRealArg("timeout");
	m_pTimeArg->setDefault("-1");
	m_pTimeArg->setDescription("The maximum amount of time the search for the equilibrium situation can take. Set to a negative value to disable.");
	addArgument(m_pTimeArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically."); 
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	setDescription("Instead of using a time step based simulation, use a direct search for the steady state solution. This version uses a multi-resolution approach: first a solution is sought to a low-resolution approximation, which is used as starting situation in a higher resolution version. This process is continued until the true resolution is reached.");
}

CmdSimulation2DNEWRunDirectMultiRes::~CmdSimulation2DNEWRunDirectMultiRes()
{
	delete m_pMaxIterationsArg;
	delete m_pShowCurrentsArg;
	delete m_pMinXArg;
	delete m_pMinYArg;
	delete m_pStepsArg;
	delete m_pErrorArg;
	//delete m_pFileNameArg;
	delete m_pTimeArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
}

bool CmdSimulation2DNEWRunDirectMultiRes::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pState = pInst->getSimulationState2D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	int numY = pState->getNumberOfYPixels();
	double pixW = pState->getPixelWidth();
	double pixH = pState->getPixelHeight();
	double simWidth = (double)(numX)*pixW;
	double simHeight = (double)(numY-1)*pixH;
	bool useLog = m_pLogArg->getValue();

	Simulation2DNR2 sim;
	Simulation2DNR2 *pSim = &sim;

	if (!pSim->init(numX, numY, simWidth, simHeight, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	double errLimit = m_pErrorArg->getValue();
	double timeOut = m_pTimeArg->getValue();

	if (timeOut < 0)
		timeOut = 1e200; // should be more than enough

	double stopTime = getTimeOfDay() + timeOut;

	//double T = pSim->getTemperature();
	//double densScale = pSim->getDensScale();
	//double diffScale = pSim->getDiffusionScale();

	//bool useLog = pSim->isUsingLog();

	int iterationsLeft = m_pMaxIterationsArg->getValue();
	bool showCurrent = m_pShowCurrentsArg->getValue();

	int minX = m_pMinXArg->getValue();
	int minY = m_pMinYArg->getValue();
	int steps = m_pStepsArg->getValue();

	int maxX = pSim->getNumXPixels();
	int maxY = pSim->getNumYPixels();
	double realWidth = pSim->getPixelWidth() * (double)maxX;
	double realHeight = pSim->getPixelHeight() * (double)(maxY-1);

	if (minX > maxX || minY > maxY)
	{
		setErrorString("Minimum dimensions exceed the current grid size");
		return false;
	}

	Simulation2DNR2 *pPrevSim = createSimAndImport(minX, minY, realWidth, realHeight, T, diffScale, densScale, useLog, *pState);

	if (pPrevSim == 0)
	{
		// error has already been set
		return false;
	}

	bool skipped = false;

	pIOSys->setPercentageFeedback(0);

	for (int i = 0 ; i < steps-1 ; i++)
	{
		int iterations = 0;
		std::string errStr;

		if (!skipped)
		{
			if (!CmdSimulation2DNEWRunDirect::searchEquilibrium(pPrevSim, iterationsLeft, showCurrent, true, errLimit, iterations, errStr, stopTime, i != 0))
			{
				setErrorString(errStr);
				delete pPrevSim;
				return false;
			}
			
			iterationsLeft -= iterations;
		}
		
		int newX = (int)((double)(maxX-minX)*(double)(i+1)/(double)(steps-1) + (double)minX + 0.5);
		int newY = (int)((double)(maxY-minY)*(double)(i+1)/(double)(steps-1) + (double)minY + 0.5);

		if (newX > maxX)
			newX = maxX;
		if (newY > maxY)
			newY = maxY;

		if (newX == pPrevSim->getNumXPixels() && newY == pPrevSim->getNumYPixels())
		{
			skipped = true;
			continue;
		}
		else
			skipped = false;

		SimulationState newState;

		if (!transferState(pPrevSim, newX, newY, realWidth, realHeight, *pState, newState))
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		Simulation2DNR2 *pNewSim = createSimAndImport(newX, newY, realWidth, realHeight, T, diffScale, densScale, useLog, newState);

		if (pNewSim == 0)
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		delete pPrevSim;
		pPrevSim = pNewSim;

		pIOSys->setPercentageFeedback(MIN((double)(i+1)/(double)steps*100.0+0.5,100.0));
	}

	{
		SimulationState newState;

		if (!transferState(pPrevSim, maxX, maxY, realWidth, realHeight, *pState, newState))
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		delete pPrevSim;

		std::vector<std::string> warnings;

		if (!pSim->setState(newState, warnings))
		{
			setErrorString("Unable to store full resolution state for final simulation: " + pSim->getErrorString());
			return false;
		}

		int iterations;
		std::string errStr;

		if (!CmdSimulation2DNEWRunDirect::searchEquilibrium(pSim, iterationsLeft, showCurrent, true, errLimit, iterations, errStr, stopTime, true))
		{
			setErrorString(errStr);
			return false;
		}
		
		pIOSys->setPercentageFeedback(100);
	}

	pIOSys->print("Total iterations: %d", m_pMaxIterationsArg->getValue() - iterationsLeft);

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

bool CmdSimulation2DNEWRunDirectMultiRes::transferState(const Simulation2DNR2 *pPrevSim, int newX, int newY,
		                                        double realWidth, double realHeight,
							const SimulationState &highResState,
							SimulationState &newState)
{
	bool gridImportFlags[SIMSTATE_GRIDPROP_MAX];
	bool doubleImportFlags[SIMSTATE_PROP_MAX];

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		gridImportFlags[i] = false;
	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		doubleImportFlags[i] = false;

	gridImportFlags[SIMSTATE_GRIDPROP_N] = true;
	gridImportFlags[SIMSTATE_GRIDPROP_P] = true;
	gridImportFlags[SIMSTATE_GRIDPROP_V] = true;

	SimulationState resultState;
	int resultNumX = pPrevSim->getNumXPixels();
	int resultNumY = pPrevSim->getNumYPixels();
	double resultPixWidth = realWidth/(double)(resultNumX); // periodic in X
	double resultPixHeight = realHeight/(double)(resultNumY-1); 

	if (!resultState.init(resultNumX, resultNumY, resultPixWidth, resultPixHeight))
	{
		setErrorString("Couldn't initialize intermediate state: " + resultState.getErrorString());
		return false;
	}

	if (!pPrevSim->storeState(resultState))
	{
		setErrorString("Couldn't store intermediate state: " + pPrevSim->getErrorString());
		return false;
	}

	int newPixWidth = realWidth/(double)(newX);
	int newPixHeight = realHeight/(double)(newY-1);

	if (!newState.init(newX, newY, newPixWidth, newPixHeight))
	{
		setErrorString("Couldn't init intermediate new state: " + newState.getErrorString());
		return false;
	}

	if (!newState.import(highResState))
	{
		setErrorString("Unable to import high resolution settings in new intermediate state: " + newState.getErrorString());
		return false;
	}

	std::vector<bool> excludeMap(newX*newY);

	for (int i = 0 ; i < excludeMap.size() ; i++)
		excludeMap[i] = false;

	for (int x = 0 ; x < newX ; x++)
	{
		excludeMap[x] = true;
		excludeMap[x+(newY-1)*newX] = true;
	}

	// Keep the border conditions from the high resolution version

	if (!newState.import(resultState, gridImportFlags, doubleImportFlags, excludeMap))
	{
		setErrorString("Unable to import low resolution results for n, p and V in new intermediate state: " + newState.getErrorString());
		return false;
	}
	
	// Set the potential at the borders
	
	double V = newState.getDoubleProperty(SIMSTATE_PROP_VDIFF);

	for (int x = 0 ; x < newX ; x++)
	{
		newState.setGridProperty(SIMSTATE_GRIDPROP_V, x, 0, 0.0);
		newState.setGridProperty(SIMSTATE_GRIDPROP_V, x, newY-1, V);
	}

	return true;
}

Simulation2DNR2 *CmdSimulation2DNEWRunDirectMultiRes::createSimAndImport(int numX, int numY, double realWidth, 
								        double realHeight, double T,
									double diffScale, double densScale,
									bool useLog, SimulationState &state)
{
	Simulation2DNR2 *pSim = new Simulation2DNR2();

	if (!pSim->init(numX, numY, realWidth, realHeight, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		delete pSim;
		return 0;
	}

	std::vector<std::string> warnings;

	if (state.getNumberOfXPixels() == pSim->getNumXPixels() && state.getNumberOfYPixels() == pSim->getNumYPixels())
	{
		if (!pSim->setState(state, warnings))
		{
			setErrorString("Unable to set the state: " + pSim->getErrorString());
			delete pSim;
			return 0;
		}
	}
	else
	{
		double pixWidth = realWidth/(double)(numX); // periodic in X
		double pixHeight = realHeight/(double)(numY-1);

		SimulationState tmpState;

		if (!tmpState.init(numX, numY, pixWidth, pixHeight))
		{
			setErrorString("Unable to initialize temporary state: " + tmpState.getErrorString());
			delete pSim;
			return 0;
		}

		if (!tmpState.import(state))
		{
			setErrorString("Unable to import state into temporary state: " + tmpState.getErrorString());
			delete pSim;
			return 0;
		}

		if (!pSim->setState(tmpState, warnings))
		{
			setErrorString("Unable to set the state: " + pSim->getErrorString());
			delete pSim;
			return 0;
		}
	}
	return pSim;
}



CmdSimulation2DNEWRun::CmdSimulation2DNEWRun(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pStepsArg = new ShellCmdIntArg("steps");
	m_pStepsArg->setMin(1);
	m_pStepsArg->setDescription("Number of time steps to advance the simulation with.");
	addArgument(m_pStepsArg);

	m_pDtArg = new ShellCmdRealArg("dt");
	m_pDtArg->setDescription("Size of the time step in numerrically solving the differential equation.");
	addArgument(m_pDtArg);

	m_pShowCurrentsStepsArg = new ShellCmdIntArg("ycursteps");
	m_pShowCurrentsStepsArg->setDefault("0");
	m_pShowCurrentsStepsArg->setDescription("The number of steps after which each time the y-current will be shown. A value smaller than 1 disables the feature.");
	addArgument(m_pShowCurrentsStepsArg);

	m_pSaveIntNameArg = new ShellCmdStringArg("tmpsavefile");
	m_pSaveIntNameArg->setDefault("*");
	m_pSaveIntNameArg->setDescription("Name of the file to which temporary results should be saved. The character '%' will be expanded to the current simulation step number.");
	addArgument(m_pSaveIntNameArg);

	m_pSaveIntStepsArg = new ShellCmdIntArg("tmpsavesteps");
	m_pSaveIntStepsArg->setDefault("0");
	m_pSaveIntStepsArg->setDescription("Each multiple of this number of steps, the current simulation state will be saved to a file. A value smaller than 1 disables this feature.");
	addArgument(m_pSaveIntStepsArg);

	m_pPlotFileNameArg = new ShellCmdStringArg("plotfile");
	m_pPlotFileNameArg->setDefault("*");
	m_pPlotFileNameArg->setDescription("Name of the file to which the simulation plot data should be written each time a number of steps has passed. If the file exists, data will be appended. The character '%' will be expanded to the current simulation step number.");
	addArgument(m_pPlotFileNameArg);

	m_pPlotFileStepsArg = new ShellCmdIntArg("plotfilesteps");
	m_pPlotFileStepsArg->setDefault("0");
	m_pPlotFileStepsArg->setDescription("Each time this number of steps has passed, the current simulation plot data will be written to the specified file. A value smaller than 1 disables the feature.");
	addArgument(m_pPlotFileStepsArg);

	m_pPlotWindowStepsArg = new ShellCmdIntArg("plotwinsteps");
	m_pPlotWindowStepsArg->setDefault("0");
	m_pPlotWindowStepsArg->setDescription("Each time this number of steps has passed, the current simulation data will be plotted on screen. A value smaller than 1 disables the feature.");
	addArgument(m_pPlotWindowStepsArg);

	m_pTypeArg = new ShellCmdChoiceArg("simtype");
	m_pTypeArg->registerChoice("double", "Straightforward double based simulation.", 0);
	m_pTypeArg->registerChoice("doublerel", "Used current state as base and calculates relative changes each time.", 1);
	m_pTypeArg->registerChoice("gpu", "GPU based implementation. Should be faster for larger grids, but since on the GPU a float representation is used, it may be less precise.", 2);
	m_pTypeArg->setDefault("double");
	addArgument(m_pTypeArg);
	
	m_pInverseSolverArg = new ShellCmdBoolArg("inversematrixsolver");
	m_pInverseSolverArg->setDefault("yes");
	m_pInverseSolverArg->setDescription("Use the inverse matrix solver to solver the Poisson equation (should be slightly more accurate, causing a faster convergence to steady state)");
	addArgument(m_pInverseSolverArg);

	setDescription("Advance the simulation for a number of time steps, starting from the current state.");
}

CmdSimulation2DNEWRun::~CmdSimulation2DNEWRun()
{
	delete m_pStepsArg;
	delete m_pDtArg;
	delete m_pShowCurrentsStepsArg;
	delete m_pSaveIntNameArg;
	delete m_pSaveIntStepsArg;
	delete m_pPlotFileNameArg;
	delete m_pPlotFileStepsArg;
	delete m_pPlotWindowStepsArg;
	delete m_pTypeArg;
	delete m_pInverseSolverArg;
}

bool CmdSimulation2DNEWRun::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState2D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	bool Tset = false;
	double T = 0;

	if (pState->isDoublePropertySet(SIMSTATE_PROP_TEMP))
	{
		T = pState->getDoubleProperty(SIMSTATE_PROP_TEMP);
		Tset = true;
	}

	int numX = pState->getNumberOfXPixels();
	int numY = pState->getNumberOfYPixels();
	double pixW = pState->getPixelWidth();
	double pixH = pState->getPixelHeight();
	double realWidth = (double)(numX) * pixW;
	double realHeight = (double)(numY-1) * pixH;

	int steps = 1;
	int lowWidth = numX;
	int lowHeight = numY;
	bool done = false;

	while (lowWidth > 4 && lowHeight > 4 && !done)
	{
		if (lowWidth%2 == 0 && lowHeight%2 == 0)
		{
			steps++;
			lowWidth /= 2;
			lowHeight /= 2;
		}
		else
			done = true;
	}

	pIOSys->print("Using lowest resolution of %dx%d pixels", lowWidth, lowHeight);

	Simulation2D *pSim = 0;
	Simulation2DDouble simDouble;
	Simulation2DDoubleRel simDoubleRel;
	bool status;
	int newNumX = 0;
	int newNumY = 0;

#ifdef SIMICONDUCTOR_CONFIG_OPENCL
	Simulation2DDoubleGPURel simDoubleGPURel;

	if (m_pTypeArg->getValue() == 0)
	{
		pSim = &simDouble;
		status = simDouble.init(lowWidth, lowHeight, realWidth, realHeight, steps, &newNumX, &newNumY);
	}
	else if (m_pTypeArg->getValue() == 1)
	{
		pSim = &simDoubleRel;
		status = simDoubleRel.init(lowWidth, lowHeight, realWidth, realHeight, steps, &newNumX, &newNumY);
	}
	else
	{
		pSim = &simDoubleGPURel;
		status = simDoubleGPURel.init(lowWidth, lowHeight, realWidth, realHeight, steps, &newNumX, &newNumY);
	}
#else
	if (m_pTypeArg->getValue() == 0)
	{
		pSim = &simDouble;
		status = simDouble.init(lowWidth, lowHeight, realWidth, realHeight, steps, &newNumX, &newNumY);
	}
	else if (m_pTypeArg->getValue() == 1)
	{
		pSim = &simDoubleRel;
		status = simDoubleRel.init(lowWidth, lowHeight, realWidth, realHeight, steps, &newNumX, &newNumY);
	}
	else
	{
		setErrorString("GPU based simulation is not available");
		return false;
	}
#endif // SIMICONDUCTOR_CONFIG_OPENCL

	if (!status)
	{
		setErrorString("Unable to initialize timestep based simulation: " + pSim->getErrorString());
		return false;
	}

	if (newNumX != numX || newNumY != numY)
	{
		setErrorString("Internal error in calculation of scale steps");
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	int totalSteps = m_pStepsArg->getValue();
	double dt = m_pDtArg->getValue();

	std::vector<int> stepVector;

	bool showCurrents = false;
	int showCurrentSteps = m_pShowCurrentsStepsArg->getValue();
	if (showCurrentSteps > 0)
	{
		stepVector.push_back(showCurrentSteps);
		showCurrents = true;
	}

	bool saveSimulation = false;
	std::string saveName = m_pSaveIntNameArg->getValue();
	int saveSteps = m_pSaveIntStepsArg->getValue();
	if (saveName != std::string("*") && saveSteps > 0)
	{
		stepVector.push_back(saveSteps);
		saveSimulation = true;
	}

	bool plotToFile = false;
	std::string plotFileName = m_pPlotFileNameArg->getValue();
	int plotFileSteps = m_pPlotFileStepsArg->getValue();
	if (plotFileName != std::string("*") && plotFileSteps > 0)
	{
		stepVector.push_back(plotFileSteps);
		plotToFile = true;
	}

	bool plotToWin = false;
	int plotWinSteps = m_pPlotWindowStepsArg->getValue();
	if (plotWinSteps > 0)
	{
		stepVector.push_back(plotWinSteps);
		plotToWin = true;
	}

	pIOSys->setPercentageFeedback(0);

	int stepsLeft = totalSteps;
	int counter = 0;
	std::vector<int> maxSteps = stepVector;

	std::vector<double> dst;

	//pInst->set2DSavedFlag(false);

	bool inverseSolver = m_pInverseSolverArg->getValue();

	while (stepsLeft > 0)
	{
		int steps = stepsLeft;

		for (int i = 0 ; i < stepVector.size() ; i++)
		{
			if (stepVector[i] < steps)
				steps = stepVector[i];
		}

		if (!pSim->start(5, dt, steps, inverseSolver))
		{
			setErrorString(std::string("Error running simulation: ") + pSim->getErrorString());
			return false;
		}

		// TODO: should we do this?
		// Problem is that otherwise the plot routine, file output routine, ... don't have the current values
		// Maybe we should use a copy?

		if (!pSim->storeState(*pState))
		{
			setErrorString("Unable to store the simulation results into the simulation state");
			return false;
		}

		if (Tset)
			pState->setDoubleProperty(SIMSTATE_PROP_TEMP, T);

		stepsLeft -= steps;
		counter += steps;

		for (int i = 0 ; i < stepVector.size() ; i++)
		{
			stepVector[i] -= steps;
			if (stepVector[i] <= 0)
				stepVector[i] = maxSteps[i];
		}

		if (showCurrents && counter%showCurrentSteps == 0)
		{
			char str[1024];
			double top, bottom, overall, center;

			pSim->calculateYCurrent(bottom, top, overall, center);

			bottom *= CHARGE_ELECTRON;
			top *= CHARGE_ELECTRON;
			overall *= CHARGE_ELECTRON;
			center *= CHARGE_ELECTRON;

			sprintf(str, "Bottom avg: %g | Center avg: %g | Top avg: %g | Overall avg: %g", bottom, center, top, overall);
			pIOSys->writeStatusLine(str);

		}
		if (saveSimulation && counter%saveSteps == 0)
		{
			char numStr[256];

			// snprintf
			sprintf(numStr, "%d", counter);

			std::string fileName = replaceString(saveName, "%", numStr);

			if (!pState->save(fileName))
				pIOSys->writeOutputLine(std::string("Warning: couldn't save state to file ") + fileName + std::string(": ") + pSim->getErrorString());

		}
		if (plotToFile && counter%plotFileSteps == 0)
		{
			char numStr[256];

			// snprintf
			sprintf(numStr, "%d", counter);

			std::string fileName = replaceString(plotFileName, "%", numStr);

			FILE *pFile = fopen(fileName.c_str(), "at");

			if (pFile == 0)
				pIOSys->writeOutputLine(std::string("Warning: couldn't open plot file ") + fileName);
			else
			{
				pState->writePlotData(pFile);
				fclose(pFile);
			}
		}
		if (plotToWin && counter%plotWinSteps == 0)
		{
			int width;
			int height;
			double pixelWidth = pSim->getPixelWidth();
			double pixelHeight = pSim->getPixelHeight();

			double xOffset;
			double yOffset;
			std::string title, xLabel, yLabel, zLabel;

			for (int id = 0 ; id < PLOTWINID_MAX ; id++)
			{
				if (getPlotInfo(id, pState, dst, width, height, xOffset, yOffset, title, xLabel, yLabel, zLabel))
					pIOSys->plot3D(id, title, xLabel, yLabel, zLabel, xOffset, yOffset, pixelWidth, pixelHeight, 1, 1, 1, dst, width, height, false);
			}
		}
		
		double pct = ((double)counter/(double)totalSteps)*100.0;
		pIOSys->setPercentageFeedback(pct);

		if (pInst->isInterrupted())
		{
			if (showCurrents)
				pIOSys->writeStatusLine("");

			setErrorString("Simulation interrupted by user");
			return false;
		}
	}

	pIOSys->setPercentageFeedback(100);
	pIOSys->writeOutputLine("Simulation completed");

	if (showCurrents)
		pIOSys->writeStatusLine("");

	return true;
}

std::string CmdSimulation2DNEWRun::replaceString(const std::string &str, const std::string &findStr, const std::string &replaceStr)
{
	std::string newStr;
	std::string::size_type prevPos = 0;
	std::string::size_type newPos = 0;
	size_t findLen = findStr.length();

	while ((newPos = str.find(findStr, prevPos)) != std::string::npos)
	{
		newStr += str.substr(prevPos, newPos-prevPos);
		newStr += replaceStr;

		prevPos = newPos+findLen;
	}

	newStr += str.substr(prevPos, str.length()-prevPos);

	return newStr;
}


