#include "cmdsimulation1dnew.h"
#include "simiconductorconfig.h"
#include "simiconductorinstance.h"
#include "simulation1ddouble.h"
#include "simulation1dnr.h"
#include "simulation1dave.h"
#include "simulationstate.h"
#include "constants.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/shellcmdboolarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdchoicearg.h>
#include <shellp/iosystem.h>
#include <serut/fileserializer.h>
#include <stdio.h>
#include <cmath>
#include <algorithm>

#ifdef WIN32
#define ISFINITE _finite
#define MIN(a,b) (((a)<(b))?(a):(b))
#else
#define ISFINITE std::isfinite
#define MIN(a,b) std::min(a,b)
#endif // WIN32

using namespace shellp;

#define PLOTWINID_N					0
#define PLOTWINID_P					1
#define PLOTWINID_BG					2
#define PLOTWINID_G					3
#define PLOTWINID_RF					4
#define PLOTWINID_DN					5
#define PLOTWINID_DP					6
#define PLOTWINID_NMOB					7
#define PLOTWINID_PMOB					8
#define PLOTWINID_EPS					9
#define PLOTWINID_V					10
#define PLOTWINID_R					11
#define PLOTWINID_JX					12
#define PLOTWINID_JNX					13
#define PLOTWINID_JPX					14
#define PLOTWINID_NI					15
#define PLOTWINID_VN					16
#define PLOTWINID_VP					17
#define PLOTWINID_MAX					18

#define PLOTWINNAME_N					"n"
#define PLOTWINNAME_P					"p"
#define PLOTWINNAME_BG					"bg"
#define PLOTWINNAME_G					"g"
#define PLOTWINNAME_RF					"rf"
#define PLOTWINNAME_DN					"dn"
#define PLOTWINNAME_DP					"dp"
#define PLOTWINNAME_NMOB				"nmob"
#define PLOTWINNAME_PMOB				"pmob"
#define PLOTWINNAME_EPS					"eps"
#define PLOTWINNAME_V					"V"
#define PLOTWINNAME_R					"r"
#define PLOTWINNAME_JX					"jx"
#define PLOTWINNAME_JNX					"jnx"
#define PLOTWINNAME_JPX					"jpx"
#define PLOTWINNAME_NI					"ni"
#define PLOTWINNAME_GAMMA_N				"gamman"
#define PLOTWINNAME_GAMMA_P				"gammap"
#define PLOTWINNAME_VN					"Vn"
#define PLOTWINNAME_VP					"Vp"

bool get1DPlotInfo(int plotID, SimulationState *pSim, std::vector<double> &dstx, std::vector<double> &dsty,
		                       std::string &title, std::string &xLabel, std::string &yLabel)
{
	std::vector<double> tmp2, tmp3, dst;
	int numTotal = pSim->getNumberOfXPixels();
	int subX = 0;
	bool status = true;

	switch(plotID)
	{
	case PLOTWINID_N:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_N, dst);
		title = PLOTWINNAME_N;
		break;
	case PLOTWINID_P:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_P, dst);
		title = PLOTWINNAME_P;
		break;
	case PLOTWINID_BG:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_BG, dst);
		title = PLOTWINNAME_BG;
		break;
	case PLOTWINID_G:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_G, dst);
		title = PLOTWINNAME_G;
		break;
	case PLOTWINID_RF:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_RF, dst);
		title = PLOTWINNAME_RF;
		break;
	case PLOTWINID_DN:
		status =pSim->getGridProperty(SIMSTATE_GRIDPROP_DN, dst);
		title = PLOTWINNAME_DN;
		break;
	case PLOTWINID_DP:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_DP, dst);
		title = PLOTWINNAME_DP;
		break;
	case PLOTWINID_NMOB:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_NMOB, dst);
			title = PLOTWINNAME_NMOB;

			if (pSim->isGridPropertySet(SIMSTATE_GRIDPROP_NMOB_GAMMA))
			{
				status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_NMOB_GAMMA, tmp2);
				status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_V, tmp3);
				double dx = pSim->getPixelWidth();

				for (int x = 0 ; x < numTotal-1 ; x++)
				{
					double muInt = (dst[x]+dst[x+1])/2.0;
					double Eabs = std::abs(tmp3[x+1]-tmp3[x])/dx;
					double gammaInt = (tmp2[x]+tmp2[x+1])/2.0;

					// Danger: we're modifying the same array that we're reading from!
					dst[x] = muInt * std::exp(gammaInt*std::sqrt(Eabs));
				}
				subX = 1;
			}
		}
		break;
	case PLOTWINID_PMOB:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_PMOB, dst);
			title = PLOTWINNAME_PMOB;

			if (pSim->isGridPropertySet(SIMSTATE_GRIDPROP_PMOB_GAMMA))
			{
				status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_PMOB_GAMMA, tmp2);
				status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_V, tmp3);
				double dx = pSim->getPixelWidth();

				for (int x = 0 ; x < numTotal-1 ; x++)
				{
					double muInt = (dst[x]+dst[x+1])/2.0;
					double Eabs = std::abs(tmp3[x+1]-tmp3[x])/dx;
					double gammaInt = (tmp2[x]+tmp2[x+1])/2.0;

					// Danger: we're modifying the same array that we're reading from!
					dst[x] = muInt * std::exp(gammaInt*std::sqrt(Eabs));
				}
				subX = 1;
			}
		}
		break;
	case PLOTWINID_EPS:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_EPSREL, dst);
		title = PLOTWINNAME_EPS;
		break;
	case PLOTWINID_V:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_V, dst);
		title = PLOTWINNAME_V;
		break;
	case PLOTWINID_R:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_R, dst);
		title = PLOTWINNAME_R;
		break;
	case PLOTWINID_JX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPX, dst);
			status &= pSim->getGridProperty(SIMSTATE_GRIDPROP_JNX, tmp2);

			for (int i = 0 ; i < numTotal ; i++)
			{
				dst[i] -= tmp2[i];
				dst[i] *= CHARGE_ELECTRON;
			}

			subX = 1;
			title = PLOTWINNAME_JX;
		}
		break;
	case PLOTWINID_JNX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JNX, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= -CHARGE_ELECTRON;

			subX = 1;
			title = PLOTWINNAME_JNX;
		}
		break;
	case PLOTWINID_JPX:
		{
			status = pSim->getGridProperty(SIMSTATE_GRIDPROP_JPX, dst);

			for (int i = 0 ; i < numTotal ; i++)
				dst[i] *= CHARGE_ELECTRON;

			subX = 1;
			title = PLOTWINNAME_JPX;
		}
		break;
	case PLOTWINID_NI:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_NI, dst);
		title = PLOTWINNAME_NI;
		break;
	case PLOTWINID_VN:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_VNEXTRA, dst);
		title = PLOTWINNAME_VN;
		break;
	case PLOTWINID_VP:
		status = pSim->getGridProperty(SIMSTATE_GRIDPROP_VPEXTRA, dst);
		title = PLOTWINNAME_VP;
		break;
	default:
		return false;
	}

	if (!status)
		return false;

	double pixelWidth = pSim->getPixelWidth();
	double offset = pixelWidth*0.5*(double)subX;

	dstx.resize(numTotal-subX);
	dsty.resize(numTotal-subX);

	for (int i = 0 ; i < dsty.size() ; i++)
	{
		dsty[i] = dst[i];
		dstx[i] = (double)i * pixelWidth + offset;
	}

	xLabel = "X";
	yLabel = "Y";

	return true;
}

CmdSimulation1DNEWClear::CmdSimulation1DNEWClear(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag force clearing the simulation even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Clear the current simulation entirely.");
}

CmdSimulation1DNEWClear::~CmdSimulation1DNEWClear()
{
	delete m_pForceArg;
}

bool CmdSimulation1DNEWClear::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is1DSaved())) )
		{
			setErrorString("Simulation not saved, specify force flag to override");
			return false;
		}
	}

	pInst->setSimulationState1D(0);
	//pInst->set1DSavedFlag(true);

	return true;
}

CmdSimulation1DNEWNew::CmdSimulation1DNEWNew(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNumXArg = new ShellCmdIntArg("numx", "128");
	m_pNumXArg->setMin(3);
	m_pNumXArg->setDescription("Number of pixels in X direction.");
	addArgument(m_pNumXArg);
	
	m_pWidthArg = new ShellCmdRealArg("width");
	m_pWidthArg->setMin(0); // can't be negative
	m_pWidthArg->setDescription("Actual physical size in X direction.");
	addArgument(m_pWidthArg);

	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag forces a new grid even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Create a new 1D simulation state.");
}

CmdSimulation1DNEWNew::~CmdSimulation1DNEWNew()
{
	delete m_pNumXArg;
	delete m_pWidthArg;
	delete m_pForceArg;
}

bool CmdSimulation1DNEWNew::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is1DSaved())) )
		{
			setErrorString("Simulation state not saved, specify force flag to override");
			return false;
		}
	}

	int numX = m_pNumXArg->getValue();
	double width = m_pWidthArg->getValue();
	double pixelWidth = width/(double)(numX-1);

	SimulationState *pNewState = new SimulationState();

	if (!pNewState->init(numX, pixelWidth))
	{
		setErrorString("Unable to create new simulation state: " + pNewState->getErrorString());
		delete pNewState;
		return false;
	}

	pInst->setSimulationState1D(pNewState);
	//pInst->set1DSavedFlag(true);

	pIOSys->print("Created a new grid with %d pixels", numX);
	return true;
}

CmdSimulation1DNEWSave::CmdSimulation1DNEWSave(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to write the simulation to.");
	addArgument(m_pFileNameArg);

	setDescription("Save current simulation settings.");
}

CmdSimulation1DNEWSave::~CmdSimulation1DNEWSave()
{
	delete m_pFileNameArg;
}

bool CmdSimulation1DNEWSave::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::string fileName = m_pFileNameArg->getValue();

	if (!pSim->save(fileName))
	{
		setErrorString(pSim->getErrorString());
		return false;
	}

	pSim->setModified(false);

	pIOSys->writeOutputLine("Simulation state saved.");

	//pInst->set1DSavedFlag(true);

	return true;
}

CmdSimulation1DNEWLoad::CmdSimulation1DNEWLoad(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to read the simulation state from.");
	addArgument(m_pFileNameArg);

	m_pForceArg = new ShellCmdBoolArg("force", "no");
	m_pForceArg->setDescription("In interactive mode, enabling this flag forces loading a grid even if the previous one hasn't been saved. Ignored in non-interactive mode.");
	addArgument(m_pForceArg);

	setDescription("Load simulation state.");
}

CmdSimulation1DNEWLoad::~CmdSimulation1DNEWLoad()
{
	delete m_pFileNameArg;
	delete m_pForceArg;
}

bool CmdSimulation1DNEWLoad::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pInst->isInteractive())
	{
		bool force = m_pForceArg->getValue();

		if (! (force || (!force && pInst->is1DSaved())) )
		{
			setErrorString("Simulation not saved, specify force flag to override");
			return false;
		}
	}

	std::string fileName = m_pFileNameArg->getValue();
	std::string errStr, simType;

	SimulationState *pSim = new SimulationState();
	if (!pSim->load(fileName))
	{
		setErrorString("Couldn't load specified simulation: " + pSim->getErrorString());
		delete pSim;
		return false;
	}

	if (pSim->getDimensions() != 1)
	{
		setErrorString("Specified simulation file does not describe a one-dimensional state");
		delete pSim;
		return false;
	}

	pInst->setSimulationState1D(pSim);

	pSim->setModified(false);
	//pInst->set1DSavedFlag(true);

	int numX = pSim->getNumberOfXPixels();

	pIOSys->print("Loaded simulation state on a new grid of %d pixels wide", numX);

	return true;
}

CmdSimulation1DNEWImport::CmdSimulation1DNEWImport(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file from which the simulation state should be imported into the current state.");
	addArgument(m_pFileNameArg);

	setDescription("Import data from a specified simulation state save file into the current simulation state. Grid sizes do not need to match, values will be interpolated if needed.");
}

CmdSimulation1DNEWImport::~CmdSimulation1DNEWImport()
{
	delete m_pFileNameArg;
}

bool CmdSimulation1DNEWImport::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::string errStr, simType;
	std::string fileName = m_pFileNameArg->getValue();

	SimulationState *pImportSim = new SimulationState();
	if (!pImportSim->load(fileName))
	{
		setErrorString("Couldn't load simulation state to be imported: " + pImportSim->getErrorString());
		delete pImportSim;
		return false;
	}

	if (!pSim->import(*pImportSim))
	{
		setErrorString("Could not import: " + pSim->getErrorString());
		delete pImportSim;
		return false;
	}

	delete pImportSim;

	return true;
}


CmdSimulation1DNEWSetGridProperty::CmdSimulation1DNEWSetGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*(n*p-ni*ni)", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "base electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "base hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NI, "intrinsic electron number density", SIMSTATE_GRIDPROP_NI);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_N, "sensitivity of electron mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_NMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_P, "sensitivity of hole mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_PMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pX0 = new ShellCmdIntArg("x1");
	m_pX0->setMin(1);
	m_pX0->setDescription("Left pixel coordinate of the region.");
	addArgument(m_pX0);
	m_pX1 = new ShellCmdIntArg("x2");
	m_pX1->setMin(1);
	m_pX1->setDescription("Right pixel coordinate of the region.");
	addArgument(m_pX1);

	m_pValueArg = new ShellCmdRealArg("value");
	m_pValueArg->setDescription("Value to be set in the specified region.");
	addArgument(m_pValueArg);

	setDescription("Set the value of a property of the simulation in a specific region.");
}

CmdSimulation1DNEWSetGridProperty::~CmdSimulation1DNEWSetGridProperty()
{
	delete m_pPropertyArg;
	delete m_pX0;
	delete m_pX1;
	delete m_pValueArg;
}

bool CmdSimulation1DNEWSetGridProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	int x0 = m_pX0->getValue();
	int x1 = m_pX1->getValue();
	int width = pSim->getNumberOfXPixels();

	if (x0 > x1 || x1 > width)
	{
		setErrorString("Invalid region specified");
		return false;
	}

	x0--;

	int propID = m_pPropertyArg->getValue();
	double value = m_pValueArg->getValue();

	for (int x = x0 ; x < x1 ; x++)
	{
		if (!pSim->setGridProperty(propID, x, value))
		{
			setErrorString("Error setting grid property: " + pSim->getErrorString());
			return false;
		}
	}

	return true;
}

CmdSimulation1DNEWSetPotentialDifference::CmdSimulation1DNEWSetPotentialDifference(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPhiArg = new ShellCmdRealArg("potentialdifference");
	m_pPhiArg->setDescription("Difference in potential between right and left pixel in the simulation.");
	addArgument(m_pPhiArg);

	m_pInitGridArg = new ShellCmdBoolArg("initgrid");
	m_pInitGridArg->setDefault("no");
	m_pInitGridArg->setDescription("If this flag is set, not only the potential at the edges will be set, but also the potential on other grid points, using a linear interpolation.");
	addArgument(m_pInitGridArg);

	setDescription("Sets the potential difference to be used in the simulation. Note that this includes the possible internal potential difference of the simulated device.");
}

CmdSimulation1DNEWSetPotentialDifference::~CmdSimulation1DNEWSetPotentialDifference()
{
	delete m_pInitGridArg;
	delete m_pPhiArg;
}

bool CmdSimulation1DNEWSetPotentialDifference::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double V = m_pPhiArg->getValue();

	if (!pSim->setDoubleProperty(SIMSTATE_PROP_VDIFF, V))
	{
		setErrorString("Couldn't set property: " + pSim->getErrorString());
		return false;
	}

	int w = pSim->getNumberOfXPixels();

	if (m_pInitGridArg->getValue())
	{
		
		for (int x = 0 ; x < w ; x++)
		{
			double v = ((double)x/(double)(w-1))*V;
			
			pSim->setGridProperty(SIMSTATE_GRIDPROP_V, x, v);
		}
	}

	pSim->setGridProperty(SIMSTATE_GRIDPROP_V, 0, 0.0);
	pSim->setGridProperty(SIMSTATE_GRIDPROP_V, w-1, V);

	return true;
}

CmdSimulation1DNEWShowPotentialDifference::CmdSimulation1DNEWShowPotentialDifference(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Shows the potential difference set in the current simulation (Vright-Vleft).");
}

CmdSimulation1DNEWShowPotentialDifference::~CmdSimulation1DNEWShowPotentialDifference()
{
}

bool CmdSimulation1DNEWShowPotentialDifference::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	if (!pSim->isDoublePropertySet(SIMSTATE_PROP_VDIFF))
	{
		setErrorString("No potential difference has been set yet");
		return false;
	}

	double Vdiff = pSim->getDoubleProperty(SIMSTATE_PROP_VDIFF);

	pIOSys->print("%g V", Vdiff);

	return true;
}

CmdSimulation1DNEWPlotGridProperty::CmdSimulation1DNEWPlotGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to plot.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", PLOTWINID_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", PLOTWINID_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", PLOTWINID_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", PLOTWINID_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*(n*p-ni*ni)", PLOTWINID_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", PLOTWINID_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", PLOTWINID_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "total electron mobility", PLOTWINID_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "total hole mobility", PLOTWINID_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", PLOTWINID_EPS);
	m_pPropertyArg->registerChoice(PLOTWINNAME_R, "recombination rate", PLOTWINID_R);
	m_pPropertyArg->registerChoice(PLOTWINNAME_V, "potential", PLOTWINID_V);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JX, "current in x-direction", PLOTWINID_JX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JNX, "electron current in x-direction", PLOTWINID_JNX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_JPX, "hole current in x-direction", PLOTWINID_JPX);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NI, "intrinsic electron number density", PLOTWINID_NI);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	setDescription("Plots a property of the simulation.");
}

CmdSimulation1DNEWPlotGridProperty::~CmdSimulation1DNEWPlotGridProperty()
{
	delete m_pPropertyArg;
}

bool CmdSimulation1DNEWPlotGridProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (!pIOSys->supports2DPlot())
	{
		setErrorString("Interactive 2D plotting is not supported by the current IO system");
		return false;
	}

	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	int id = m_pPropertyArg->getValue();
	double pixelWidth = pSim->getPixelWidth();

	std::string title, xLabel, yLabel, zLabel;

	std::vector<double> dstx;
	std::vector<double> dsty;

	if (!get1DPlotInfo(id, pSim, dstx, dsty, title, xLabel, yLabel))
	{
		setErrorString("Couldn't get plot data, perhaps the specified property has not been set yet");
		return false;
	}

	if (!pIOSys->plot2D(id, title, xLabel, yLabel, dstx, dsty, true))
	{
		setErrorString("Unable to plot desired property: " + pIOSys->getErrorString());
		return false;
	}

	return true;
}

CmdSimulation1DNEWFilePlotGridProperty::CmdSimulation1DNEWFilePlotGridProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);
	m_pAppendArg = new ShellCmdBoolArg("append");
	m_pAppendArg->setDescription("Flag indicating if the plot data should be appended to an existing file.");
	addArgument(m_pAppendArg);

	setDescription("Write all data of the current simulation to a file which can be plotted by gnuplot. The columns are: " + SimulationState::getPlotFileColumns(1));
}

CmdSimulation1DNEWFilePlotGridProperty::~CmdSimulation1DNEWFilePlotGridProperty()
{
	delete m_pFileNameArg;
	delete m_pAppendArg;
}

bool CmdSimulation1DNEWFilePlotGridProperty::execute()
{	
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::string fileName = m_pFileNameArg->getValue();
	bool append = m_pAppendArg->getValue();
	FILE *pFile = 0;

	if (append)
		pFile = fopen(fileName.c_str(), "at");
	else
		pFile = fopen(fileName.c_str(), "wt");

	if (pFile == 0)
	{
		setErrorString("Unable to write to the specified file");
		return false;
	}

	pSim->writePlotData(pFile);

	fclose(pFile);
	return true;
}

CmdSimulation1DNEWSetRegionProperty::CmdSimulation1DNEWSetRegionProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*(n*p-ni*ni)", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "base electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "base hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NI, "intrinsic electron number density", SIMSTATE_GRIDPROP_NI);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_N, "sensitivity of electron mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_NMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_P, "sensitivity of hole mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_PMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pRegionArg = new ShellCmdStringArg("regionname");
	m_pRegionArg->setDescription("Name of the region in which this value should be set.");
	addArgument(m_pRegionArg);

	m_pValueArg = new ShellCmdRealArg("value");
	m_pValueArg->setDescription("Value to be set in the specified region.");
	addArgument(m_pValueArg);

	setDescription("Set the value of a property of the simulation in a specific named region.");
}

CmdSimulation1DNEWSetRegionProperty::~CmdSimulation1DNEWSetRegionProperty()
{
	delete m_pPropertyArg;
	delete m_pRegionArg;
	delete m_pValueArg;
}


bool CmdSimulation1DNEWSetRegionProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	std::string regionName = m_pRegionArg->getValue();
	std::vector<const SimiConductorRegion1D *> regionParts;

	if (!pInst->get1DRegions(regionName, regionParts))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	double value = m_pValueArg->getValue();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int numX = pSim->getNumberOfXPixels();

	std::vector<bool> regionPixels(numX);

	for (int i = 0 ; i < numX ; i++)
		regionPixels[i] = false;

	for (int i = 0 ; i < regionParts.size() ; i++)
	{
		if (!regionParts[i]->fillPixels(regionPixels, numX))
		{
			setErrorString(std::string("One of the areas of the region produced an error: ") + regionParts[i]->getErrorString());
			return false;
		}
	}

	int propID = m_pPropertyArg->getValue();

	for (int x = 0 ; x < numX ; x++) 
	{
		if (regionPixels[x])
		{
			if (!pSim->setGridProperty(propID, x, value))
			{
				setErrorString("Unable to set the grid property to a specific value: " + pSim->getErrorString());
				return false;
			}
		}
	}

	return true;
}

CmdSimulation1DNEWGetPixelProperty::CmdSimulation1DNEWGetPixelProperty(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to retrieve.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*(n*p-ni*ni)", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "base electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "base hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NI, "intrinsic electron number density", SIMSTATE_GRIDPROP_NI);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_N, "sensitivity of electron mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_NMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_P, "sensitivity of hole mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_PMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pXArg = new ShellCmdIntArg("x");
	m_pXArg->setDescription("X coordinate of the simulation pixel.");
	m_pXArg->setMin(1);
	addArgument(m_pXArg);

	m_pVarName = new ShellCmdStringArg("variable");
	m_pVarName->setDefault("*");
	m_pVarName->setDescription("If specified, the resulting pixel value will be stored in this variable.");
	addArgument(m_pVarName);

	setDescription("Retrieves the value of a simulation property at a specific pixel coordinate, and optionally stores the result in a variable.");
}

CmdSimulation1DNEWGetPixelProperty::~CmdSimulation1DNEWGetPixelProperty()
{
	delete m_pPropertyArg;
	delete m_pXArg;
	delete m_pVarName;
}

bool CmdSimulation1DNEWGetPixelProperty::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	int xPos = m_pXArg->getValue()-1;
	int width = pSim->getNumberOfXPixels();

	if (xPos < 0 || xPos >= width)
	{
		setErrorString("Invalid pixel coordinate");
		return false;
	}

	int propID = m_pPropertyArg->getValue();
	double propValue = 0;

	if (!pSim->isGridPropertySet(propID))
	{
		setErrorString("Specified property has not been set");
		return false;
	}

	propValue = pSim->getGridProperty(propID, xPos);

	std::string varName = m_pVarName->getValue();

	if (varName != std::string("*"))
	{
		if (!pInst->storeVariable(varName, propValue))
		{
			setErrorString(std::string("Unable to store result in variable: ") + pInst->getErrorString());
			return false;
		}
		
		pIOSys->print("Result: %s = %g", m_pVarName->getValue().c_str(), propValue);
	}
	else		
		pIOSys->print("Result: %g", propValue);

	return true;
}

#define setPropertyFormula(methodName)	\
{ \
	for (int x = 0 ; x < numX ; x++) \
	{\
		if (regionPixels[x]) \
		{ \
			double X = (double)x*pixelWidth; \
			int PX = x + 1; \
			double value = 0; \
			\
			pInst->defineConstant(SIMI_CONSTNAME_X, X); \
			pInst->defineConstant(SIMI_CONSTNAME_PX, PX); \
			\
			if (!pInst->evaluateExpression(formula, value)) \
			{\
				setErrorString("Could not evaluate the specified formula: " + pInst->getErrorString()); \
				return false; \
			} \
			pSim->methodName(x, value);  \
		} \
	} \
}

CmdSimulation1DNEWSetRegionPropertyFormula::CmdSimulation1DNEWSetRegionPropertyFormula(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->setDescription("Specifies the property to set.");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->registerChoice(PLOTWINNAME_BG, "fixed background number density", SIMSTATE_GRIDPROP_BG);
	m_pPropertyArg->registerChoice(PLOTWINNAME_G, "generation rate for electron-hole pairs", SIMSTATE_GRIDPROP_G);
	m_pPropertyArg->registerChoice(PLOTWINNAME_RF, "recombination rate will be rf*(n*p-ni*ni)", SIMSTATE_GRIDPROP_RF);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DN, "diffusion constant for electrons", SIMSTATE_GRIDPROP_DN);
	m_pPropertyArg->registerChoice(PLOTWINNAME_DP, "diffusion constant for holes", SIMSTATE_GRIDPROP_DP);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NMOB, "base electron mobility", SIMSTATE_GRIDPROP_NMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_PMOB, "base hole mobility", SIMSTATE_GRIDPROP_PMOB);
	m_pPropertyArg->registerChoice(PLOTWINNAME_EPS, "relative permittivity", SIMSTATE_GRIDPROP_EPSREL);
	m_pPropertyArg->registerChoice(PLOTWINNAME_NI, "intrinsic electron number density", SIMSTATE_GRIDPROP_NI);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_N, "sensitivity of electron mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_NMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_GAMMA_P, "sensitivity of hole mobility to electric field (Poole-Frenkel)", SIMSTATE_GRIDPROP_PMOB_GAMMA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VN, "extra electron potential", SIMSTATE_GRIDPROP_VNEXTRA);
	m_pPropertyArg->registerChoice(PLOTWINNAME_VP, "extra hole potential", SIMSTATE_GRIDPROP_VPEXTRA);
	addArgument(m_pPropertyArg);

	m_pRegionArg = new ShellCmdStringArg("regionname");
	m_pRegionArg->setDescription("Name of the region in which this value should be set.");
	addArgument(m_pRegionArg);

	m_pFormulaArg = new ShellCmdStringArg("formula");
	m_pFormulaArg->setDescription("The formula to apply to the pixels in the region. You can use 'X' to specify the physical coordinate of a pixel (which begins at 0), and "
			              "'PX' to specify a pixel value (which begins at 1)");
	addArgument(m_pFormulaArg);
	
	setDescription("Set the value of a property of the simulation in a specific named region according to a specific formula.");
}

CmdSimulation1DNEWSetRegionPropertyFormula::~CmdSimulation1DNEWSetRegionPropertyFormula()
{
	delete m_pPropertyArg;
	delete m_pRegionArg;
	delete m_pFormulaArg;
}

bool CmdSimulation1DNEWSetRegionPropertyFormula::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	std::string regionName = m_pRegionArg->getValue();
	std::string formula = m_pFormulaArg->getValue();
	std::vector<const SimiConductorRegion1D *> regionParts;

	if (!pInst->get1DRegions(regionName, regionParts))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int numX = pSim->getNumberOfXPixels();
	double pixelWidth = pSim->getPixelWidth();

	std::vector<bool> regionPixels(numX);

	for (int i = 0 ; i < numX ; i++)
		regionPixels[i] = false;

	for (int i = 0 ; i < regionParts.size() ; i++)
	{
		if (!regionParts[i]->fillPixels(regionPixels, numX))
		{
			setErrorString(std::string("One of the areas of the region produced an error: ") + regionParts[i]->getErrorString());
			return false;
		}
	}

	int propID = m_pPropertyArg->getValue();

	for (int x = 0 ; x < numX ; x++) 
	{
		if (regionPixels[x]) 
		{ 
			double X = (double)x*pixelWidth; 
			int PX = x + 1; 
			double value = 0; 
			
			pInst->defineConstant(SIMI_CONSTNAME_X, X); 
			pInst->defineConstant(SIMI_CONSTNAME_PX, PX); 
			
			if (!pInst->evaluateExpression(formula, value)) 
			{
				setErrorString("Could not evaluate the specified formula: " + pInst->getErrorString()); 
				return false; 
			} 

			if (!pSim->setGridProperty(propID, x, value))
			{
				setErrorString("Couldn't set specified grid property: " + pSim->getErrorString());
				return false;
			}
		}
	}

	return true;
}

CmdSimulation1DNEWInitDens::CmdSimulation1DNEWInitDens(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPropertyArg = new ShellCmdChoiceArg("property");
	m_pPropertyArg->registerChoice(PLOTWINNAME_N, "electron number density", SIMSTATE_GRIDPROP_N);
	m_pPropertyArg->registerChoice(PLOTWINNAME_P, "hole number density", SIMSTATE_GRIDPROP_P);
	m_pPropertyArg->setDescription("Specifies the property to initialize based on the boundary conditions.");
	addArgument(m_pPropertyArg);

	m_pLogArg = new ShellCmdBoolArg("logarithm");
	m_pLogArg->setDefault("yes");
	m_pLogArg->setDescription("If not set, the number densities themselves will be used, otherwise the logarithm of the number densities will be used in the linear interpolation.");
	addArgument(m_pLogArg);

	setDescription("Initialize one of the densities based on the values set at the boundaries. A linear interpolation of either the densities themselves or their logarithms is used.");
}

CmdSimulation1DNEWInitDens::~CmdSimulation1DNEWInitDens()
{
	delete m_pLogArg;
	delete m_pPropertyArg;
}

// TODO: should take the extra potentials into account
bool CmdSimulation1DNEWInitDens::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	int w = pSim->getNumberOfXPixels();
	int propID = m_pPropertyArg->getValue();
	bool useLog = m_pLogArg->getValue();
	std::vector<double> dens(w);

	if (!pSim->isGridPropertySet(propID))
	{
		setErrorString("Specified property has not been set yet, can't interpolate");
		return false;
	}

	double v0 = pSim->getGridProperty(propID, 0);
	double v1 = pSim->getGridProperty(propID, w-1);

	if (useLog)
	{
		v0 = std::log(v0);
		v1 = std::log(v1);
	}

	for (int x = 0 ; x < w ; x++)
	{
		double frac = ((double)x/(double)(w-1));
		double v = (v1-v0)*frac + v0;			

		if (useLog)
			v = std::exp(v);

		if (!pSim->setGridProperty(propID, x, v))
		{
			setErrorString("Can't set grid property to a specific value");
			return false;
		}
	}
	
	return true;
}

CmdSimulation1DNEWShowCurrent::CmdSimulation1DNEWShowCurrent(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Calculate and display the net current in the simulation at certain positions.");
}

CmdSimulation1DNEWShowCurrent::~CmdSimulation1DNEWShowCurrent()
{
}

bool CmdSimulation1DNEWShowCurrent::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	double left, right, overall, center;
	char str[256];
	std::string errStr;

	if (!pSim->calculateCurrent1D(left, center, right, overall))
	{
		setErrorString("Unable to calculate the current: " + pSim->getErrorString());
		return false;
	}
	
	left *= CHARGE_ELECTRON;
	right *= CHARGE_ELECTRON;
	overall *= CHARGE_ELECTRON;
	center *= CHARGE_ELECTRON;

	pIOSys->print("Left average:    %g A/m^2", left);
	pIOSys->print("Center average:  %g A/m^2", center);
	pIOSys->print("Right average:   %g A/m^2", right);
	pIOSys->print("Overall average: %g A/m^2", overall);

	return true;
}

CmdSimulation1DNEWSetExtendedRecombination::CmdSimulation1DNEWSetExtendedRecombination(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPairDistArg = new ShellCmdRealArg("pairdistance");
	m_pPairDistArg->setMin(0);
	m_pPairDistArg->setDescription("Electron-hole pair distance parameter.");
	addArgument(m_pPairDistArg);

	m_pGroundStateDecayRateArg = new ShellCmdRealArg("kf");
	m_pGroundStateDecayRateArg->setMin(0);
	m_pGroundStateDecayRateArg->setDescription("Rate for a bound electron-hole pair to decay to the ground state.");
	addArgument(m_pGroundStateDecayRateArg);

	setDescription("Enable the extended recombination model and set its parameters.");
}

CmdSimulation1DNEWSetExtendedRecombination::~CmdSimulation1DNEWSetExtendedRecombination()
{
	delete m_pPairDistArg;
	delete m_pGroundStateDecayRateArg;
}

bool CmdSimulation1DNEWSetExtendedRecombination::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double a = m_pPairDistArg->getValue();
	double kf = m_pGroundStateDecayRateArg->getValue();

	if (!pSim->setRecombinationModel(SimulationState::Braun))
	{
		setErrorString("Unable to set the recombination model to 'Braun'");
		return false;
	}
	if (!pSim->setDoubleProperty(SIMSTATE_PROP_PAIRDIST, a))
	{
		setErrorString("Unable to set the pair distance parameter");
		return false;
	}
	if (!pSim->setDoubleProperty(SIMSTATE_PROP_KF, kf))
	{
		setErrorString("Unable to set the pair dissociation parameter 'kf'");
		return false;
	}

	return true;
}

CmdSimulation1DNEWSetBasicRecombinationModel::CmdSimulation1DNEWSetBasicRecombinationModel(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Enable the default, basic recombination model.");
}

CmdSimulation1DNEWSetBasicRecombinationModel::~CmdSimulation1DNEWSetBasicRecombinationModel()
{
}

bool CmdSimulation1DNEWSetBasicRecombinationModel::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	if (!pSim->setRecombinationModel(SimulationState::Simple))
	{
		setErrorString("Unable to set the recombination model to 'Simple'");
		return false;
	}

	return true;
}

CmdSimulation1DNEWGetRecombinationModel::CmdSimulation1DNEWGetRecombinationModel(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Show the current recombination model and its parameters.");
}

CmdSimulation1DNEWGetRecombinationModel::~CmdSimulation1DNEWGetRecombinationModel()
{
}

bool CmdSimulation1DNEWGetRecombinationModel::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double a, kf;

	if (pSim->getRecombinationModel() == SimulationState::Simple)
		pIOSys->writeOutputLine("Using basic recombination model");
	else
	{
		if (!pSim->getDoubleProperty(SIMSTATE_PROP_PAIRDIST, a) ||
		    !pSim->getDoubleProperty(SIMSTATE_PROP_KF, kf))
		{
			setErrorString("'Braun' model selected, but necessary parameters have not yet been set");
			return false;
		}

		pIOSys->writeOutputLine("Using extended recombination model");
		pIOSys->print("  pairDist = %.5g", a);
		pIOSys->print("  kf       = %.5g", kf);
	}

	return true;
}

CmdSimulation1DNEWGetAvgDissociationProbability::CmdSimulation1DNEWGetAvgDissociationProbability(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("If the extended recombination model is used, display the current average electron-hole pair dissociation probability.");
}

CmdSimulation1DNEWGetAvgDissociationProbability::~CmdSimulation1DNEWGetAvgDissociationProbability()
{
}

bool CmdSimulation1DNEWGetAvgDissociationProbability::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	std::vector<double> P;
	double sum = 0;

	if (!pSim->getGridProperty(SIMSTATE_GRIDPROP_DISSPROB, P))
	{
		setErrorString("The dissociation probability has not been set yet");
		return false;
	}
	
	for (int i = 1 ; i < P.size()-1 ; i++)
		sum += P[i];

	sum /= (double)(P.size()-2);

	pIOSys->print("Average dissociation probability: %.5g", sum);

	return true;
}

CmdSimulation1DNEWRunDirect::CmdSimulation1DNEWRunDirect(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("1000");
	m_pMaxIterationsArg->setDescription("Maximum number of iterations to perform.");
	addArgument(m_pMaxIterationsArg);

	m_pShowCurrentsArg = new ShellCmdBoolArg("showcurrents");
	m_pShowCurrentsArg->setDefault("yes");
	m_pShowCurrentsArg->setDescription("If enabled, after each iteration the current in the x-direction will be shown.");
	addArgument(m_pShowCurrentsArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	setDescription("Use a direct search for the steady state solution (uses a Newton-Raphson search).");
}

CmdSimulation1DNEWRunDirect::~CmdSimulation1DNEWRunDirect()
{
	delete m_pMaxIterationsArg;
	delete m_pShowCurrentsArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
}

bool CmdSimulation1DNEWRunDirect::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState1D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int maxIterations = m_pMaxIterationsArg->getValue();
	bool showCurrent = m_pShowCurrentsArg->getValue();
	int iterations = 0;

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	double pixW = pState->getPixelWidth();
	double simWidth = (double)(numX-1)*pixW;
	bool useLog = m_pLogArg->getValue();

	Simulation1DNR sim;
	Simulation1DNR *pSim = &sim;

	if (!pSim->init(numX, simWidth, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	bool status = searchEquilibrium(pSim, maxIterations, showCurrent, false, -1, iterations, errStr);

	if (!status)
	{
		setErrorString(errStr);
		return false;
	}

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

bool CmdSimulation1DNEWRunDirect::searchEquilibrium(Simulation1D *pSim, int maxIterations, bool showCurrent, 
		bool errorOnMaxCount, double errorTolerance, int &iterations, std::string &errStr)
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	double sigmaPrev = 1e100;
	double sigma = sigmaPrev; 
	int count = 0;
	bool done = false;

	while (!done && count++ < maxIterations)
	{
		if (pInst->isInterrupted())
		{
			if (showCurrent)
				pIOSys->writeStatusLine("");

			errStr = "Simulation interrupted by user";
			return false;
		}

		if (!pSim->start(sigma))
		{
			errStr = std::string("Error looking for steady state solution: ") + pSim->getErrorString();
			return false;
		}

		if (showCurrent)
		{
			char str[1024];
			double top, bottom, overall, center;

			pSim->calculateXCurrent(bottom, top, overall, center);

			bottom *= CHARGE_ELECTRON;
			top *= CHARGE_ELECTRON;
			overall *= CHARGE_ELECTRON;
			center *= CHARGE_ELECTRON;

			sprintf(str, "Error: %g | Left avg: %g | Center avg: %g | Right avg: %g | Overall avg: %g", sigma, bottom, center, top, overall);
			pIOSys->writeStatusLine(str);
		}

#ifdef WIN32
		if (!_finite(sigma))
#else
		if (!std::isfinite(sigma))
#endif // WIN32
			break;

		if (sigma >= sigmaPrev)
			done = true;

		sigmaPrev = sigma;
	}

	iterations = count;

#ifdef WIN32
	if (!done || !_finite(sigmaPrev))
#else
	if (!done || !std::isfinite(sigmaPrev))
#endif // WIN32
	{
		if (!done)
		{
			pIOSys->print("Maximum count reached, current error is %g", sigma);
			if (errorOnMaxCount)
			{
				errStr = "Maximum count reached";
				return false;
			}
		}
		else
		{
			errStr = "Got NaN or inf value";
			return false;
		}
	}
	else
	{
		pIOSys->print("Convergence reached after %d iterations, current error is %g", count, sigma);
		if (errorTolerance > 0 && (sigma > errorTolerance))
		{
			errStr = "Current error exceeds the specified error tolerance";
			return false;
		}
	}
	return true;
}

CmdSimulation1DNEWSetTemperature::CmdSimulation1DNEWSetTemperature(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pTemp = new ShellCmdRealArg("T");
	m_pTemp->setDescription("Temperature for the simulation.");
	m_pTemp->setMin(0);
	m_pTemp->setMax(10000);
	addArgument(m_pTemp);

	setDescription("Sets the temperature parameter of the simulation.");
}

CmdSimulation1DNEWSetTemperature::~CmdSimulation1DNEWSetTemperature()
{
	delete m_pTemp;
}

bool CmdSimulation1DNEWSetTemperature::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();

	if (pSim == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double T = m_pTemp->getValue();

	if (!pSim->setDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("Couldn't set property: " + pSim->getErrorString());
		return false;
	}

	return true;
}

CmdSimulation1DNEWShowTemperature::CmdSimulation1DNEWShowTemperature(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Shows the potential difference set in the current simulation (Vright-Vleft).");
}

CmdSimulation1DNEWShowTemperature::~CmdSimulation1DNEWShowTemperature()
{
}

bool CmdSimulation1DNEWShowTemperature::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimulationState *pSim = pInst->getSimulationState1D();
	IOSystem *pIOSys = pInst->getIOSystem();

	if (pSim == 0)
	{
		setErrorString("No simulation has been created yet");
		return false;
	}

	if (!pSim->isDoublePropertySet(SIMSTATE_PROP_TEMP))
	{
		setErrorString("No temperature has been set yet");
		return false;
	}

	double T = pSim->getDoubleProperty(SIMSTATE_PROP_TEMP);

	pIOSys->print("%g K", T);

	return true;
}

CmdSimulation1DNEWIVCurve::CmdSimulation1DNEWIVCurve(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pVIntArg = new ShellCmdRealArg("vint");
	m_pVIntArg->setDescription("The built-in potential.");
	addArgument(m_pVIntArg);

	m_pVStartArg = new ShellCmdRealArg("vstart");
	m_pVStartArg->setDescription("Start value of the applied potential.");
	addArgument(m_pVStartArg);

	m_pVStopArg = new ShellCmdRealArg("vstop");
	m_pVStopArg->setDescription("End value of the applied potential.");
	addArgument(m_pVStopArg);

	m_pNumStepsArg = new ShellCmdIntArg("steps");
	m_pNumStepsArg->setMin(2);
	m_pNumStepsArg->setDefault("100");
	m_pNumStepsArg->setDescription("The number of points in the IV curve.");
	addArgument(m_pNumStepsArg);

	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDefault("*");
	m_pFileNameArg->setDescription("Name of the file to which the plot data should be written.");
	addArgument(m_pFileNameArg);

	m_pAppendArg = new ShellCmdBoolArg("append");
	m_pAppendArg->setDescription("Flag indicating if the plot data should be appended to an existing file.");
	m_pAppendArg->setDefault("yes");
	addArgument(m_pAppendArg);

	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("1000");
	m_pMaxIterationsArg->setDescription("Maximum number of solution iterations to perform, for each potential difference.");
	addArgument(m_pMaxIterationsArg);

	m_pErrorArg = new ShellCmdRealArg("errlimit");
	m_pErrorArg->setDefault("-1");
	m_pErrorArg->setDescription("If the error after convergence exceeds this value, the multiresolution search will stop. A negative value disables this feature.");
	addArgument(m_pErrorArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	m_pWarnError = new ShellCmdBoolArg("warningsaserrors");
	m_pWarnError->setDescription("Treat warnings as errors");
	m_pWarnError->setDefault("yes");
	addArgument(m_pWarnError);

	setDescription("Calculate an IV curve using the direct solution method. The voltage between right and left "
		       "contacts is set to 'vint-vapplied'. The Newton-Raphson based solver is used for this.");
}

CmdSimulation1DNEWIVCurve::~CmdSimulation1DNEWIVCurve()
{
	delete m_pVIntArg;
	delete m_pVStartArg;
	delete m_pVStopArg;
	delete m_pNumStepsArg;
	delete m_pFileNameArg;
	delete m_pAppendArg;
	delete m_pMaxIterationsArg;
	delete m_pErrorArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
	delete m_pWarnError;
}

bool CmdSimulation1DNEWIVCurve::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState1D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int iterations = 0;

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	double pixW = pState->getPixelWidth();
	double simWidth = (double)(numX-1)*pixW;
	bool useLog = m_pLogArg->getValue();

	Simulation1DNR sim;
	Simulation1DNR *pSim = &sim;

	if (!pSim->init(numX, simWidth, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
		
		if (m_pWarnError->getValue())
		{
			setErrorString("Treating warnings as errors!");
			return false;
		}
	}

	std::string fileName = m_pFileNameArg->getValue();
	bool append = m_pAppendArg->getValue();
	FILE *pFile = 0;
	
	if (fileName != std::string("*"))
	{
		if (append)
			pFile = fopen(fileName.c_str(), "at");
		else
			pFile = fopen(fileName.c_str(), "wt");

		if (pFile == 0)
		{
			setErrorString("Unable to write to the specified file");
			return false;
		}
	}

	double Vint = m_pVIntArg->getValue();
	double V0 = m_pVStartArg->getValue();
	double V1 = m_pVStopArg->getValue();
	double errLimit = m_pErrorArg->getValue();
	int maxIterations = m_pMaxIterationsArg->getValue();
	int steps = m_pNumStepsArg->getValue();
	double dV = (V1-V0)/(double)(steps-1);

	double V = V0;

	std::vector<double> Vvalues(steps);
	std::vector<double> Jvalues;

	for (int i = 0 ; i < steps ; i++, V += dV)
		Vvalues[i] = Vint-V;

	double curVdiff = pSim->getPotentialDifference();
	double minDiff = std::abs(curVdiff-Vvalues[0]);

	int minPos = 0;

	for (int i = 1 ; i < steps ; i++)
	{
		double d = std::abs(curVdiff - Vvalues[i]);

		if (d < minDiff)
		{
			minDiff = d;
			minPos = i;
		}
	}

	std::vector<double> Vdiffs[2];
	int targetIteration = -1;
	int targetIndex = -1;
	double targetVDiff = Vvalues[steps-1];

	for (int i = minPos ; i >= 0 ; i--)
		Vdiffs[0].push_back(Vvalues[i]);

	for (int i = minPos+1 ; i < steps ; i++)
		Vdiffs[1].push_back(Vvalues[i]);

	if (minPos+1 < steps)
	{
		targetIteration = 1;
		targetIndex = (steps-1)-(minPos+1);
	}
	else // should mean that minPos+1 = steps <=> minPos = steps-1
	{
		targetIteration = 0;
		targetIndex = 0;
	}

	Vvalues.clear();

	std::vector<double> n, p, potential;
	std::vector<double> targetN, targetP, targetV;

	if (Vdiffs[0].size() > 0 && Vdiffs[1].size() > 0)
	{
		pSim->getElectronNumberDensity(n);
		pSim->getHoleNumberDensity(p);
		pSim->getPotential(potential);
	}

	int totalCount = 0;

	pIOSys->setPercentageFeedback(0);

	for (int j = 0 ; j < 2 ; j++)
	{
		for (int i = 0 ; i < Vdiffs[j].size() ; i++, totalCount++)
		{
			double Vdiff = Vdiffs[j][i]; // Vdiff = Vint - Vapp
			double Vapp = Vint - Vdiff;

			pSim->setPotentialDifference(Vdiff);

			bool showCurrent = true;
			std::string errStr;
			int iterations = 0;

			if (!CmdSimulation1DNEWRunDirect::searchEquilibrium(pSim, maxIterations, showCurrent, true, errLimit, iterations, errStr))
			{
				setErrorString(errStr);
				finishFile(pFile, Vvalues, Jvalues);
				return false;
			}

			double top, bottom, overall, center;

			pSim->calculateXCurrent(bottom, top, overall, center);

			//double J = overall;
			double J = center;

			J *= CHARGE_ELECTRON;

			Vvalues.push_back(Vapp);
			Jvalues.push_back(J);

			pIOSys->plot2D(PLOTWINID_1DIV, PLOTWINIDNAME_1DIV, "V", "J", Vvalues, Jvalues, true);

			if (j == targetIteration && i == targetIndex)
			{
				pSim->getElectronNumberDensity(targetN);
				pSim->getHoleNumberDensity(targetP);
				pSim->getPotential(targetV);
			}

			pIOSys->setPercentageFeedback(MIN((double)(totalCount+1)/(double)steps*100.0+0.5,100.0));
		}

		if (j == 0)
		{
			if (n.size() > 0)
			{
				int numX = pSim->getNumXPixels();

				pSim->setPotentialDifference(curVdiff);

				for (int i = 0 ; i < numX ; i++)
				{
					pSim->setElectronNumberDensity(i, n[i]);
					pSim->setHoleNumberDensity(i, p[i]);
					pSim->setPotential(i, potential[i]);
				}
			}

			// Swap order

			int num = Vvalues.size();
			int num2 = num/2;

			for (int i = 0 ; i < num2 ; i++)
			{
				double tmp = Vvalues[i];
				Vvalues[i] = Vvalues[num-1-i];
				Vvalues[num-1-i] = tmp;

				tmp = Jvalues[i];
				Jvalues[i] = Jvalues[num-1-i];
				Jvalues[num-1-i] = tmp;
			}
		}
	}

//	int numX = pSim->getNumXPixels();

	pSim->setPotentialDifference(targetVDiff);

	for (int i = 0 ; i < numX ; i++)
	{
		pSim->setElectronNumberDensity(i, targetN[i]);
		pSim->setHoleNumberDensity(i, targetP[i]);
		pSim->setPotential(i, targetV[i]);
	}

	finishFile(pFile, Vvalues, Jvalues);

	double Voc = -1;
	double Jsc = -1;

	if (findVoc(Vvalues, Jvalues, Voc))
		pIOSys->print("Estimated Voc = %g", Voc);
	else
		pIOSys->writeOutputLine("Unable to estimate Voc from curve");

	if (findVoc(Jvalues, Vvalues, Jsc)) // Abusing the Voc routine to find Jsc
		pIOSys->print("Estimated Jsc = %g", Jsc);
	else
		pIOSys->writeOutputLine("Unable to estimate Jsc from curve");

	pIOSys->setPercentageFeedback(100);

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

void CmdSimulation1DNEWIVCurve::finishFile(FILE *pFile, const std::vector<double> &V, const std::vector<double> &J)
{
	if (pFile == 0)
		return;

	for (int i = 0 ; i < V.size() ; i++)
		fprintf(pFile, "%g %g\n", V[i], J[i]);

	fprintf(pFile, "\n\n");
	fclose(pFile);
}

bool CmdSimulation1DNEWIVCurve::findVoc(const std::vector<double> &V, const std::vector<double> &J, double &Voc)
{
	int i = 0;
	int signChanges = 0;
	int lastChangePos = -1;

	for (int i = 0 ; i < V.size() - 1 ; i++)
	{
		if (J[i]*J[i+1] < 0)
		{
			signChanges++;
			lastChangePos = i;
		}
		else if (J[i] == 0)
		{
			Voc = V[i];
			return true;
		}
		else if (J[i+1] == 0)
		{
			Voc = V[i+1];
			return true;
		}
	}

	if (signChanges != 1)
		return false;
	
	if (V.size() == 2) // only two points, so can only perform a linear interpolation
	{
		double frac = ( /* 0 */ - J[lastChangePos]) / (J[lastChangePos+1] - J[lastChangePos]);

		Voc = frac * (V[lastChangePos+1] - V[lastChangePos]) + V[lastChangePos];
	}
	else
	{
		double Vvalues[3];
		double Jvalues[3];
		int offset = 0;

		if (lastChangePos == 0)
			offset = 0;
		else if (lastChangePos == V.size()-2)
			offset = -1;
		else // determine which point is closest to 0
		{
			if (std::abs(J[lastChangePos]) < std::abs(J[lastChangePos+1]))
				offset = -1;
			else
				offset = 0;
		}

		for (int i = 0 ; i < 3 ; i++)
		{
			Vvalues[i] = V[lastChangePos+i+offset];
			Jvalues[i] = J[lastChangePos+i+offset];
		}

		double vScale = Jvalues[2]-Jvalues[0];
		double hScale = Vvalues[2]-Vvalues[0];

		double xMid = (Vvalues[1]-Vvalues[0])/hScale;
		double yMid = (Jvalues[1]-Jvalues[0])/vScale;

		double yZero = ( /* 0 */ - Jvalues[0])/vScale;

		double A = (yMid/xMid-1.0)/(xMid-1.0);

		double sqrtD = std::sqrt(1.0-2.0*A+A*A+4.0*A*yZero);
		double xZero_1 = 0.5*((A-1.0) + sqrtD)/A;
		double xZero_2 = 0.5*((A-1.0) - sqrtD)/A;

		if (!(ISFINITE(xZero_1) || ISFINITE(xZero_2)))
			return false;

		double xZero = -1;

		if (ISFINITE(xZero_1) && xZero_1 >= 0 && xZero_1 <= 1.0)
			xZero = xZero_1;
		else if (ISFINITE(xZero_2) && xZero_2 >= 0 && xZero_2 <= 1.0)
			xZero = xZero_2;
		else
			return false;

		Voc = (xZero * hScale) + Vvalues[0];
	}

	return true;
}

CmdSimulation1DNEWRunDirectMultiRes::CmdSimulation1DNEWRunDirectMultiRes(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("100000");
	m_pMaxIterationsArg->setDescription("Maximum number of iterations to perform.");
	addArgument(m_pMaxIterationsArg);

	m_pShowCurrentsArg = new ShellCmdBoolArg("showcurrents");
	m_pShowCurrentsArg->setDefault("yes");
	m_pShowCurrentsArg->setDescription("If enabled, after each iteration the current in the x-direction will be shown.");
	addArgument(m_pShowCurrentsArg);

	m_pMinXArg = new ShellCmdIntArg("minx");
	m_pMinXArg->setMin(3);
	m_pMinXArg->setDefault("8");
	m_pMinXArg->setDescription("Minimal grid size in the x-direction");
	addArgument(m_pMinXArg);

	m_pStepsArg = new ShellCmdIntArg("steps");
	m_pStepsArg->setMin(2);
	m_pStepsArg->setDefault("5");
	m_pStepsArg->setDescription("Number of multiresolution steps.");
	addArgument(m_pStepsArg);

	m_pErrorArg = new ShellCmdRealArg("errlimit");
	m_pErrorArg->setDefault("-1");
	m_pErrorArg->setDescription("If the error after convergence exceeds this value, the multiresolution search will stop. A negative value disables this feature.");
	addArgument(m_pErrorArg);

//	m_pFileNameArg = new ShellCmdStringArg("filename");
//	m_pFileNameArg->setDefault("*");
//	m_pFileNameArg->setDescription("If specified, the grids obtained during the search will be written to this file. First in the file will be "
//			               "the initial situation of the target grid, then, for each resolution the begin and end situation is written.");
//	addArgument(m_pFileNameArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);


	setDescription("Instead of using a time step based simulation, use a direct search for the steady state solution. This version uses a multi-resolution approach: first a solution is sought to a low-resolution approximation, which is used as starting situation in a higher resolution version. This process is continued until the true resolution is reached.");
}

CmdSimulation1DNEWRunDirectMultiRes::~CmdSimulation1DNEWRunDirectMultiRes()
{
	delete m_pMaxIterationsArg;
	delete m_pShowCurrentsArg;
	delete m_pMinXArg;
	delete m_pStepsArg;
	delete m_pErrorArg;
//	delete m_pFileNameArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
}

bool CmdSimulation1DNEWRunDirectMultiRes::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState1D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int iterations = 0;

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	double pixW = pState->getPixelWidth();
	double simWidth = (double)(numX-1)*pixW;
	bool useLog = m_pLogArg->getValue();

	Simulation1DNR sim; 
	Simulation1DNR *pSim = &sim;

	if (!pSim->init(numX, simWidth, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	double errLimit = m_pErrorArg->getValue();

	int iterationsLeft = m_pMaxIterationsArg->getValue();
	bool showCurrent = m_pShowCurrentsArg->getValue();

	int minX = m_pMinXArg->getValue();
	int steps = m_pStepsArg->getValue();

	int maxX = pSim->getNumXPixels();
	double realWidth = pSim->getPixelWidth() * (double)(maxX - 1);

	if (minX > maxX)
	{
		setErrorString("Minimum dimensions exceed the current grid size");
		return false;
	}

	Simulation1DNR *pPrevSim = createSimAndImport(minX, realWidth, T, diffScale, densScale, useLog, *pState);
	if (pPrevSim == 0)
	{
		return false;
	}	

	pIOSys->setPercentageFeedback(0);

	for (int i = 0 ; i < steps-1 ; i++)
	{
		int iterations = 0;

		if (!CmdSimulation1DNEWRunDirect::searchEquilibrium(pPrevSim, iterationsLeft, showCurrent, true, errLimit, iterations, errStr))
		{
			delete pPrevSim;
			setErrorString(errStr);
			return false;
		}
		
		iterationsLeft -= iterations;
		
		int newX = (int)((double)(maxX-minX)*(double)(i+1)/(double)(steps-1) + (double)minX + 0.5);

		if (newX > maxX)
			newX = maxX;

		SimulationState newState;

		if (!transferState(pPrevSim, newX, realWidth, *pState, newState))
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		Simulation1DNR *pNewSim = createSimAndImport(newX, realWidth, T, diffScale, densScale, useLog, newState);

		if (pNewSim == 0)
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		delete pPrevSim;
		pPrevSim = pNewSim;

		pIOSys->setPercentageFeedback(MIN((double)(i+1)/(double)steps*100.0+0.5,100.0));
	}

	{
		SimulationState newState;

		if (!transferState(pPrevSim, maxX, realWidth, *pState, newState))
		{
			// error has already been set
			delete pPrevSim;
			return false;
		}

		delete pPrevSim;

		std::vector<std::string> warnings;

		if (!pSim->setState(newState, warnings))
		{
			setErrorString("Unable to store full resolution state for final simulation: " + pSim->getErrorString());
			return false;
		}

		int iterations;

		if (!CmdSimulation1DNEWRunDirect::searchEquilibrium(pSim, iterationsLeft, showCurrent, true, errLimit, iterations, errStr))
		{
			setErrorString(errStr);
			return false;
		}

		pIOSys->setPercentageFeedback(100);
	}

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

bool CmdSimulation1DNEWRunDirectMultiRes::transferState(const Simulation1DNR *pPrevSim, int newX, 
		                                        double realWidth, const SimulationState &highResState,
							SimulationState &newState)
{
	bool gridImportFlags[SIMSTATE_GRIDPROP_MAX];
	bool doubleImportFlags[SIMSTATE_PROP_MAX];

	for (int i = 0 ; i < SIMSTATE_GRIDPROP_MAX ; i++)
		gridImportFlags[i] = false;
	for (int i = 0 ; i < SIMSTATE_PROP_MAX ; i++)
		doubleImportFlags[i] = false;

	gridImportFlags[SIMSTATE_GRIDPROP_N] = true;
	gridImportFlags[SIMSTATE_GRIDPROP_P] = true;
	gridImportFlags[SIMSTATE_GRIDPROP_V] = true;

	SimulationState resultState;
	int resultNumX = pPrevSim->getNumXPixels();
	double resultPixWidth = realWidth/(double)(resultNumX-1);

	if (!resultState.init(resultNumX, resultPixWidth))
	{
		setErrorString("Couldn't initialize intermediate state: " + resultState.getErrorString());
		return false;
	}

	if (!pPrevSim->storeState(resultState))
	{
		setErrorString("Couldn't store intermediate state: " + pPrevSim->getErrorString());
		return false;
	}

	int newPixWidth = realWidth/(double)(newX-1);

	if (!newState.init(newX, newPixWidth))
	{
		setErrorString("Couldn't init intermediate new state: " + newState.getErrorString());
		return false;
	}

	if (!newState.import(highResState))
	{
		setErrorString("Unable to import high resolution settings in new intermediate state: " + newState.getErrorString());
		return false;
	}

	if (!newState.import(resultState, gridImportFlags, doubleImportFlags))
	{
		setErrorString("Unable to import low resolution results for n, p and V in new intermediate state: " + newState.getErrorString());
		return false;
	}

	double V = newState.getDoubleProperty(SIMSTATE_PROP_VDIFF);

	newState.setGridProperty(SIMSTATE_GRIDPROP_V, 0, 0.0);
	newState.setGridProperty(SIMSTATE_GRIDPROP_V, newX-1, V);

	return true;
}

Simulation1DNR *CmdSimulation1DNEWRunDirectMultiRes::createSimAndImport(int numX, double realWidth, double T,
									double diffScale, double densScale,
									bool useLog, SimulationState &state)
{
	Simulation1DNR *pSim = new Simulation1DNR();

	if (!pSim->init(numX, realWidth, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		delete pSim;
		return 0;
	}

	std::vector<std::string> warnings;

	if (state.getNumberOfXPixels() == pSim->getNumXPixels())
	{
		if (!pSim->setState(state, warnings))
		{
			setErrorString("Unable to set the state: " + pSim->getErrorString());
			delete pSim;
			return 0;
		}
	}
	else
	{
		double pixWidth = realWidth/(double)(numX-1);

		SimulationState tmpState;

		if (!tmpState.init(numX, pixWidth))
		{
			setErrorString("Unable to initialize temporary state: " + tmpState.getErrorString());
			delete pSim;
			return 0;
		}

		if (!tmpState.import(state))
		{
			setErrorString("Unable to import state into temporary state: " + tmpState.getErrorString());
			delete pSim;
			return 0;
		}

		if (!pSim->setState(tmpState, warnings))
		{
			setErrorString("Unable to set the state: " + pSim->getErrorString());
			delete pSim;
			return 0;
		}
	}
	return pSim;
}


CmdSimulation1DNEWRun::CmdSimulation1DNEWRun(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pStepsArg = new ShellCmdIntArg("steps");
	m_pStepsArg->setMin(1);
	m_pStepsArg->setDescription("Number of time steps to advance the simulation with.");
	addArgument(m_pStepsArg);

	m_pDtArg = new ShellCmdRealArg("dt");
	m_pDtArg->setDescription("Size of the time step in numerrically solving the differential equation.");
	addArgument(m_pDtArg);

	m_pShowCurrentsStepsArg = new ShellCmdIntArg("xcursteps");
	m_pShowCurrentsStepsArg->setDefault("0");
	m_pShowCurrentsStepsArg->setDescription("The number of steps after which each time the x-current will be shown. A value smaller than 1 disables the feature.");
	addArgument(m_pShowCurrentsStepsArg);

	m_pSaveIntNameArg = new ShellCmdStringArg("tmpsavefile");
	m_pSaveIntNameArg->setDefault("*");
	m_pSaveIntNameArg->setDescription("Name of the file to which temporary results should be saved. The character '%' will be expanded to the current simulation step number.");
	addArgument(m_pSaveIntNameArg);

	m_pSaveIntStepsArg = new ShellCmdIntArg("tmpsavesteps");
	m_pSaveIntStepsArg->setDefault("0");
	m_pSaveIntStepsArg->setDescription("Each multiple of this number of steps, the current simulation state will be saved to a file. A value smaller than 1 disables this feature.");
	addArgument(m_pSaveIntStepsArg);

	m_pPlotFileNameArg = new ShellCmdStringArg("plotfile");
	m_pPlotFileNameArg->setDefault("*");
	m_pPlotFileNameArg->setDescription("Name of the file to which the simulation plot data should be written each time a number of steps has passed. If the file exists, data will be appended. The character '%' will be expanded to the current simulation step number.");
	addArgument(m_pPlotFileNameArg);

	m_pPlotFileStepsArg = new ShellCmdIntArg("plotfilesteps");
	m_pPlotFileStepsArg->setDefault("0");
	m_pPlotFileStepsArg->setDescription("Each time this number of steps has passed, the current simulation plot data will be written to the specified file. A value smaller than 1 disables the feature.");
	addArgument(m_pPlotFileStepsArg);

	m_pPlotWindowStepsArg = new ShellCmdIntArg("plotwinsteps");
	m_pPlotWindowStepsArg->setDefault("0");
	m_pPlotWindowStepsArg->setDescription("Each time this number of steps has passed, the current simulation data will be plotted on screen. A value smaller than 1 disables the feature.");
	addArgument(m_pPlotWindowStepsArg);

	m_pInverseMatrixSolver = new ShellCmdBoolArg("inversematrixsolver");
	m_pInverseMatrixSolver->setDefault("yes");
	m_pInverseMatrixSolver->setDescription("Use the inverse matrix based potential solver (should be somewhat more accurate which can lead to faster convergence to steady state).");
	addArgument(m_pInverseMatrixSolver);

	setDescription("Advance the simulation for a number of time steps, starting from the current state.");
}

CmdSimulation1DNEWRun::~CmdSimulation1DNEWRun()
{
	delete m_pStepsArg;
	delete m_pDtArg;
	delete m_pShowCurrentsStepsArg;
	delete m_pSaveIntNameArg;
	delete m_pSaveIntStepsArg;
	delete m_pPlotFileNameArg;
	delete m_pPlotFileStepsArg;
	delete m_pPlotWindowStepsArg;
	delete m_pInverseMatrixSolver;
}

bool CmdSimulation1DNEWRun::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState1D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	double pixW = pState->getPixelWidth();
	double realWidth = (double)(numX-1) * pixW;

	int steps = 1;
	int lowWidth = numX;
	bool done = false;

	while (lowWidth > 4 && !done)
	{
		if (lowWidth%2 == 0)
		{
			steps++;
			lowWidth /= 2;
		}
		else
			done = true;
	}

	pIOSys->print("Using lowest resolution of %d pixels", lowWidth);

	Simulation1DDouble sim;
	Simulation1DDouble *pSim = &sim;
	int newNumX = 0;

	if (!pSim->init(lowWidth, realWidth, steps, &newNumX, T))
	{
		setErrorString("Unable to initialize timestep based simulation: " + pSim->getErrorString());
		return false;
	}

	if (newNumX != numX)
	{
		setErrorString("Internal error in calculation of scale steps");
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	int totalSteps = m_pStepsArg->getValue();
	double dt = m_pDtArg->getValue();

	std::vector<int> stepVector;

	bool showCurrents = false;
	int showCurrentSteps = m_pShowCurrentsStepsArg->getValue();
	if (showCurrentSteps > 0)
	{
		stepVector.push_back(showCurrentSteps);
		showCurrents = true;
	}

	bool saveSimulation = false;
	std::string saveName = m_pSaveIntNameArg->getValue();
	int saveSteps = m_pSaveIntStepsArg->getValue();
	if (saveName != std::string("*") && saveSteps > 0)
	{
		stepVector.push_back(saveSteps);
		saveSimulation = true;
	}

	bool plotToFile = false;
	std::string plotFileName = m_pPlotFileNameArg->getValue();
	int plotFileSteps = m_pPlotFileStepsArg->getValue();
	if (plotFileName != std::string("*") && plotFileSteps > 0)
	{
		stepVector.push_back(plotFileSteps);
		plotToFile = true;
	}

	bool plotToWin = false;
	int plotWinSteps = m_pPlotWindowStepsArg->getValue();
	if (plotWinSteps > 0)
	{
		stepVector.push_back(plotWinSteps);
		plotToWin = true;
	}

	pIOSys->setPercentageFeedback(0);

	int stepsLeft = totalSteps;
	int counter = 0;
	std::vector<int> maxSteps = stepVector;

	std::vector<double> dst;

	//pInst->set1DSavedFlag(false);

	bool inverseSolver = m_pInverseMatrixSolver->getValue();

	while (stepsLeft > 0)
	{
		int steps = stepsLeft;

		for (int i = 0 ; i < stepVector.size() ; i++)
		{
			if (stepVector[i] < steps)
				steps = stepVector[i];
		}

		if (!pSim->start(5, dt, steps, -1, inverseSolver))
		{
			setErrorString(std::string("Error running simulation: ") + pSim->getErrorString());
			return false;
		}

		// TODO: should we do this?
		// Problem is that otherwise the plot routine, file output routine, ... don't have the current values
		// Maybe we should use a copy?

		if (!pSim->storeState(*pState))
		{
			setErrorString("Unable to store the simulation results into the simulation state");
			return false;
		}

		stepsLeft -= steps;
		counter += steps;

		for (int i = 0 ; i < stepVector.size() ; i++)
		{
			stepVector[i] -= steps;
			if (stepVector[i] <= 0)
				stepVector[i] = maxSteps[i];
		}

		if (showCurrents && counter%showCurrentSteps == 0)
		{
			char str[1024];
			double top, bottom, overall, center;

			pSim->calculateXCurrent(bottom, top, overall, center);

			bottom *= CHARGE_ELECTRON;
			top *= CHARGE_ELECTRON;
			overall *= CHARGE_ELECTRON;
			center *= CHARGE_ELECTRON;

			sprintf(str, "Left avg: %g | Center avg: %g | Right avg: %g | Overall avg: %g", bottom, center, top, overall);
			pIOSys->writeStatusLine(str);
		}
		if (saveSimulation && counter%saveSteps == 0)
		{
			char numStr[256];

			// snprintf
			sprintf(numStr, "%d", counter);

			std::string fileName = replaceString(saveName, "%", numStr);

			if (!pState->save(fileName))
				pIOSys->writeOutputLine(std::string("Warning: couldn't save state to file ") + fileName + std::string(": ") + pSim->getErrorString());
		}
		if (plotToFile && counter%plotFileSteps == 0)
		{
			char numStr[256];

			// snprintf
			sprintf(numStr, "%d", counter);

			std::string fileName = replaceString(plotFileName, "%", numStr);

			FILE *pFile = fopen(fileName.c_str(), "at");

			if (pFile == 0)
				pIOSys->writeOutputLine(std::string("Warning: couldn't open plot file ") + fileName);
			else
			{
				pState->writePlotData(pFile);
				fclose(pFile);
			}
		}
		if (plotToWin && counter%plotWinSteps == 0)
		{
			int width;
			double pixelWidth = pSim->getPixelWidth();

			std::vector<double> dstx, dsty;
			std::string title, xLabel, yLabel;

			for (int id = 0 ; id < PLOTWINID_MAX ; id++)
			{
				if (get1DPlotInfo(id, pState, dstx, dsty, title, xLabel, yLabel))
					pIOSys->plot2D(id, title, xLabel, yLabel, dstx, dsty, false);
			}
		}
		
		double pct = ((double)counter/(double)totalSteps)*100.0;
		pIOSys->setPercentageFeedback(pct);

		if (pInst->isInterrupted())
		{
			if (showCurrents)
				pIOSys->writeStatusLine("");

			setErrorString("Simulation interrupted by user");
			return false;
		}
	}

	pIOSys->setPercentageFeedback(100);
	pIOSys->writeOutputLine("Simulation completed");

	if (showCurrents)
		pIOSys->writeStatusLine("");

	return true;
}

std::string CmdSimulation1DNEWRun::replaceString(const std::string &str, const std::string &findStr, const std::string &replaceStr)
{
	std::string newStr;
	std::string::size_type prevPos = 0;
	std::string::size_type newPos = 0;
	size_t findLen = findStr.length();

	while ((newPos = str.find(findStr, prevPos)) != std::string::npos)
	{
		newStr += str.substr(prevPos, newPos-prevPos);
		newStr += replaceStr;

		prevPos = newPos+findLen;
	}

	newStr += str.substr(prevPos, str.length()-prevPos);

	return newStr;
}

#ifdef SIMICONDUCTOR_CONFIG_AVE

CmdSimulation1DNEWRunDirectAVE::CmdSimulation1DNEWRunDirectAVE(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pMaxIterationsArg = new ShellCmdIntArg("maxit");
	m_pMaxIterationsArg->setMin(1);
	m_pMaxIterationsArg->setDefault("1000");
	m_pMaxIterationsArg->setDescription("Maximum number of iterations to perform.");
	addArgument(m_pMaxIterationsArg);

	m_pShowCurrentsArg = new ShellCmdBoolArg("showcurrents");
	m_pShowCurrentsArg->setDefault("yes");
	m_pShowCurrentsArg->setDescription("If enabled, after each iteration the current in the x-direction will be shown.");
	addArgument(m_pShowCurrentsArg);

	m_pDensArg = new ShellCmdRealArg("densscale");
	m_pDensArg->setDescription("Density scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDensArg->setDefault("-1");
	addArgument(m_pDensArg);

	m_pDiffArg = new ShellCmdRealArg("diffscale");
	m_pDiffArg->setDescription("Diffusion scale to be used in the NR based simulation. A negative value causes the scale to be estimated automatically.");
	m_pDiffArg->setDefault("-1");
	addArgument(m_pDiffArg);

	m_pLogArg = new ShellCmdBoolArg("uselog");
	m_pLogArg->setDescription("Use a logarithmic scale for the electron and hole densities (works better in some cases).");
	m_pLogArg->setDefault("no");
	addArgument(m_pLogArg);

	setDescription("Use a direct search for the steady state solution (uses an accellerated gradient based search).");
}

CmdSimulation1DNEWRunDirectAVE::~CmdSimulation1DNEWRunDirectAVE()
{
	delete m_pMaxIterationsArg;
	delete m_pShowCurrentsArg;
	delete m_pDensArg;
	delete m_pDiffArg;
	delete m_pLogArg;
}

bool CmdSimulation1DNEWRunDirectAVE::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	SimulationState *pState = pInst->getSimulationState1D();

	if (pState == 0)
	{
		setErrorString("No simulation state has been created yet");
		return false;
	}

	int maxIterations = m_pMaxIterationsArg->getValue();
	bool showCurrent = m_pShowCurrentsArg->getValue();
	int iterations = 0;

	double densScale = m_pDensArg->getValue();
	double diffScale = m_pDiffArg->getValue();

	std::string errStr;

	double T = 0;

	if (!pState->getDoubleProperty(SIMSTATE_PROP_TEMP, T))
	{
		setErrorString("The temperature has not been set for this simulation.");
		return false;
	}

	int numX = pState->getNumberOfXPixels();
	double pixW = pState->getPixelWidth();
	double simWidth = (double)(numX-1)*pixW;
	bool useLog = m_pLogArg->getValue();

	Simulation1DAVE sim;
	Simulation1DAVE *pSim = &sim;

	if (!pSim->init(numX, simWidth, T, diffScale, densScale, useLog))
	{
		setErrorString("Unable to initialize the simulation: " + pSim->getErrorString());
		return false;
	}

	std::vector<std::string> warnings;

	if (!pSim->setState(*pState, warnings))
	{
		setErrorString("Unable to initialize the simulation from the current state: " + pSim->getErrorString());
		return false;
	}

	if (!warnings.empty())
	{
		for (int i = 0 ; i < warnings.size() ; i++)
			pIOSys->print("WARNING: %s", warnings[i].c_str());
	}

	bool status = searchEquilibrium(pSim, maxIterations, showCurrent, false, -1, iterations, errStr);

	if (!status)
	{
		setErrorString(errStr);
		return false;
	}

	if (!pSim->storeState(*pState))
	{
		setErrorString("Unable to store the simulation results into the simulation state");
		return false;
	}

	return true;
}

bool CmdSimulation1DNEWRunDirectAVE::searchEquilibrium(Simulation1D *pSim, int maxIterations, bool showCurrent, 
		bool errorOnMaxCount, double errorTolerance, int &iterations, std::string &errStr)
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	double sigmaPrev = 1e100;
	double sigma = sigmaPrev; 
	int count = 0;
	bool done = false;

	while (!done && count++ < maxIterations)
	{
		if (pInst->isInterrupted())
		{
			if (showCurrent)
				pIOSys->writeStatusLine("");

			errStr = "Simulation interrupted by user";
			return false;
		}

		if (!pSim->start(sigma))
		{
			errStr = std::string("Error looking for steady state solution: ") + pSim->getErrorString();
			return false;
		}

		if (showCurrent)
		{
			char str[1024];
			double top, bottom, overall, center;

			pSim->calculateXCurrent(bottom, top, overall, center);

			bottom *= CHARGE_ELECTRON;
			top *= CHARGE_ELECTRON;
			overall *= CHARGE_ELECTRON;
			center *= CHARGE_ELECTRON;

			sprintf(str, "Error: %g | Left avg: %g | Center avg: %g | Right avg: %g | Overall avg: %g", sigma, bottom, center, top, overall);
			pIOSys->writeStatusLine(str);
		}

#ifdef WIN32
		if (!_finite(sigma))
#else
		if (!std::isfinite(sigma))
#endif // WIN32
			break;

		if (sigma >= sigmaPrev)
			done = true;

		sigmaPrev = sigma;
	}

	iterations = count;

#ifdef WIN32
	if (!done || !_finite(sigmaPrev))
#else
	if (!done || !std::isfinite(sigmaPrev))
#endif // WIN32
	{
		if (!done)
		{
			pIOSys->print("Maximum count reached, current error is %g", sigma);
			if (errorOnMaxCount)
			{
				errStr = "Maximum count reached";
				return false;
			}
		}
		else
		{
			errStr = "Got NaN or inf value";
			return false;
		}
	}
	else
	{
		pIOSys->print("Convergence reached after %d iterations, current error is %g", count, sigma);
		if (errorTolerance > 0 && (sigma > errorTolerance))
		{
			errStr = "Current error exceeds the specified error tolerance";
			return false;
		}
	}
	return true;
}

#endif // SIMICONDUCTOR_CONFIG_AVE

