#include "cmdregion2d.h"
#include "simiconductorinstance.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/shellcmdboolarg.h>
#include <shellp/iosystem.h>
#include <stdio.h>
#include <string.h>

using namespace shellp;

CmdRegion2DNew::CmdRegion2DNew(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the new region.");
	addArgument(m_pNameArg);

	setDescription("Create a new (but still empty) region with a specified name.");
}

CmdRegion2DNew::~CmdRegion2DNew()
{
	delete m_pNameArg;
}

bool CmdRegion2DNew::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();

	if (!pInst->createNew2DRegionName(name))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion2DAddRect::CmdRegion2DAddRect(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the region to which the specified area should be added.");
	addArgument(m_pNameArg);

	m_pX1Arg = new ShellCmdIntArg("x1");
	m_pX1Arg->setMin(1);
	m_pX1Arg->setDescription("Left pixel coordinate of the area.");
	addArgument(m_pX1Arg);
	m_pY1Arg = new ShellCmdIntArg("y1");
	m_pY1Arg->setMin(1);
	m_pY1Arg->setDescription("Bottom pixel coordinate of the area.");
	addArgument(m_pY1Arg);
	m_pX2Arg = new ShellCmdIntArg("x2");
	m_pX2Arg->setMin(1);
	m_pX2Arg->setDescription("Right pixel coordinate of the area.");
	addArgument(m_pX2Arg);
	m_pY2Arg = new ShellCmdIntArg("y2");
	m_pY2Arg->setMin(1);
	m_pY2Arg->setDescription("Top pixel coordinate of the area.");
	addArgument(m_pY2Arg);

	setDescription("Add an area to the region with the specified name.");
}

CmdRegion2DAddRect::~CmdRegion2DAddRect()
{
	delete m_pNameArg;
	delete m_pX1Arg;
	delete m_pY1Arg;
	delete m_pX2Arg;
	delete m_pY2Arg;
}

bool CmdRegion2DAddRect::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();
	int x1 = m_pX1Arg->getValue();
	int x2 = m_pX2Arg->getValue();
	int y1 = m_pY1Arg->getValue();
	int y2 = m_pY2Arg->getValue();

	if (x1 > x2 || y1 > y2)
	{
		setErrorString("Bad coordinate order");
		return false;
	}

	SimiConductorRectRegion r(x1, y1, x2, y2);

	if (!pInst->append2DRegion(name, r))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion2DAddImg::CmdRegion2DAddImg(const std::string &cmdName) : ShellCommand(cmdName)
{ 
	m_pNameArg = new ShellCmdStringArg("regionname");
	m_pNameArg->setDescription("Name of the region to which the specified area should be added.");
	addArgument(m_pNameArg);

	m_pFileNameArg = new ShellCmdStringArg("filename");
	m_pFileNameArg->setDescription("Name of the image file (e.g. a PNG file) used to define this sub-region. All non-black pixels will be considered a part of this sub-region.");
	addArgument(m_pFileNameArg);

	m_pXScaleArg = new ShellCmdRealArg("xscale");
	m_pXScaleArg->setDescription("Factor by which the sub-area corresponding to the image file should be scaled in the X-direction.");
	m_pXScaleArg->setMin(0);
	m_pXScaleArg->setDefault("1");
	addArgument(m_pXScaleArg);

	m_pYScaleArg = new ShellCmdRealArg("yscale");
	m_pYScaleArg->setDescription("Factor by which the sub-area corresponding to the image file should be scaled in the Y-direction.");
	m_pYScaleArg->setMin(0);
	m_pYScaleArg->setDefault("1");
	addArgument(m_pYScaleArg);

	m_pXOffsetArg = new ShellCmdIntArg("xoffset");
	m_pXOffsetArg->setDescription("The sub-area loaded from the image and scaled by the previous factors, will be offset by this amount in the X-direction.");
	m_pXOffsetArg->setDefault("0");
	addArgument(m_pXOffsetArg);

	m_pYOffsetArg = new ShellCmdIntArg("yoffset");
	m_pYOffsetArg->setDescription("The sub-area loaded from the image and scaled by the previous factors, will be offset by this amount in the Y-direction.");
	m_pYOffsetArg->setDefault("0");
	addArgument(m_pYOffsetArg);
}

CmdRegion2DAddImg::~CmdRegion2DAddImg()
{
	delete m_pNameArg;
	delete m_pFileNameArg;
	delete m_pXScaleArg;
	delete m_pYScaleArg;
	delete m_pXOffsetArg;
	delete m_pYOffsetArg;
}

bool CmdRegion2DAddImg::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	SimiConductorMapRegion mapRegion;

	std::string fileName = m_pFileNameArg->getValue();
	double xScale = m_pXScaleArg->getValue();
	double yScale = m_pYScaleArg->getValue();
	int xOffset = m_pXOffsetArg->getValue();
	int yOffset = m_pYOffsetArg->getValue();

	if (!mapRegion.import(fileName, xScale, yScale, xOffset, yOffset))
	{
		setErrorString(std::string("Couldn't create map from image file: ") + mapRegion.getErrorString());
		return false;
	}

	std::string regionName = m_pNameArg->getValue();

	if (!pInst->append2DRegion(regionName, mapRegion))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion2DDelete::CmdRegion2DDelete(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the region to delete.");
	addArgument(m_pNameArg);

	setDescription("Delete a specific region name.");
}

CmdRegion2DDelete::~CmdRegion2DDelete()
{
	delete m_pNameArg;
}

bool CmdRegion2DDelete::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();

	if (!pInst->delete2DRegion(name))
	{
		setErrorString(std::string("Can't delete region '") + name + std::string("': ") + pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion2DClearAll::CmdRegion2DClearAll(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Clear all currently defined region names.");
}

CmdRegion2DClearAll::~CmdRegion2DClearAll()
{
}

bool CmdRegion2DClearAll::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();

	pInst->clear2DRegions();

	return true;
}

CmdRegion2DList::CmdRegion2DList(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPrintDescArg = new ShellCmdBoolArg("printdescription");
	m_pPrintDescArg->setDefault("no");
	m_pPrintDescArg->setDescription("If set to true, not only the region names but also a description (if available) will be shown.");
	addArgument(m_pPrintDescArg);

	setDescription("List currently defined region names and their descriptions.");
}

CmdRegion2DList::~CmdRegion2DList()
{
	delete m_pPrintDescArg;
}

bool CmdRegion2DList::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	std::vector<std::string> names;
	pInst->get2DRegionNames(names);

	if (names.empty())
	{
		pIOSys->writeOutputLine("There are currently no defined regions");
		return true;
	}

	bool printDesc = m_pPrintDescArg->getValue();

	pIOSys->writeOutputLine("Currently defined regions: ");

	if (!printDesc)
	{
		for (int i = 0 ; i < names.size() ; i++)
		{
			pIOSys->print("  %s", names[i].c_str());
		}
	}
	else
	{
		for (int i = 0 ; i < names.size() ; i++)
		{
			std::vector<const SimiConductorRegion2D *> subRegions;

			pIOSys->print("  Name:        %s", names[i].c_str());

			if (!pInst->get2DRegions(names[i], subRegions))
			{
				setErrorString(std::string("Internal error (invalid region name): ") + pInst->getErrorString());
				return false;
			}

			if (subRegions.empty())
				pIOSys->print("  Description: empty");
			else
			{
				pIOSys->print("  Description:");
				char str[1024];
				int len;

				sprintf(str, "%d", subRegions.size());
				len = strlen(str);

				sprintf(str, "    %%%dd. %%s", len);

				for (int j = 0 ; j < subRegions.size() ; j++)
				{
					const SimiConductorRegion2D *pReg = subRegions[j];

					std::string desc = pReg->getDescription();

					pIOSys->print(str, j+1, pReg->getDescription().c_str());
				}

				pIOSys->writeOutputLine("");
			}
		}
	}

	return true;
}
