#include "cmdregion1d.h"
#include "simiconductorinstance.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/shellcmdboolarg.h>
#include <shellp/iosystem.h>
#include <stdio.h>
#include <string.h>

using namespace shellp;

CmdRegion1DNew::CmdRegion1DNew(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the new region.");
	addArgument(m_pNameArg);

	setDescription("Create a new (but still empty) region with a specified name.");
}

CmdRegion1DNew::~CmdRegion1DNew()
{
	delete m_pNameArg;
}

bool CmdRegion1DNew::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();

	if (!pInst->createNew1DRegionName(name))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion1DAddLine::CmdRegion1DAddLine(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the region to which the specified area should be added.");
	addArgument(m_pNameArg);

	m_pX1Arg = new ShellCmdIntArg("x1");
	m_pX1Arg->setMin(1);
	m_pX1Arg->setDescription("Left pixel coordinate of the area.");
	addArgument(m_pX1Arg);
	m_pX2Arg = new ShellCmdIntArg("x2");
	m_pX2Arg->setMin(1);
	m_pX2Arg->setDescription("Right pixel coordinate of the area.");
	addArgument(m_pX2Arg);

	setDescription("Add an area to the region with the specified name.");
}

CmdRegion1DAddLine::~CmdRegion1DAddLine()
{
	delete m_pNameArg;
	delete m_pX1Arg;
	delete m_pX2Arg;
}

bool CmdRegion1DAddLine::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();
	int x1 = m_pX1Arg->getValue();
	int x2 = m_pX2Arg->getValue();

	if (x1 > x2)
	{
		setErrorString("Bad coordinate order");
		return false;
	}

	SimiConductorLineRegion r(x1, x2);

	if (!pInst->append1DRegion(name, r))
	{
		setErrorString(pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion1DDelete::CmdRegion1DDelete(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNameArg = new ShellCmdStringArg("name");
	m_pNameArg->setDescription("Name of the region to delete.");
	addArgument(m_pNameArg);

	setDescription("Delete a specific region name.");
}

CmdRegion1DDelete::~CmdRegion1DDelete()
{
	delete m_pNameArg;
}

bool CmdRegion1DDelete::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	std::string name = m_pNameArg->getValue();

	if (!pInst->delete1DRegion(name))
	{
		setErrorString(std::string("Can't delete region '") + name + std::string("': ") + pInst->getErrorString());
		return false;
	}

	return true;
}

CmdRegion1DClearAll::CmdRegion1DClearAll(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Clear all currently defined region names.");
}

CmdRegion1DClearAll::~CmdRegion1DClearAll()
{
}

bool CmdRegion1DClearAll::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();

	pInst->clear1DRegions();

	return true;
}

CmdRegion1DList::CmdRegion1DList(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pPrintDescArg = new ShellCmdBoolArg("printdescription");
	m_pPrintDescArg->setDefault("no");
	m_pPrintDescArg->setDescription("If set to true, not only the region names but also a description (if available) will be shown.");
	addArgument(m_pPrintDescArg);

	setDescription("List currently defined region names and their descriptions.");
}

CmdRegion1DList::~CmdRegion1DList()
{
	delete m_pPrintDescArg;
}

bool CmdRegion1DList::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	std::vector<std::string> names;
	pInst->get1DRegionNames(names);

	if (names.empty())
	{
		pIOSys->writeOutputLine("There are currently no defined regions");
		return true;
	}

	bool printDesc = m_pPrintDescArg->getValue();

	pIOSys->writeOutputLine("Currently defined regions: ");

	if (!printDesc)
	{
		for (int i = 0 ; i < names.size() ; i++)
		{
			pIOSys->print("  %s", names[i].c_str());
		}
	}
	else
	{
		for (int i = 0 ; i < names.size() ; i++)
		{
			std::vector<const SimiConductorRegion1D *> subRegions;

			pIOSys->print("  Name:        %s", names[i].c_str());

			if (!pInst->get1DRegions(names[i], subRegions))
			{
				setErrorString(std::string("Internal error (invalid region name): ") + pInst->getErrorString());
				return false;
			}

			if (subRegions.empty())
				pIOSys->print("  Description: empty");
			else
			{
				pIOSys->print("  Description:");
				char str[1024];
				int len;

				sprintf(str, "%d", subRegions.size());
				len = strlen(str);

				sprintf(str, "    %%%dd. %%s", len);

				for (int j = 0 ; j < subRegions.size() ; j++)
				{
					const SimiConductorRegion1D *pReg = subRegions[j];

					std::string desc = pReg->getDescription();

					pIOSys->print(str, j+1, pReg->getDescription().c_str());
				}

				pIOSys->writeOutputLine("");
			}
		}
	}

	return true;
}
