#include <mpi.h>
#include "cmdoptimizempi.h"
#include "mainmpi.h"
#include "extproggafactory.h"
#include "extprogmogafactory.h"
#include "simiconductorinstance.h"
#include "simimpiga.h"
#include "simimpimoga.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/iosystem.h>
#include <shellp/fileptriosystem.h>
#include <serut/memoryserializer.h>
#include <serut/dummyserializer.h>
#include <stdint.h>

using namespace shellp;

CmdOptimizeMPIGASingle::CmdOptimizeMPIGASingle(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNumParamArg = new ShellCmdIntArg("numparams");
	m_pNumParamArg->setMin(1);
	m_pNumParamArg->setDescription("The number of parameters that will be optimized.");
	addArgument(m_pNumParamArg);

	m_pPopSizeArg = new ShellCmdIntArg("popsize");
	m_pPopSizeArg->setMin(10);
	m_pPopSizeArg->setDescription("The population size of the genetic algorithm.");
	addArgument(m_pPopSizeArg);

	m_pCommandArg = new ShellCmdStringArg("command");
	m_pCommandArg->setDescription("The command name that will be executed, with the different parameters as arguments.");
	addArgument(m_pCommandArg);

	m_pTranslateCommandArg = new ShellCmdStringArg("outputcommand");
	m_pTranslateCommandArg->setDescription("When displaying the results, this command (if specified) will be executed to translate the parameters (which lie in [0,1]) to more readable output.");
	m_pTranslateCommandArg->setDefault("*");
	addArgument(m_pTranslateCommandArg);

	setDescription("This command uses a single objective genetic algorithm to search for the minimum value "
		       "of some function of a number of parameters. The parameters have values in the [0,1] interval, and are passed to the specified program "
		       "for evaluation. This program must output one floating point value. The program can of course translate the parameter values to other "
		       "values, and to be able to provide more human readable output when displaying the currently best parameter set, a second program can "
		       "be specified. The same parameters will be passed to this program, and all of its output will be displayed.");
}

CmdOptimizeMPIGASingle::~CmdOptimizeMPIGASingle()
{
	delete m_pNumParamArg;
	delete m_pPopSizeArg;
	delete m_pCommandArg;
	delete m_pTranslateCommandArg;
}

bool CmdOptimizeMPIGASingle::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	int numParams = m_pNumParamArg->getValue();
	int popSize = m_pPopSizeArg->getValue();
	std::string executable = m_pCommandArg->getValue();
	std::string outputCommand = m_pTranslateCommandArg->getValue();

	ExtProgGAFactoryParams factoryParams(numParams, executable);
	ExtProgGAFactory factory;

	if (!factory.init(&factoryParams))
	{
		setErrorString("Unable to initialize the genetic algorithm factory: " + factory.getErrorString());
		return false;
	}

	serut::DummySerializer dumSer;

	if (!factoryParams.write(dumSer))
	{
		setErrorString("Unable to count the number of bytes to serialize: " + factoryParams.getErrorString());
		return false;
	}

	int paramSize = dumSer.getBytesWritten();
	
	std::vector<uint8_t> buffer(paramSize);
	serut::MemorySerializer mSer(0, 0, &(buffer[0]), buffer.size());

	if (!factoryParams.write(mSer))
	{
		setErrorString("Unable to serialize the genetic algorithm factory parameters to memory: " + factoryParams.getErrorString());
		return false;
	}
	
	// Send the command and transmit the parameters to the other ranks
	int command = SIMICONDUCTOR_MPICOMMAND_SINGLEOBJECTIVE;

	MPI_Bcast(&command, 1, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(&paramSize, 1, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(&(buffer[0]), paramSize, MPI_BYTE, 0, MPI_COMM_WORLD);

	SimiMPIGA ga(outputCommand);

	if (!ga.runMPI(factory, popSize, pInst->getGAParams()))
	{
		setErrorString("Unable to run genetic algorithm: " + ga.getErrorString());
		return false;
	}

	return true;
}

CmdOptimizeMPIGAMulti::CmdOptimizeMPIGAMulti(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNumParamArg = new ShellCmdIntArg("numparams");
	m_pNumParamArg->setMin(1);
	m_pNumParamArg->setDescription("The number of parameters that will be optimized.");
	addArgument(m_pNumParamArg);

	m_pNumFitnessArg = new ShellCmdIntArg("numfitness");
	m_pNumFitnessArg->setMin(2);
	m_pNumFitnessArg->setMax(32);
	m_pNumFitnessArg->setDescription("The number of fitness components that are used.");
	addArgument(m_pNumFitnessArg);

	m_pPopSizeArg = new ShellCmdIntArg("popsize");
	m_pPopSizeArg->setMin(10);
	m_pPopSizeArg->setDescription("The population size of the genetic algorithm.");
	addArgument(m_pPopSizeArg);

	m_pCommandArg = new ShellCmdStringArg("command");
	m_pCommandArg->setDescription("The command name that will be executed, with the different parameters as arguments.");
	addArgument(m_pCommandArg);

	m_pTranslateCommandArg = new ShellCmdStringArg("outputcommand");
	m_pTranslateCommandArg->setDescription("When displaying the results, this command (if specified) will be executed to translate the parameters (which lie in [0,1]) to more readable output.");
	m_pTranslateCommandArg->setDefault("*");
	addArgument(m_pTranslateCommandArg);

	setDescription("This command uses a multi objective genetic algorithm to search for the optimal value "
		       "of some function of a number of parameters. The parameters have values in the [0,1] interval, and are passed to the specified program "
		       "for evaluation. This program must output a number of floating point values, and the genetic algorithm will try to minimize all of these. "
		       "The program can of course translate the parameter values to other "
		       "values, and to be able to provide more human readable output when displaying the currently best parameter set, a second program can "
		       "be specified. The same parameters will be passed to this program, and all of its output will be displayed.");
}

CmdOptimizeMPIGAMulti::~CmdOptimizeMPIGAMulti()
{
	delete m_pNumParamArg;
	delete m_pNumFitnessArg;
	delete m_pPopSizeArg;
	delete m_pCommandArg;
	delete m_pTranslateCommandArg;
}

bool CmdOptimizeMPIGAMulti::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	int numParams = m_pNumParamArg->getValue();
	int numFitness = m_pNumFitnessArg->getValue();
	int popSize = m_pPopSizeArg->getValue();
	std::string executable = m_pCommandArg->getValue();
	std::string outputCommand = m_pTranslateCommandArg->getValue();

	ExtProgMOGAFactoryParams factoryParams(numParams, numFitness, executable);
	ExtProgMOGAFactory factory;

	if (!factory.init(&factoryParams))
	{
		setErrorString("Unable to initialize the genetic algorithm factory: " + factory.getErrorString());
		return false;
	}

	serut::DummySerializer dumSer;

	if (!factoryParams.write(dumSer))
	{
		setErrorString("Unable to count the number of bytes to serialize: " + factoryParams.getErrorString());
		return false;
	}

	int paramSize = dumSer.getBytesWritten();
	
	std::vector<uint8_t> buffer(paramSize);
	serut::MemorySerializer mSer(0, 0, &(buffer[0]), buffer.size());

	if (!factoryParams.write(mSer))
	{
		setErrorString("Unable to serialize the genetic algorithm factory parameters to memory: " + factoryParams.getErrorString());
		return false;
	}
	
	// Send the command and transmit the parameters to the other ranks
	int command = SIMICONDUCTOR_MPICOMMAND_MULTIOBJECTIVE;

	MPI_Bcast(&command, 1, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(&paramSize, 1, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(&(buffer[0]), paramSize, MPI_BYTE, 0, MPI_COMM_WORLD);

	SimiMPIMOGA ga(outputCommand, &factory);

	if (!ga.runMPI(factory, popSize, pInst->getGAParams()))
	{
		setErrorString("Unable to run genetic algorithm: " + ga.getErrorString());
		return false;
	}

	ga.printSolutions();

	return true;
}


