#include "cmdoptimize.h"
#include "amoeba.h"
#include "extprogsearchfunction.h"
#include "extproggafactory.h"
#include "simiconductorinstance.h"
#include <shellp/shellcmdintarg.h>
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/shellcmdboolarg.h>
#include <shellp/iosystem.h>
#include <shellp/fileptriosystem.h>
#include <mogal/geneticalgorithm.h>
#include <errno.h>
#include <string.h>
#include <vector>

using namespace shellp;
using namespace mogal;

#ifdef WIN32
#define popen _popen
#define pclose _pclose
#endif // WIN32

CmdOptimizeAmoeba::CmdOptimizeAmoeba(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pNumParamArg = new ShellCmdIntArg("numparams");
	m_pNumParamArg->setMin(1);
	m_pNumParamArg->setDescription("The number of parameters that will be optimized.");
	addArgument(m_pNumParamArg);

	m_pCommandArg = new ShellCmdStringArg("command");
	m_pCommandArg->setDescription("The command name that will be executed, with the different parameters as arguments.");
	addArgument(m_pCommandArg);

	m_pTranslateCommandArg = new ShellCmdStringArg("outputcommand");
	m_pTranslateCommandArg->setDescription("When displaying the results, this command (if specified) will be executed to translate the parameters (which lie in [0,1]) to more readable output.");
	m_pTranslateCommandArg->setDefault("*");
	addArgument(m_pTranslateCommandArg);

	setDescription("This command uses the Amoeba method (also known as the Nelder-Mead method or downhill simplex method) to search for the minimum value "
		       "of some function of a number of parameters. The parameters initially have values in the [0,1] interval, and are passed to the specified program "
		       "for evaluation; during the course of the algorithm, a parameter can move out of the [0,1] interval. This program must output one floating point value. The program can of course translate the parameter values to other "
		       "values, and to be able to provide more human readable output when displaying the currently best parameter set, a second program can "
		       "be specified. The same parameters will be passed to this program, and all of its output will be displayed.");
}

CmdOptimizeAmoeba::~CmdOptimizeAmoeba()
{
	delete m_pNumParamArg;
	delete m_pCommandArg;
	delete m_pTranslateCommandArg;
}

bool CmdOptimizeAmoeba::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	std::string executable = m_pCommandArg->getValue();
	std::string outputCommand = m_pTranslateCommandArg->getValue();
	int numParams = m_pNumParamArg->getValue();
	bool gotOutputCommand = false;
	bool outputCommandError = false;

	if (outputCommand != std::string("*"))
		gotOutputCommand = true;

	ExtProgGAFactory factory;
	ExtProgGAFactoryParams factoryParams(numParams, executable);

	if (!factory.init(&factoryParams))
	{
		setErrorString("Couldn't initialize factory: " + factory.getErrorString());
		return false;
	}
	
	std::vector<SearchFunction *> initPoints(numParams+1);

	for (int i = 0 ; i < initPoints.size() ; i++)
		initPoints[i] = new ExtProgSearchFunction(&factory);

	Amoeba amoeba;
	double dummy;

	if (!amoeba.init(initPoints))
	{
		setErrorString("Couldn't initialize Amoeba routine: " + amoeba.getErrorString());
		return false;
	}

	bool interrupted = false;

	// TODO: make the number of steps configurable
	for (int i = 0 ; !interrupted &&  i < 500 ; i++)
	{
		if (pInst->isInterrupted())
			interrupted = true;
		else
		{

			pIOSys->print("Step %d", i);
			if (!amoeba.step(dummy))
			{
				setErrorString("Couldn't perform an optimization step: " + amoeba.getErrorString());
				return false;
			}

			const ExtProgSearchFunction *pBestPoint = (const ExtProgSearchFunction *)amoeba.getBestPoint();
			const double *pBestCoords = pBestPoint->getCurrentPoint();

			std::string bestPointString = pBestPoint->getParameterString();

			if (gotOutputCommand && !outputCommandError)
			{
				std::string cmd = outputCommand;

				cmd += std::string(" ");
				cmd += bestPointString;

				FILE *pFile = popen(cmd.c_str(), "r");
				if (pFile == 0)
				{
					outputCommandError = true;
					pIOSys->print("Warning: couldn't execute translate command: %s", strerror(errno));
					pIOSys->print("Current best (%.5g): %s", pBestPoint->getValue(), pBestPoint->getParameterString().c_str());
				}
				else
				{
					pIOSys->print("Current best (%.5g):", pBestPoint->getValue());

					FilePtrIOSystem io(pFile, 0);
					std::string nullPrefix, line;
					
					while (io.readInputLine(nullPrefix, line))
						pIOSys->writeOutputLine(line);

					int status;

					if ((status = pclose(pFile)) != 0)
					{
						outputCommandError = true;
						pIOSys->print("Warning: got non-zero return value (%d) from translate command", status);
						pIOSys->print("Current best (%.5g): %s", pBestPoint->getValue(), pBestPoint->getParameterString().c_str());
					}
				}
			}
			else
				pIOSys->print("Current best (%.5g): %s", pBestPoint->getValue(), pBestPoint->getParameterString().c_str());
		}
	}
	if (interrupted)
	{
		setErrorString("Optimization interrupted by user");
		return false;
	}

	return true;
}

CmdOptimizeGetGAParams::CmdOptimizeGetGAParams(const std::string &cmdName) : ShellCommand(cmdName)
{
	setDescription("Show the current general genetic algorithm parameters.");
}

CmdOptimizeGetGAParams::~CmdOptimizeGetGAParams()
{
}

bool CmdOptimizeGetGAParams::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();
	const GeneticAlgorithmParams *pGAParams = pInst->getGAParams();

	if (pGAParams == 0)
		pIOSys->writeOutputLine("No genetic algorithm parameters have been defined.");
	else
	{
		double beta = pGAParams->getBeta();
		bool useElitism = pGAParams->useElitism();
		bool includeBest = pGAParams->alwaysIncludeBestGenome();
		double crossOverRate = pGAParams->getCrossOverRate();

		pIOSys->print("  Selection pressure (beta):    %g", beta);
		pIOSys->print("  Use elitism:                  %s", useElitism?"yes":"no");
		pIOSys->print("  Always include best genome:   %s", includeBest?"yes":"no");
		pIOSys->print("  Cross-over rate:              %g", crossOverRate);
	}

	return true;
}

CmdOptimizeSetGAParams::CmdOptimizeSetGAParams(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pBetaArg = new ShellCmdRealArg("beta");
	m_pBetaArg->setMin(0);
	m_pBetaArg->setDescription("Sets the selection pressure beta. If the position of a genome is 'n' after "
			        "the ranking procedure (with 'n' being higher for better genomes), the probability "
				"of it being selected for reproduction or cloning is proportional to n^beta. A "
				"value of zero causes genomes to be selected at random, while higher values of "
				"beta favor fitter genomes.");
	addArgument(m_pBetaArg);

	m_pElitismArg = new ShellCmdBoolArg("elitism");
	m_pElitismArg->setDescription("If elitism is used, the best genome of a generation is copied to the next "
			        "generation. Afterwards, it is still subjected to the mutation procedure.");
	addArgument(m_pElitismArg);

	m_pIncludeBest = new ShellCmdBoolArg("keepbest");
	m_pIncludeBest->setDescription("If set to true, the best genome of a generation is copied to the next "
			        "generation and is not subjected to mutations afterwards.");
	addArgument(m_pIncludeBest);

	m_pCrossArg = new ShellCmdRealArg("crossoverrate");
	m_pCrossArg->setDescription("Sets the fraction of genomes in a generation which are obtained by "
			         "reproduction. The other genomes are created by cloning.");
	m_pCrossArg->setMin(0);
	m_pCrossArg->setMax(1);
	addArgument(m_pCrossArg);

	setDescription("Set the current genetic algorithm parameters to the specified values.");
}

CmdOptimizeSetGAParams::~CmdOptimizeSetGAParams()
{
	delete m_pBetaArg;
	delete m_pElitismArg;
	delete m_pIncludeBest;
	delete m_pCrossArg;
}

bool CmdOptimizeSetGAParams::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();

	double beta = m_pBetaArg->getValue();
	bool useElitism = m_pElitismArg->getValue();
	bool includeBest = m_pIncludeBest->getValue();
	double crossOver = m_pCrossArg->getValue();

	GeneticAlgorithmParams *pGAParams = new GeneticAlgorithmParams(beta, useElitism, includeBest, crossOver);
	pInst->setGAParams(pGAParams);

	return true;
}

