#include "cmdevalmu.h"
#include "simiconductorinstance.h"
#include <shellp/shellcmdstringarg.h>
#include <shellp/shellcmdrealarg.h>
#include <shellp/iosystem.h>
#include <stdio.h>
#include <cmath>

using namespace shellp;

// Looks for the chemical potential of a semiconductor material
// When a first estimate is found, a refinement of this estimate will
// be sought in the neighbourhoud of the estimate.

double findMuSub(double Estart, double Estop, double NC, double NV, double NA, double ND, double EA_kT, double ED_kT, 
              double EG_kT, int steps, int iterations)
{
	double bestMu = 0;
	double minVal = 1e100;
	double Ediff = Estop-Estart;

	for (int i = 0 ; i < steps ; i++)
	{
		double mu = (double)i/((double)(steps-1))*Ediff + Estart;

		double val = 1.0*std::exp(mu-EG_kT) + (NA/NC)*1.0/(std::exp(EA_kT-mu)+1.0)
			   - (NV/NC)*exp(-mu) - (ND/NC)*1.0/(1.0+std::exp(mu-EG_kT+ED_kT));

		if (std::abs(val) < minVal)
		{
			minVal = std::abs(val);
			bestMu = mu;
		}
	}

	if (iterations > 0)
	{
		double dE = 2.0/((double)(steps-1))*Ediff;
		return findMuSub(bestMu-dE, bestMu+dE, NC, NV, NA, ND, EA_kT, ED_kT, EG_kT, steps, iterations-1);
	}

	return bestMu;
}

CmdEvalMu::CmdEvalMu(const std::string &cmdName) : ShellCommand(cmdName)
{
	m_pEMinArg = new ShellCmdRealArg("E_min");
	m_pEMinArg->setDescription("The chemical potential is assumed to lie in the inverval [Emin, Emax]");
	addArgument(m_pEMinArg);
	m_pEMaxArg = new ShellCmdRealArg("E_max");
	m_pEMaxArg->setDescription("The chemical potential is assumed to lie in the inverval [Emin, Emax]");
	addArgument(m_pEMaxArg);
	m_pEAArg = new ShellCmdRealArg("E_A");
	m_pEAArg->setDescription("The difference between the acceptor energy level and the valence band top energy level.");
	addArgument(m_pEAArg);
	m_pEDArg = new ShellCmdRealArg("E_D");
	m_pEDArg->setDescription("The difference between the conduction band bottom energy level and the donor energy level.");
	addArgument(m_pEDArg);
	m_pEGArg = new ShellCmdRealArg("E_gap");
	m_pEGArg->setDescription("The size of the energy band gap.");
	addArgument(m_pEGArg);
	m_pkTArg = new ShellCmdRealArg("kT");
	m_pkTArg->setDescription("The value of k*T, in the same units as the energies specified in the other parameters.");
	addArgument(m_pkTArg);

	m_pNCArg = new ShellCmdRealArg("N_C");
	m_pNCArg->setDescription("Effective density of states of the conduction band.");
	m_pNCArg->setMin(0);
	addArgument(m_pNCArg);
	m_pNVArg = new ShellCmdRealArg("N_V");
	m_pNVArg->setDescription("Effective density of states of the valence band.");
	m_pNVArg->setMin(0);
	addArgument(m_pNVArg);

	m_pNAArg = new ShellCmdRealArg("N_A");
	m_pNAArg->setDescription("Acceptor density.");
	m_pNAArg->setMin(0);
	addArgument(m_pNAArg);
	m_pNDArg = new ShellCmdRealArg("N_D");
	m_pNDArg->setDescription("Donor density.");
	m_pNDArg->setMin(0);
	addArgument(m_pNDArg);

	m_pVarName = new ShellCmdStringArg("variable");
	m_pVarName->setDescription("Variable to store the result of the calculation in.");
	addArgument(m_pVarName);

	setDescription("Finds the chemical potential for a semiconductor. The precise units used for energy or densities don't matter "
		       "as long as the same unit is used for all energy values (including k*T) and the same unit is used for all "
		       "density values. The chemical potential value that is calculated will be in the same units as the energies specified.");
}

CmdEvalMu::~CmdEvalMu()
{
	delete m_pEMinArg;
	delete m_pEMaxArg;
	delete m_pEAArg;
	delete m_pEDArg;
	delete m_pEGArg;
	delete m_pkTArg;
	delete m_pNCArg;
	delete m_pNVArg;
	delete m_pNAArg;
	delete m_pNDArg;
	delete m_pVarName;
}

bool CmdEvalMu::execute()
{
	SimiConductorInstance *pInst = (SimiConductorInstance *)ShellInstance::getInstance();
	IOSystem *pIOSys = pInst->getIOSystem();

	double Emin = m_pEMinArg->getValue();
	double Emax = m_pEMaxArg->getValue();

	if (Emin >= Emax)
	{
		setErrorString("The minimum energy should be smaller than the maximum energy");
		return false;
	}

	double kT = m_pkTArg->getValue();
	double mu = findMuSub(Emin/kT,
			      Emax/kT,
			      m_pNCArg->getValue(),
			      m_pNVArg->getValue(),
			      m_pNAArg->getValue(),
			      m_pNDArg->getValue(),
			      m_pEAArg->getValue()/kT,
			      m_pEDArg->getValue()/kT,
			      m_pEGArg->getValue()/kT,
			      1000, 1000);

	mu *= kT;

	if (mu < Emin || mu > Emax)
	{
		setErrorString("No valid chemical potential was found in the specified interval.");
		return false;
	}

	// Store in variable
	
	std::string varName = m_pVarName->getValue();

	if (!pInst->storeVariable(varName, mu))
	{
		setErrorString(std::string("Unable to store result in variable: ") + pInst->getErrorString());
		return false;
	}

	pIOSys->print("Result: %s = %g", m_pVarName->getValue().c_str(), mu);

	return true;

}

