#include "simiconductorconfig.h"
#include "binimage.h"
#ifdef SIMICONDUCTOR_CONFIG_QT4
	#include <QImage>
#endif // SIMICONDUCTOR_CONFIG_QT4
#include <stdio.h>
#include <string.h>

// TODO: for debugging
#include <iostream>

BinImage::BinImage()
{
	m_width = 0;
	m_height = 0;
}

BinImage::~BinImage()
{
}

bool BinImage::init(int width, int height)
{
	if (width < 1 || height < 1)
	{
		setErrorString("Invalid width or height specified");
		return false;
	}

	int totalSize = width*height;

	m_values.resize(totalSize);
	memset(&(m_values[0]), 0, sizeof(uint16_t)*totalSize);

	return true;
}

bool BinImage::load(const std::string &fileName, bool flipX, bool flipY)
{
	FILE *pFile = fopen(fileName.c_str(), "rb");
	if (pFile == 0)
	{
		setErrorString("Can't open file " + fileName);
		return false;
	}

	// TODO: I'm guessing that the format is (all little endian):
	//  - number of dimensions (16-bit)
	//  - number of x-pixels (32-bit)
	//  - number of y-pixels (32-bit)
	//  - data (16-bit)

	uint8_t bytes[4];
	if (fread(bytes, 1, 2, pFile) != 2)
	{
		setErrorString("Couldn't read number of dimensions");
		fclose(pFile);
		return false;
	}

	uint16_t dimensions = bytes[0] | (bytes[1] << 8);
	char str[1024];

	if (dimensions != 2)
	{
		sprintf(str, "Number of dimensions is %d, but we can only handle 2", (int)dimensions);
		setErrorString(str);
		fclose(pFile);
		return false;
	}

	if (fread(bytes, 1, 4, pFile) != 4)
	{
		setErrorString("Can't read number of x-pixels");
		fclose(pFile);
		return false;
	}

	uint32_t xPix = bytes[0] | (bytes[1] << 8) | (bytes[2] << 16) | (bytes[3] << 24);
	int width = (int)xPix;
	int maxWidth = 32768;

	if (width < 1 || width > maxWidth)
	{
		sprintf(str, "Read invalid number of x-pixels %d (can only support %d)", width, maxWidth);
		setErrorString(str);
		fclose(pFile);
		return false;
	}

	if (fread(bytes, 1, 4, pFile) != 4)
	{
		setErrorString("Can't read number of y-pixels");
		fclose(pFile);
		return false;
	}

	uint32_t yPix = bytes[0] | (bytes[1] << 8) | (bytes[2] << 16) | (bytes[3] << 24);
	int height = (int)yPix;
	int maxHeight = 32768;

	if (height < 1 || height > maxHeight)
	{
		sprintf(str, "Read invalid number of y-pixels %d (can only support %d)", height, maxHeight);
		setErrorString(str);
		fclose(pFile);
		return false;
	}

	int totalPixels = width*height;

	m_width = width;
	m_height = height;
	m_values.resize(totalPixels);
	memset(&(m_values[0]), 0, sizeof(uint16_t)*totalPixels);

	for (int y = 0 ; y < height ; y++)
	{
		//int yPos = (flipY)?(height-1-y):y;
		int yPos = (flipY)?y:(height-1-y);

		for (int x = 0 ; x < width ; x++)
		{
			if (fread(bytes, 1, 2, pFile) != 2)
			{
				sprintf(str, "Can't read pixel (%d,%d) from file", x, y);
				setErrorString(str);
				fclose(pFile);
				return false;
			}

			uint16_t v = bytes[0] | (bytes[1] << 8);
			int xPos = (flipX)?(width-1-x):x;

			m_values[xPos + yPos*width] = v;
		}
	}

	fclose(pFile);
	return true;
}

#ifdef SIMICONDUCTOR_CONFIG_QT4

bool BinImage::importImage(const std::string &fileName)
{
	QImage img;

	if (!img.load(fileName.c_str()))
	{
		setErrorString("Couldn't load image file " + fileName);
		return false;
	}

	m_width = img.width();
	m_height = img.height();

	m_values.resize(m_width*m_height);

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			QRgb color = img.pixel(x, m_height-1-y);

			m_values[x+y*m_width] = qGray(color);
		}
	}
	return true;
}

bool BinImage::exportImage(const std::string &fileName, int minValue, int maxValue) const
{
	if (m_width <= 0 || m_height <= 0 || m_values.size() == 0)
	{
		setErrorString("No valid BIN data have been loaded or created yet");
		return false;
	}

	if (minValue < 0 && maxValue < 0)
	{
		minValue = 65535;
		maxValue = 0;

		for (int y = 0 ; y < m_height ; y++)
		{
			for (int x = 0 ; x < m_width ; x++)
			{
				int value = (int)m_values[x+y*m_width];
				
				if (value > maxValue)
					maxValue = value;
				if (value < minValue)
					minValue = value;
			}
		}
	}
	else
	{
		char str[1024];

		if (maxValue < minValue)
		{
			sprintf(str, "Specified maximum value %d is less than the specified minimal value %d", maxValue, minValue);
			setErrorString(str);
			return false;
		}
	}

	if (maxValue == minValue) // Avoid division by zero below
		maxValue++;

	QImage img(m_width, m_height, QImage::Format_RGB32);

	for (int y = 0 ; y < m_height ; y++)
	{
		for (int x = 0 ; x < m_width ; x++)
		{
			int value = (int)m_values[x+y*m_width];
			int value2 = 0;

			if (value < minValue)
				value2 = 0;
			else if (value > maxValue)
				value2 = 255;
			else
				value2 = (int) ((double)(value-minValue)/(double)(maxValue-minValue)*255.0 + 0.5);

			img.setPixel(x, m_height-1-y, qRgb(value2, value2, value2));
		}
	}

	if (!img.save(fileName.c_str()))
	{
		setErrorString("Couldn't write to specified file " + fileName);
		return false;
	}
	return true;
}

#else

bool BinImage::importImage(const std::string &fileName)
{
	setErrorString("Importing an image into a BIN file requires Qt4 support at compile time");
	return false;
}

bool BinImage::exportImage(const std::string &fileName, int minValue, int maxValue) const
{
	setErrorString("Exporting a BIN file to an image requires Qt4 support at compile time");
	return false;
}

#endif // SIMICONDUCTOR_CONFIG_QT4

bool BinImage::save(const std::string &fileName) const
{
	setErrorString("TODO");
	return false;

	return true;
}

bool BinImage::getMinMax(uint16_t &min, uint16_t &max) const
{
	if (m_width <= 0 || m_height <= 0 || m_values.size() == 0)
	{
		setErrorString("No valid BIN data have been loaded or created yet");
		return false;
	}

	uint16_t minValue = 65535;
	uint16_t maxValue = 0;
	size_t num = m_values.size();

	for (int i = 0 ; i < num ; i++)
	{
		uint16_t v = m_values[i];

		if (v < minValue)
			minValue = v;
		if (v > maxValue)
			maxValue = v;
	}

	min = minValue;
	max = maxValue;

	return true;
}

bool BinImage::resize(int width, int height)
{
	if (m_width <= 0 || m_height <= 0 || m_values.size() == 0)
	{
		setErrorString("No valid BIN data have been loaded or created yet");
		return false;
	}

	if (width < 1 || height < 1)
	{
		setErrorString("Specified dimensions are invalid");
		return false;
	}
	
	if (m_width == width && m_height == height)
		return true;

	std::vector<uint16_t> newPixels(width*height);

	for (int y = 0 ; y < height ; y++)
	{
		double yStart = (double)y/(double)height * (double)m_height;
		double yStop = (double)(y+1)/(double)height * (double)m_height;

		if (yStart < 0)
			yStart = 0;
		if (yStop > m_height)
			yStop = m_height;

		double yDiff = yStop-yStart;

		for (int x = 0 ; x < width ; x++)
		{
			double xStart = (double)x/(double)width * (double)m_width;
			double xStop = (double)(x+1)/(double)width * (double)m_width;

			if (xStart < 0)
				xStart = 0;
			if (xStop > m_width)
				xStop = m_width;

			double xDiff = xStop-xStart;

			double sum = integrate(xStart, xStop, yStart, yStop);

			newPixels[x+y*width] = sum/(yDiff*xDiff);
		}
	}

	m_height = height;
	m_width = width;
	m_values = newPixels;

	return true;
}

double BinImage::integrate(double x1, double x2, double y1, double y2) const
{
	double sum = 0;

	int X1 = (int)x1;
	int X2 = (int)x2;
	int Y1 = (int)y1;
	int Y2 = (int)y2;

	if (X2 > m_width-1)
	{
	//	std::cerr << "Warning X2 > m_width - 1: " << X2 << " > " << m_width << std::endl;
		X2 = m_width-1;
	}
	if (Y2 > m_height-1)
	{
	//	std::cerr << "Warning Y2 > m_height - 1: " << Y2 << " > " << m_height << std::endl;
		Y2 = m_height-1;
	}

	double x1frac = x1-(double)X1;
	double x2frac = x2-(double)X2;
	double y1frac = y1-(double)Y1;
	double y2frac = y2-(double)Y2;

	// Bulk
	for (int Y = Y1+1 ; Y < Y2 ; Y++)
	{
		for (int X = X1+1 ; X < X2 ; X++)
		{
			if (X < 0 || X >= m_width)
				std::cerr << "Warning: X = " << X << std::endl;
			if (Y < 0 || Y >= m_height)
				std::cerr << "Warning: Y = " << Y << std::endl;
	
			sum += m_values[X+Y*m_width];
		}
	}

	// Upper and lower edge
	{
		for (int X = X1+1 ; X < X2 ; X++)
		{
			sum += m_values[X+Y2*m_width]*y2frac;
			sum += m_values[X+Y1*m_width]*(1.0-y1frac);
		}
	}

	// Left and right edge
	{
		for (int Y = Y1+1 ; Y < Y2 ; Y++)
		{
			sum += m_values[X1+Y*m_width]*(1.0-x1frac);
			sum += m_values[X2+Y*m_width]*x2frac;
		}
	}

	// Rest
	
	if (X1 == X2)
	{
		if (Y1 == Y2)
		{
			sum += m_values[X1+Y1*m_width]*(x2frac-x1frac)*(y2frac-y1frac);
		}
		else // Y2 > Y1
		{
			sum += m_values[X1+Y1*m_width]*(x2frac-x1frac)*(1.0-y1frac);
			sum += m_values[X1+Y2*m_width]*(x2frac-x1frac)*y2frac;
		}
	}
	else // X2 > X1
	{
		if (Y1 == Y2)
		{
			sum += m_values[X1+Y1*m_width]*(y2frac-y1frac)*(1.0-x1frac);
			sum += m_values[X2+Y1*m_width]*(y2frac-y1frac)*x2frac;
		}
		else // Y2 > Y1
		{
			sum += m_values[X1+Y1*m_width]*(1.0-x1frac)*(1.0-y1frac);
			sum += m_values[X2+Y1*m_width]*x2frac*(1.0-y1frac);
			sum += m_values[X1+Y2*m_width]*(1.0-x1frac)*y2frac;
			sum += m_values[X2+Y2*m_width]*x2frac*y2frac;
		}
	}

	return sum;
}

