#!/usr/bin/env python

from PySide import QtCore, QtGui
from regionimage import *
from simiexception import *
from matpropwidget import *
import ui_regioneditor
import sys
import os
import pprint
import pickle

def processRegionRects(regDefs, prefix, regionWidgetNames, isTwoD = True):
    simiRegDefString = ""
    for reg in regDefs:
        idx = idx = reg['index']
        regName = prefix + regionWidgetNames[idx]
        
        if isTwoD:
            simiRegDefString += "reg2/new " + regName + "\n"
        else:
            simiRegDefString += "reg1/new " + regName + "\n"

        for rect in reg['rects']:
            x1 = rect[0][0]
            y1 = rect[0][1]
            x2 = rect[1][0]
            y2 = rect[1][1]
            # +1 since the counting starts at 1 in simiconductor
            if isTwoD:
                simiRegDefString += "reg2/append/rect " + regName + " " + str(x1+1) + " " + str(y1+1) + " " + str(x2+1) + " " + str(y2+1) + "\n" 
            else:
                simiRegDefString += "reg1/append/line " + regName + " " + str(y1+1) + " " + str(y2+1) + "\n" 

        simiRegDefString += "\n"

    return simiRegDefString

def performCheck(regImage, regDefs):
    w = regImage.getWidth()
    h = regImage.getHeight()
    initVal = -123456
    indices = [ initVal for idx in range(0,w*h) ]
    for reg in regDefs:
        index = reg['index']
        rects = reg['rects']
        for r in rects:
            x1 = r[0][0]
            y1 = r[0][1]
            x2 = r[1][0]
            y2 = r[1][1]

            for y in range(y1, y2+1):
                for x in range(x1, x2+1):
                    if x < 0 or y < 0 or x >= w or y >= h:
                        raise SimiException("Region finder sanity check failed: invalid coordinates (" + str(x) + "," + str(y) + ")")
                    if indices[x+y*w] != initVal:
                        raise SimiException("Region finder sanity check failed: point (" + str(x) + "," + str(y) + ") set multiple times")
                    indices[x+y*w] = index

    for y in range(0, h):
        for x in range(0, w):
            idx = indices[x+y*w]
            if idx != regImage.getRegionIndex(x, y):
                raise SimiException("Region finder sanity check failed: point (" + str(x) + "," + str(y) + ") index doesnt match")


def expandRect(xDir, yDir, curRect, regImage, pointProcessed, regNumber):
    w = regImage.getWidth()
    h = regImage.getHeight()
    x1 = curRect[0][0]
    y1 = curRect[0][1]
    x2 = curRect[1][0]
    y2 = curRect[1][1]

    if yDir == 0:
        if xDir == -1:
            x = x1-1
            newRect = ( (x1-1, y1), (x2, y2) )
        else:
            x = x2+1
            newRect = ( (x1, y1), (x2+1, y2) )

        if x >= 0 and x < w:
            found = True
            for y in range(y1, y2+1):
                if pointProcessed[x+y*w] or regImage.getRegionIndex(x, y) != regNumber:
                    found = False

            if found:
                for y in range(y1, y2+1):
                    pointProcessed[x+y*w] = True
                return ( newRect, True)

    else: # yDir != 0, so xDir == 0
        if yDir == -1:
            y = y1-1
            newRect = ( (x1, y1-1), (x2, y2) )
        else:
            y = y2+1
            newRect = ( (x1, y1), (x2, y2+1) )

        if y >= 0 and y < h:
            found = True
            for x in range(x1, x2+1):
                if pointProcessed[x+y*w] or regImage.getRegionIndex(x, y) != regNumber:
                    found = False

            if found:
                for x in range(x1, x2+1):
                    pointProcessed[x+y*w] = True
                return (newRect, True)

    # if not found, return old region and False
    return (curRect, False)

def findRectRegion(regNumber, xPos, yPos, regImage, pointProcessed):
    curRect = ( (xPos, yPos), (xPos, yPos) )
    w = regImage.getWidth()
    h = regImage.getHeight()
    pointProcessed[xPos + yPos*w] = True
    
    done = False
    while not done:
        (curRect, flag1) = expandRect(-1, 0, curRect, regImage, pointProcessed, regNumber)
        (curRect, flag2) = expandRect(+1, 0, curRect, regImage, pointProcessed, regNumber)
        (curRect, flag3) = expandRect(0, -1, curRect, regImage, pointProcessed, regNumber)
        (curRect, flag4) = expandRect(0, +1, curRect, regImage, pointProcessed, regNumber)

        if not flag1 and not flag2 and not flag3 and not flag4: # couldn't expand
            done = True

    return curRect

def findRegionDefinition(regImage, regNumber, contact, contactY):
    w = regImage.getWidth()
    h = regImage.getHeight()

    if contact:
        pointProcessed = [ True for idx in range(w*h) ]
        for x in range(0, w):
            pointProcessed[x+contactY*w] = False
    else:
        pointProcessed = [ False for idx in range(w*h) ]

    done = False
    rectRanges = [ ]
    while not done:
        xPos = -1
        yPos = -1
        done = True
        for idx in range(w*h):
            xPos = idx%w
            yPos = idx/w
            if not pointProcessed[idx] and regImage.getRegionIndex(xPos, yPos) == regNumber:
                done = False
                break

        if not done:
            regionDefinition = findRectRegion(regNumber, xPos, yPos, regImage, pointProcessed)
            rectRanges.append(regionDefinition)

    return rectRanges

def findAllRegionDefinitions(regImage, contact = False, contactY = -1):
    regDefs = [ ]
    numRegions = regImage.getNumRegions()
    for idx in range(0, numRegions):
        col = regImage.getRegionColor(idx)
        ranges = findRegionDefinition(regImage, idx, contact, contactY)
        if len(ranges) > 0:
            curRegDef = { 'index': idx, 'color': col, 'rects': ranges }
            regDefs.append(curRegDef)

    return regDefs

class RegionEditor(QtGui.QMainWindow):

    def __init__(self, parent = None):
        super(RegionEditor, self).__init__(parent)

        self.ui = ui_regioneditor.Ui_RegionEditor()
        self.ui.setupUi(self)

        self.ui.actionQuit.triggered.connect(self.onQuit)
        self.ui.actionImport_image.triggered.connect(self.onImportImage)
        self.ui.actionGenerate1D.triggered.connect(self.onGenerate1D)
        self.ui.actionGenerate2D.triggered.connect(self.onGenerate2D)
        self.ui.actionLoad.triggered.connect(self.onLoad)
        self.ui.actionSave.triggered.connect(self.onSave)

#        validator = QtGui.QDoubleValidator(self)
#        validator.setRange(0,1e50)
#        self.ui.widthValue.setValidator(validator)
#        self.ui.heightValue.setValidator(validator)
#        self.ui.ncvValue.setValidator(validator)
#        self.ui.ncvValue.setText("1e25")
        self.ui.matTabWidget.clear()
        self.ui.matTabWidget.setMovable(True)

        self.ui.graphicsView.imageClickedSignal.connect(self.regionPointClicked)
        self.ui.widthValue.textChanged.connect(self.onPhysicalWidthChanged)
        self.ui.heightValue.textChanged.connect(self.onPhysicalHeightChanged)
        self.ui.tempValue.valueChanged.connect(self.onTemperatureChanged)
        self.ui.addMatButton.clicked.connect(self.onAddMaterial)
        self.ui.delMatButton.clicked.connect(self.onDeleteMaterial)

        self.regionImage = None
        self.selectedColor = None

        self.lastDir = os.getcwd()

    def onImportImage(self):
        (selectedFile, filt) = QtGui.QFileDialog.getOpenFileName(None, "Select region image PNG to import", self.lastDir, "PNG Files (*.png *.PNG)")
        if not selectedFile:
            return

        try:
            regImage = RegionImage(selectedFile)
        except SimiException as e:
            QtGui.QMessageBox.warning(None, "Error","Couldn't import selected file: " + str(e))
            return
        
        self.setRegionImage(regImage)
        self.lastDir = os.path.dirname(selectedFile)

    def setRegionImage(self, regImage):

        if self.regionImage:
            try:
                if QtGui.QMessageBox.question(None, "Preserve pixel dimensions?", "Should the old pixel dimensions be preserved for the new image?", 
                        QtGui.QMessageBox.No|QtGui.QMessageBox.Yes, QtGui.QMessageBox.No) == QtGui.QMessageBox.Yes:
                    regImage.copyPixelDimensions(self.regionImage)
            except:
                pass

        self.regionImage = regImage
        self.regionPointClicked(0, 0) # To initialize the selected color
        self.ui.graphicsView.setRegionImage(regImage)
        self.syncRegionImageValues()

    def syncRegionImageValues(self):
        regImage = self.regionImage

        self.ui.graphicsView.setWidthHeight(regImage)

        if regImage:
            self.ui.xPixels.setText(str(regImage.getWidth()))
            self.ui.yPixels.setText(str(regImage.getHeight()))
            self.ui.widthValue.setText(regImage.getPhysicalWidth())
            self.ui.heightValue.setText(regImage.getPhysicalHeight())
            self.ui.tempValue.setValue(regImage.getTemperature())

    def onQuit(self):
        self.close()

    def closeEvent(self, e):
        # TODO: check save state
        e.accept()

    def regionPointClicked(self, x, y):
        if not self.regionImage:
            return

        idx = self.regionImage.getRegionIndex(x, y)
        col = self.regionImage.getRegionColor(idx)

        self.setSelectedRegionColor(col)

    def setSelectedRegionColor(self, col):
        widget = self.ui.selColor
        widget.setAutoFillBackground(True)
        pal = widget.palette()
        pal.setColor(QtGui.QPalette.Window, QtGui.QColor(col))
        widget.setPalette(pal)
        self.selectedColor = col

    def onTemperatureChanged(self, value):
        if self.regionImage:
            self.regionImage.setTemperature(value)
            self.syncRegionImageValues()

    def onPhysicalWidthChanged(self, value):
        if self.regionImage:
            try:
                #w = float(value)
                w = value
                self.regionImage.setPhysicalWidth(w)
                self.syncRegionImageValues()
            except:
                pass
    
    def onPhysicalHeightChanged(self, value):
        if self.regionImage:
            try:
                # h = float(value)
                h = value
                self.regionImage.setPhysicalHeight(h)
                self.syncRegionImageValues()
            except:
                pass

    def onAddMaterial(self):
        w = MatPropWidget(self.ui.matTabWidget)
        self.ui.matTabWidget.addTab(w, w.getMaterialName())
        self.ui.matTabWidget.setCurrentIndex(self.ui.matTabWidget.count()-1)
        w.ui.matColorButton.clicked.connect(self.onTransferSelectedColor)
        if self.selectedColor:
            w.setMaterialColor(self.selectedColor)

    def onDeleteMaterial(self):
        idx = self.ui.matTabWidget.currentIndex()
        if idx >= 0:
            self.ui.matTabWidget.removeTab(idx)

    def onTransferSelectedColor(self):
        if self.selectedColor == None:
            return

        w = self.ui.matTabWidget.currentWidget()
        w.setMaterialColor(self.selectedColor)

    def onGenerate1D(self):
        if not self.regionImage:
            QtGui.QMessageBox.warning(None, "Error","No regions defined yet")
            return

        if self.regionImage.getWidth() != 1:
            QtGui.QMessageBox.warning(None, "Warning", "Image width must be equal to one pixel for a 1D device")
            return
        
        regDefs = findAllRegionDefinitions(self.regionImage)
        performCheck(self.regionImage, regDefs)

        # get contacts
        botRegDefs = findAllRegionDefinitions(self.regionImage, True, 0)
        topRegDefs = findAllRegionDefinitions(self.regionImage, True, self.regionImage.getHeight()-1)

        if len(botRegDefs) != 1 or len(topRegDefs) != 1:
            QtGui.warning(None, "Error", "Internal error: len(botRegDefs) != 1 or len(topRegDefs) != 1")
            return

        #pprint.pprint(botRegDefs)
        #pprint.pprint(topRegDefs)

        # Get the region names

        numRegions = self.regionImage.getNumRegions()
        numMatWidgets = self.ui.matTabWidget.count()
        
        regionWidgetIdx = [ -1 for idx in range(0, numRegions) ]
        for idx in range(0, numRegions):
            for wdgIdx in range(0, numMatWidgets):
                w = self.ui.matTabWidget.widget(wdgIdx)
                col = w.getSelectedColor()
                if col != None and col == self.regionImage.getRegionColor(idx):
                    if regionWidgetIdx[idx] >= 0:
                        QtGui.QMessageBox.warning(None, "Error","Duplicate properties for a region color in the materials list")
                        return

                    regionWidgetIdx[idx] = wdgIdx

            if regionWidgetIdx[idx] < 0:
                QtGui.QMessageBox.warning(None, "Error","Not all region colors have been used in the materials list")
                return

        regionWidgetNames = [ ]
        regionWidgets = [ ]
        for idx in range(0, numRegions):
            wdgIdx = regionWidgetIdx[idx]
            w = self.ui.matTabWidget.widget(wdgIdx)
            name = w.getCleanedName()
            if name in regionWidgetNames:
                QtGui.QMessageBox.warning(None, "Error","Identifier derived from " + w.getMaterialName() + " is " + name + ", but this is already in use by another material") 
                return

            regionWidgetNames.append(name)
            regionWidgets.append(w)

        #print regionWidgetNames

        # Ok, got the region definitions

        simiRegDefString = ""


        # TODO: set variable values T, kT_ev, VBI, ni_, genRate_, genRate_ni_, recFactor_
        #                           eDiff_, hDiff_, eMob_, hMob_, epsRel_, Vn_, Vp_,
        #                           n_bot_,p_bot_,n_top_,p_top_

        simiRegDefString += "math/def T " + str(self.regionImage.getTemperature()) + "\n"
        simiRegDefString += "math/def kT_ev T*kB/e\n" 
        simiRegDefString += "math/def topContact " + str(self.ui.topContact.value()) + "\n"
        simiRegDefString += "math/def botContact " + str(self.ui.bottomContact.value()) + "\n"
        simiRegDefString += "math/def VBI (botContact-topContact)\n"
        simiRegDefString += "math/def g0 " + self.ui.ncvValue.text() + "\n"
        
        for reg in regDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            w = regionWidgets[idx]
            simiRegDefString += "math/def lumo_" + name + " " + str(w.getLUMO()) + "\n"
            simiRegDefString += "math/def homo_" + name + " " + str(w.getHOMO()) + "\n"
            simiRegDefString += "math/def ni_" + name + " g0*exp(-(homo_" + name + "-lumo_" + name +")/(2.0*kT_ev))\n"
            simiRegDefString += "math/def genRate_" + name + " " + w.getGenerationRate() + "\n"
            simiRegDefString += "math/def eMob_" + name + " " + w.getElectronMobility() + "\n"
            simiRegDefString += "math/def hMob_" + name + " " + w.getHoleMobility() + "\n"
            simiRegDefString += "math/def epsRel_" + name + " " + str(w.getRelativePermittivity()) + "\n"
            if w.isLangevin():
                simiRegDefString += "math/def recFactor_" + name + " e*(eMob_" + name + "+hMob_"+name+")/(eps0*epsRel_" + name + ")\n"
            else:
                simiRegDefString += "math/def recFactor_" + name + " " + w.getRecombinationFactor() + "\n"
            simiRegDefString += "math/def genRate_ni_" + name + " recFactor_" + name + "*ni_" + name + "^2\n"
            simiRegDefString += "math/def eDiff_" + name + " eMob_" + name + "*kT_ev\n"
            simiRegDefString += "math/def hDiff_" + name + " hMob_" + name + "*kT_ev\n"
            # TODO: check this: it should be okay to just use the energy levels for this, instead
            #       of differences with a reference level
            simiRegDefString += "math/def Vn_" + name + " lumo_" + name + "\n"
            simiRegDefString += "math/def Vp_" + name + " homo_" + name + "\n"
            simiRegDefString += "\n"

        for reg in botRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "math/def n_bot_" + name + " g0*exp(-(botContact-lumo_" + name + ")/kT_ev)\n" 
            simiRegDefString += "math/def p_bot_" + name + " g0*exp(-(homo_" + name + "-botContact)/kT_ev)\n" 

        for reg in topRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "math/def n_top_" + name + " g0*exp(-(topContact-lumo_" + name + ")/kT_ev)\n" 
            simiRegDefString += "math/def p_top_" + name + " g0*exp(-(homo_" + name + "-topContact)/kT_ev)\n" 
        
        simiRegDefString += processRegionRects(regDefs, "REG_", regionWidgetNames, isTwoD = False)
        simiRegDefString += processRegionRects(botRegDefs, "BOTCONTACT_", regionWidgetNames, isTwoD = False)
        simiRegDefString += processRegionRects(topRegDefs, "TOPCONTACT_", regionWidgetNames, isTwoD = False)
    
        # nx = self.regionImage.getWidth()  # Not using this, is just one
        ny = self.regionImage.getHeight()
        # physw = self.regionImage.getPhysicalWidth() # Not using this in a one dimensional simulation
        physh = self.regionImage.getPhysicalHeight()
        simiRegDefString += "sim1/new " + str(ny) + " " + str(physh) + "\n"
        simiRegDefString += "sim1/temp/set T\n"
        simiRegDefString += "sim1/phi/set VBI yes\n"

        # Bulk regions

        for reg in regDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim1/reg/set n REG_" + name + " ni_" + name + "\n"
            simiRegDefString += "sim1/reg/set p REG_" + name + " ni_" + name + "\n"
            simiRegDefString += "sim1/reg/set bg REG_" + name + " 0" + "\n"
            simiRegDefString += "sim1/reg/set g REG_" + name + " genRate_" + name + "+genRate_ni_" + name + "\n"
            simiRegDefString += "sim1/reg/set rf REG_" + name + " recFactor_" + name + "\n"
            simiRegDefString += "sim1/reg/set dn REG_" + name + " eDiff_" + name + "\n"
            simiRegDefString += "sim1/reg/set dp REG_" + name + " hDiff_" + name + "\n"
            simiRegDefString += "sim1/reg/set nmob REG_" + name + " eMob_" + name + "\n"
            simiRegDefString += "sim1/reg/set pmob REG_" + name + " hMob_" + name + "\n"
            simiRegDefString += "sim1/reg/set eps REG_" + name + " epsRel_" + name + "\n"
            simiRegDefString += "sim1/reg/set Vn REG_" + name + " Vn_" + name + "\n"
            simiRegDefString += "sim1/reg/set Vp REG_" + name + " Vp_" + name + "\n"
            simiRegDefString += "\n"
        
        # Contacts

        for reg in botRegDefs: # Is actually just one element
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim1/reg/set n BOTCONTACT_" + name + " n_bot_" + name + "\n"
            simiRegDefString += "sim1/reg/set p BOTCONTACT_" + name + " p_bot_" + name + "\n"

        for reg in topRegDefs: # is actually just one element
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim1/reg/set n TOPCONTACT_" + name + " n_top_" + name + "\n"
            simiRegDefString += "sim1/reg/set p TOPCONTACT_" + name + " p_top_" + name + "\n"

        (fileName, filt) = QtGui.QFileDialog.getSaveFileName(None, "Enter file to save commands to", self.lastDir, "Text files (*.txt)")
        if not fileName:
            return

        f = None
        try:
            f = open(fileName, "wt")
            f.write(simiRegDefString)
            self.lastDir = os.path.dirname(fileName)
            QtGui.QMessageBox.information(None, "Info","Commands written to file")
        except Exception as e:
            QtGui.QMessageBox.warning(None, "Error","Couldn't write to file: " + str(e)) 
        finally:
            if f: f.close()

    def onGenerate2D(self):
        if not self.regionImage:
            QtGui.QMessageBox.warning(None, "Error","No regions defined yet")
            return
        
        regDefs = findAllRegionDefinitions(self.regionImage)
        performCheck(self.regionImage, regDefs)

        # get contacts
        botRegDefs = findAllRegionDefinitions(self.regionImage, True, 0)
        topRegDefs = findAllRegionDefinitions(self.regionImage, True, self.regionImage.getHeight()-1)

        #pprint.pprint(botRegDefs)
        #pprint.pprint(topRegDefs)

        # Get the region names

        numRegions = self.regionImage.getNumRegions()
        numMatWidgets = self.ui.matTabWidget.count()
        
        regionWidgetIdx = [ -1 for idx in range(0, numRegions) ]
        for idx in range(0, numRegions):
            for wdgIdx in range(0, numMatWidgets):
                w = self.ui.matTabWidget.widget(wdgIdx)
                col = w.getSelectedColor()
                if col != None and col == self.regionImage.getRegionColor(idx):
                    if regionWidgetIdx[idx] >= 0:
                        QtGui.QMessageBox.warning(None, "Error","Duplicate properties for a region color in the materials list")
                        return

                    regionWidgetIdx[idx] = wdgIdx

            if regionWidgetIdx[idx] < 0:
                QtGui.QMessageBox.warning(None, "Error","Not all region colors have been used in the materials list")
                return

        regionWidgetNames = [ ]
        regionWidgets = [ ]
        for idx in range(0, numRegions):
            wdgIdx = regionWidgetIdx[idx]
            w = self.ui.matTabWidget.widget(wdgIdx)
            name = w.getCleanedName()
            if name in regionWidgetNames:
                QtGui.QMessageBox.warning(None, "Error","Identifier derived from " + w.getMaterialName() + " is " + name + ", but this is already in use by another material") 
                return

            regionWidgetNames.append(name)
            regionWidgets.append(w)

        #print regionWidgetNames

        # Ok, got the region definitions

        simiRegDefString = ""


        # TODO: set variable values T, kT_ev, VBI, ni_, genRate_, genRate_ni_, recFactor_
        #                           eDiff_, hDiff_, eMob_, hMob_, epsRel_, Vn_, Vp_,
        #                           n_bot_,p_bot_,n_top_,p_top_

        simiRegDefString += "math/def T " + str(self.regionImage.getTemperature()) + "\n"
        simiRegDefString += "math/def kT_ev T*kB/e\n" 
        simiRegDefString += "math/def topContact " + str(self.ui.topContact.value()) + "\n"
        simiRegDefString += "math/def botContact " + str(self.ui.bottomContact.value()) + "\n"
        simiRegDefString += "math/def VBI (botContact-topContact)\n"
        simiRegDefString += "math/def g0 " + self.ui.ncvValue.text() + "\n"
        
        for reg in regDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            w = regionWidgets[idx]
            simiRegDefString += "math/def lumo_" + name + " " + str(w.getLUMO()) + "\n"
            simiRegDefString += "math/def homo_" + name + " " + str(w.getHOMO()) + "\n"
            simiRegDefString += "math/def ni_" + name + " g0*exp(-(homo_" + name + "-lumo_" + name +")/(2.0*kT_ev))\n"
            simiRegDefString += "math/def genRate_" + name + " " + w.getGenerationRate() + "\n"
            simiRegDefString += "math/def eMob_" + name + " " + w.getElectronMobility() + "\n"
            simiRegDefString += "math/def hMob_" + name + " " + w.getHoleMobility() + "\n"
            simiRegDefString += "math/def epsRel_" + name + " " + str(w.getRelativePermittivity()) + "\n"
            if w.isLangevin():
                simiRegDefString += "math/def recFactor_" + name + " e*(eMob_" + name + "+hMob_"+name+")/(eps0*epsRel_" + name + ")\n"
            else:
                simiRegDefString += "math/def recFactor_" + name + " " + w.getRecombinationFactor() + "\n"
            simiRegDefString += "math/def genRate_ni_" + name + " recFactor_" + name + "*ni_" + name + "^2\n"
            simiRegDefString += "math/def eDiff_" + name + " eMob_" + name + "*kT_ev\n"
            simiRegDefString += "math/def hDiff_" + name + " hMob_" + name + "*kT_ev\n"
            # TODO: check this: it should be okay to just use the energy levels for this, instead
            #       of differences with a reference level
            simiRegDefString += "math/def Vn_" + name + " lumo_" + name + "\n"
            simiRegDefString += "math/def Vp_" + name + " homo_" + name + "\n"
            simiRegDefString += "\n"

        for reg in botRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "math/def n_bot_" + name + " g0*exp(-(botContact-lumo_" + name + ")/kT_ev)\n" 
            simiRegDefString += "math/def p_bot_" + name + " g0*exp(-(homo_" + name + "-botContact)/kT_ev)\n" 

        for reg in topRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "math/def n_top_" + name + " g0*exp(-(topContact-lumo_" + name + ")/kT_ev)\n" 
            simiRegDefString += "math/def p_top_" + name + " g0*exp(-(homo_" + name + "-topContact)/kT_ev)\n" 
        
        simiRegDefString += processRegionRects(regDefs, "REG_", regionWidgetNames)
        simiRegDefString += processRegionRects(botRegDefs, "BOTCONTACT_", regionWidgetNames)
        simiRegDefString += processRegionRects(topRegDefs, "TOPCONTACT_", regionWidgetNames)
    
        nx = self.regionImage.getWidth()
        ny = self.regionImage.getHeight()
        physw = self.regionImage.getPhysicalWidth()
        physh = self.regionImage.getPhysicalHeight()
        simiRegDefString += "sim2/new " + str(nx) + " " + str(ny) + " " + str(physw) + " " + str(physh) + "\n"
        simiRegDefString += "sim2/temp/set T\n"
        simiRegDefString += "sim2/phi/set VBI yes\n"

        # Bulk regions

        for reg in regDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim2/reg/set n REG_" + name + " ni_" + name + "\n"
            simiRegDefString += "sim2/reg/set p REG_" + name + " ni_" + name + "\n"
            simiRegDefString += "sim2/reg/set bg REG_" + name + " 0" + "\n"
            simiRegDefString += "sim2/reg/set g REG_" + name + " genRate_" + name + "+genRate_ni_" + name + "\n"
            simiRegDefString += "sim2/reg/set rf REG_" + name + " recFactor_" + name + "\n"
            simiRegDefString += "sim2/reg/set dn REG_" + name + " eDiff_" + name + "\n"
            simiRegDefString += "sim2/reg/set dp REG_" + name + " hDiff_" + name + "\n"
            simiRegDefString += "sim2/reg/set nmob REG_" + name + " eMob_" + name + "\n"
            simiRegDefString += "sim2/reg/set pmob REG_" + name + " hMob_" + name + "\n"
            simiRegDefString += "sim2/reg/set eps REG_" + name + " epsRel_" + name + "\n"
            simiRegDefString += "sim2/reg/set Vn REG_" + name + " Vn_" + name + "\n"
            simiRegDefString += "sim2/reg/set Vp REG_" + name + " Vp_" + name + "\n"
            simiRegDefString += "\n"
        
        # Contacts

        for reg in botRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim2/reg/set n BOTCONTACT_" + name + " n_bot_" + name + "\n"
            simiRegDefString += "sim2/reg/set p BOTCONTACT_" + name + " p_bot_" + name + "\n"

        for reg in topRegDefs:
            idx = reg['index']
            name = regionWidgetNames[idx]
            simiRegDefString += "sim2/reg/set n TOPCONTACT_" + name + " n_top_" + name + "\n"
            simiRegDefString += "sim2/reg/set p TOPCONTACT_" + name + " p_top_" + name + "\n"

        (fileName, filt) = QtGui.QFileDialog.getSaveFileName(None, "Enter file to save commands to", self.lastDir, "Text files (*.txt)")
        if not fileName:
            return

        f = None
        try:
            f = open(fileName, "wt")
            f.write(simiRegDefString)
            self.lastDir = os.path.dirname(fileName)
            QtGui.QMessageBox.information(None, "Info","Commands written to file")
        except Exception as e:
            QtGui.QMessageBox.warning(None, "Error","Couldn't write to file: " + str(e)) 
        finally:
            if f: f.close()

    def onLoad(self):
        (selectedFile, filt) = QtGui.QFileDialog.getOpenFileName(None, "Select layout situation file to import", self.lastDir, "Layout situation files (*.sit)")
        if not selectedFile:
            return
        
        f = None
        stateStr = None
        try: 
            f = open(selectedFile, "rt")
            stateStr = f.read()
        except Exception as e:
            QtGui.QMessageBox.warning(None, "Error","Couldn't load situation from: " + str(e)) 
            return
        finally:
            if f: f.close()

        exec(stateStr)
        
        contacts = fullState[0]
        regions = fullState[1]

        self.ui.matTabWidget.clear()
        for idx in range(2, len(fullState)):
            matState = fullState[idx]
            w = MatPropWidget(self.ui.matTabWidget)
            self.ui.matTabWidget.addTab(w, w.getMaterialName())
            self.ui.matTabWidget.setCurrentIndex(self.ui.matTabWidget.count()-1)
            w.ui.matColorButton.clicked.connect(self.onTransferSelectedColor)
            w.setState(matState)

        if regions['regImage']:
            regImage = pickle.loads(regions['regImage'])
        else:
            regImage = None

        self.setRegionImage(regImage)

        self.ui.topContact.setValue(contacts['top'])
        self.ui.bottomContact.setValue(contacts['bottom'])
        self.ui.ncvValue.setText(contacts['Ncv'])

    def onSave(self):
        
        contacts = { }
        contacts['top'] = self.ui.topContact.value()
        contacts['bottom'] = self.ui.bottomContact.value()
        contacts['Ncv'] = self.ui.ncvValue.text()

        regions = { }
        if not self.regionImage:
            regions['regImage'] = None
        else:
            regions['regImage'] = pickle.dumps(self.regionImage)

        fullState = [ contacts, regions ]
    
        numTabs = self.ui.matTabWidget.count()
        for idx in range(0, numTabs):
            w = self.ui.matTabWidget.widget(idx)
            fullState.append(w.getState())

        (fileName, filt) = QtGui.QFileDialog.getSaveFileName(None, "Enter file to save situation to", self.lastDir, "Layout situation files (*.sit)")
        if not fileName:
            return

        f = None
        try:
            f = open(fileName, "wt")
            f.write("fullState = " + repr(fullState))
            self.lastDir = os.path.dirname(fileName)
            QtGui.QMessageBox.information(None, "Info","Situation written to file")
        except Exception as e:
            QtGui.QMessageBox.warning(None, "Error","Couldn't write to file: " + str(e)) 
        finally:
            if f: f.close()

def main():
    app = QtGui.QApplication(sys.argv)
    
    # Set US locale (for . as decimal separator)
    country = QtCore.QLocale.UnitedStates
    language = QtCore.QLocale.English
    locale = QtCore.QLocale(language, country)
    QtCore.QLocale.setDefault(locale)

    win = RegionEditor()
    win.show()
    sys.exit(app.exec_())

if __name__ == "__main__":
    main()

