#include "amoeba.h"
#include <algorithm>

#include <iostream>

#define AMOEBA_MAXDIM 64

Amoeba::Amoeba()
{
	m_pBestPoint = 0;
}

Amoeba::~Amoeba()
{
}

bool Amoeba::init(std::vector<SearchFunction *> initialPoints, double alpha, double gamma, double rho, double sigma)
{
	if (initialPoints.empty())
	{
		setErrorString("No initial point set specified");
		return false;
	}

	int dimension = initialPoints[0]->getDimension();

	if (dimension < 1 || dimension > AMOEBA_MAXDIM)
	{
		setErrorString("Dimension should be at least one and at most 64");
		return false;
	}

	if (initialPoints.size() != dimension+1)
	{
		setErrorString("The number of initial points should equal the number of dimensions plus one (they must form a simplex)");
		return false;
	}

	clear();

	m_alpha = alpha;
	m_gamma = gamma;
	m_rho = rho;
	m_sigma = sigma;
	m_dimension = dimension;

	m_currentPoints.resize(m_dimension+1);

	for (int i = 0 ; i < m_currentPoints.size() ; i++)
		m_currentPoints[i] = SearchFunctionWrapper(initialPoints[i]->createCopy());

	m_pBestPoint = 0;

	return true;	
}

void Amoeba::clear()
{
	for (int i = 0 ; i < m_currentPoints.size() ; i++)
		delete m_currentPoints[i].getFunction();
	m_currentPoints.clear();
	m_pBestPoint = 0;
}

bool Amoeba::step(double &currentBestValue)
{
	double x0[AMOEBA_MAXDIM];
	std::vector<bool> errors(m_currentPoints.size());

	#pragma omp parallel for schedule(dynamic,1)
	for (int i = 0 ; i < m_currentPoints.size() ; i++)
	{
		if (!m_currentPoints[i].getFunction()->evaluate())
			errors[i] = true;
		else
			errors[i] = false;


	//	const double *pPoint = m_currentPoints[i].getFunction()->getCurrentPoint();
	//	for (int j = 0 ; j < m_dimension ; j++)
	//		std::cout << pPoint[j] << " ";
	//	std::cout << m_currentPoints[i].getFunction()->getValue() << std::endl;
	}
	/*{
		int i = 0;

		const double *pPoint = m_currentPoints[i].getFunction()->getCurrentPoint();
		for (int j = 0 ; j < m_dimension ; j++)
			std::cout << pPoint[j] << " ";
		std::cout << m_currentPoints[i].getFunction()->getValue() << std::endl;

		std::cout << std::endl;
		std::cout << std::endl;
	}*/
	
	for (int i = 0 ; i < m_currentPoints.size() ; i++)
	{
		if (errors[i])
		{
			setErrorString("Unable to evaluate point " + m_currentPoints[i].getFunction()->getPointString());
			return false;	
		}
	}

	std::sort(m_currentPoints.begin(), m_currentPoints.end());
	
	double value0 = m_currentPoints[0].getFunction()->getValue();
	double valueN = m_currentPoints[m_dimension-1].getFunction()->getValue();
	double valueLast = m_currentPoints[m_dimension].getFunction()->getValue();

	double bestValue = value0;
	m_pBestPoint = m_currentPoints[0].getFunction();

	// Calculate centroid

	for (int i = 0 ; i < m_dimension ; i++)
		x0[i] = 0;

	for (int i = 0 ; i < m_dimension ; i++) // is one less than number of points
	{
		const double *pPoint = m_currentPoints[i].getFunction()->getCurrentPoint();

		for (int j = 0 ; j < m_dimension ; j++)
			x0[j] += pPoint[j];
	}

	for (int i = 0 ; i < m_dimension ; i++)
		x0[i] /= (double)m_dimension;

	// Calculate reflected point

	double xr[AMOEBA_MAXDIM];
	const double *pLastPoint = m_currentPoints[m_dimension].getFunction()->getCurrentPoint();

	for (int i = 0 ; i < m_dimension ; i++)
		xr[i] = x0[i] + m_alpha*(x0[i] - pLastPoint[i]);

	SearchFunction *pReflectedFunction = m_currentPoints[0].getFunction()->createNewInstance();

	pReflectedFunction->setFunctionPoint(xr);
	if (!pReflectedFunction->evaluate())
	{
		setErrorString("Unable to evaluate reflected point " + pReflectedFunction->getPointString());
		delete pReflectedFunction;
		return false;
	}

	double xrValue = pReflectedFunction->getValue();

	if (xrValue >= value0 && xrValue < valueN)
	{
		delete m_currentPoints[m_dimension].getFunction();
		m_currentPoints[m_dimension] = SearchFunctionWrapper(pReflectedFunction);
	}
	else
	{
		if (xrValue < value0)
		{

			double xe[AMOEBA_MAXDIM];

			for (int i = 0 ; i < m_dimension ; i++)
				xe[i] = x0[i] + m_gamma*(x0[i] - pLastPoint[i]);

			SearchFunction *pExpandedFunction = m_currentPoints[0].getFunction()->createNewInstance();

			pExpandedFunction->setFunctionPoint(xe);
			if (!pExpandedFunction->evaluate())
			{
				setErrorString("Unable to evaluate expanded point " + pExpandedFunction->getPointString());
				delete pReflectedFunction;
				delete pExpandedFunction;
				return false;
			}

			double xeValue = pExpandedFunction->getValue();

			if (xeValue < xrValue)
			{
				delete m_currentPoints[m_dimension].getFunction();
				delete pReflectedFunction;
				m_currentPoints[m_dimension] = SearchFunctionWrapper(pExpandedFunction);
			}
			else
			{
				delete m_currentPoints[m_dimension].getFunction();
				delete pExpandedFunction;
				m_currentPoints[m_dimension] = SearchFunctionWrapper(pReflectedFunction);
			}

		}
		else 
		{
			double xc[AMOEBA_MAXDIM];

			for (int i = 0 ; i < m_dimension ; i++)
				xc[i] = pLastPoint[i] + m_rho*(x0[i] - pLastPoint[i]);

			SearchFunction *pContractedFunction = m_currentPoints[0].getFunction()->createNewInstance();

			pContractedFunction->setFunctionPoint(xc);
			if (!pContractedFunction->evaluate())
			{
				setErrorString("Unable to evaluate contracted point " + pContractedFunction->getPointString());
				delete pContractedFunction;
				delete pReflectedFunction;
				return false;
			}

			double xcValue = pContractedFunction->getValue();

			if (xcValue < valueLast)
			{
				delete m_currentPoints[m_dimension].getFunction();
				delete pReflectedFunction;
				m_currentPoints[m_dimension] = SearchFunctionWrapper(pContractedFunction);
			}
			else
			{
				const double *pFirstPoint = m_currentPoints[0].getFunction()->getCurrentPoint();
				double xNew[AMOEBA_MAXDIM];

				for (int i = 1 ; i < m_currentPoints.size() ; i++)
				{
					SearchFunction *pFunction = m_currentPoints[i].getFunction();
					const double *pPoint = pFunction->getCurrentPoint();

					for (int j = 0 ; j < m_dimension ; j++)
						xNew[j] = pFirstPoint[j] + m_sigma*(pPoint[j] - pFirstPoint[j]);
					
					pFunction->setFunctionPoint(xNew); // we shall evaluate in the next iteration!
				}

				delete pReflectedFunction;
				delete pContractedFunction;
			}
		}
	}

	currentBestValue = bestValue;
	return true;
}

